package com.zurrtum.create.client.flywheel.backend.engine;

import com.zurrtum.create.client.flywheel.impl.compat.CompatMod;
import it.unimi.dsi.fastutil.longs.Long2ObjectFunction;
import org.jetbrains.annotations.Nullable;
import org.lwjgl.system.MemoryUtil;

import java.util.BitSet;
import java.util.Objects;
import net.minecraft.class_1936;
import net.minecraft.class_1944;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2804;
import net.minecraft.class_3558;
import net.minecraft.class_3560;
import net.minecraft.class_3562;
import net.minecraft.class_3569;
import net.minecraft.class_4076;

import static com.zurrtum.create.client.flywheel.backend.engine.LightStorage.BLOCKS_PER_SECTION;
import static com.zurrtum.create.client.flywheel.backend.engine.LightStorage.SOLID_SIZE_BYTES;

public abstract class LightDataCollector {
    private static final ConstantDataLayer ALWAYS_0 = new ConstantDataLayer(0);
    private static final ConstantDataLayer ALWAYS_15 = new ConstantDataLayer(15);

    protected final class_1936 level;
    protected final class_3562 skyLayerListener;
    protected final class_3562 blockLayerListener;

    protected LightDataCollector(class_1936 level, class_3562 skyLayerListener, class_3562 blockLayerListener) {
        this.level = level;
        this.skyLayerListener = skyLayerListener;
        this.blockLayerListener = blockLayerListener;
    }

    public static LightDataCollector of(class_1936 level) {
        class_3562 skyLayerListener = level.method_22336().method_15562(class_1944.field_9284);
        class_3562 blockLayerListener = level.method_22336().method_15562(class_1944.field_9282);

        Long2ObjectFunction<class_2804> fastSkyDataGetter = createFastSkyDataGetter(skyLayerListener);
        Long2ObjectFunction<class_2804> fastBlockDataGetter = createFastBlockDataGetter(blockLayerListener);

        if (fastSkyDataGetter != null && fastBlockDataGetter != null) {
            return new Fast(level, skyLayerListener, blockLayerListener, fastSkyDataGetter, fastBlockDataGetter);
        } else {
            return new Slow(level, skyLayerListener, blockLayerListener);
        }
    }

    private static class_2804 getSkyDataLayer(class_3569 skyStorage, long section) {
        long l = section;
        int i = class_4076.method_18689(l);
        class_3569.class_3570 skyDataLayerStorageMap = skyStorage.field_15806;
        int j = skyDataLayerStorageMap.field_15821.get(class_4076.method_18693(l));
        if (j != skyDataLayerStorageMap.field_15822 && i < j) {
            class_2804 dataLayer = skyStorage.method_20533(l);
            if (dataLayer == null) {
                for (; dataLayer == null; dataLayer = skyStorage.method_20533(l)) {
                    if (++i >= j) {
                        return null;
                    }

                    l = class_4076.method_18679(l, class_2350.field_11036);
                }
            }

            return dataLayer;
        } else {
            return null;
        }
    }

    @Nullable
    private static Long2ObjectFunction<class_2804> createFastSkyDataGetter(class_3562 layerListener) {
        if (layerListener == class_3562.class_3563.field_15812) {
            // The dummy listener always returns 0.
            // In vanilla this happens in the nether and end,
            // and the light texture is simply updated
            // to be invariant on sky light.
            return section -> ALWAYS_0;
        }

        if (layerListener instanceof class_3558<?, ?> accessor) {
            // Sky storage has a fancy way to get the sky light at a given block position, but the logic is not
            // implemented in vanilla for fetching data layers directly. We need to re-implement it here. The simplest
            // way to do it was to expose the same logic via an extension method. Re-implementing it external to the
            // SkyLightSectionStorage class would require many more accessors.
            if (accessor.field_15793 instanceof class_3569 skyStorage) {
                return section -> {
                    var out = getSkyDataLayer(skyStorage, section);

                    // Null section here means there are no blocks above us to darken this section.
                    return Objects.requireNonNullElse(out, ALWAYS_15);
                };
            }
        }

        if (CompatMod.SCALABLELUX.isLoaded) {
            return section -> Objects.requireNonNullElse(layerListener.method_15544(class_4076.method_18677(section)), ALWAYS_15);
        }

        return null;
    }

    @Nullable
    private static Long2ObjectFunction<class_2804> createFastBlockDataGetter(class_3562 layerListener) {
        if (layerListener == class_3562.class_3563.field_15812) {
            return section -> ALWAYS_0;
        }

        if (layerListener instanceof class_3558<?, ?> accessor) {
            class_3560<?> storage = accessor.field_15793;
            return section -> {
                var out = storage.method_15522(section, false);
                return Objects.requireNonNullElse(out, ALWAYS_0);
            };
        }

        if (CompatMod.SCALABLELUX.isLoaded) {
            return section -> Objects.requireNonNullElse(layerListener.method_15544(class_4076.method_18677(section)), ALWAYS_0);
        }

        return null;
    }

    public void collectSection(long ptr, long section) {
        collectSolidData(ptr, section);
        collectLightData(ptr, section);
    }

    private void collectSolidData(long ptr, long section) {
        var blockPos = new class_2338.class_2339();
        int xMin = class_4076.method_18688(class_4076.method_18686(section));
        int yMin = class_4076.method_18688(class_4076.method_18689(section));
        int zMin = class_4076.method_18688(class_4076.method_18690(section));

        var bitSet = new BitSet(BLOCKS_PER_SECTION);
        int index = 0;
        for (int y = -1; y < 17; y++) {
            for (int z = -1; z < 17; z++) {
                for (int x = -1; x < 17; x++) {
                    blockPos.method_10103(xMin + x, yMin + y, zMin + z);

                    var blockState = level.method_8320(blockPos);

                    if (blockState.method_26225() && blockState.method_26234(level, blockPos)) {
                        bitSet.set(index);
                    }

                    index++;
                }
            }
        }

        var longArray = bitSet.toLongArray();
        for (long l : longArray) {
            MemoryUtil.memPutLong(ptr, l);
            ptr += Long.BYTES;
        }
    }

    protected abstract void collectLightData(long ptr, long section);

    /**
     * Write to the given section.
     *
     * @param ptr   Pointer to the base of a section's data.
     * @param x     X coordinate in the section, from [-1, 16].
     * @param y     Y coordinate in the section, from [-1, 16].
     * @param z     Z coordinate in the section, from [-1, 16].
     * @param block The block light level, from [0, 15].
     * @param sky   The sky light level, from [0, 15].
     */
    protected static void write(long ptr, int x, int y, int z, int block, int sky) {
        int x1 = x + 1;
        int y1 = y + 1;
        int z1 = z + 1;

        int offset = x1 + z1 * 18 + y1 * 18 * 18;

        long packedByte = (block & 0xF) | ((sky & 0xF) << 4);

        MemoryUtil.memPutByte(ptr + SOLID_SIZE_BYTES + offset, (byte) packedByte);
    }

    private static class Fast extends LightDataCollector {
        private final Long2ObjectFunction<class_2804> skyDataGetter;
        private final Long2ObjectFunction<class_2804> blockDataGetter;

        public Fast(
            class_1936 level,
            class_3562 skyLayerListener,
            class_3562 blockLayerListener,
            Long2ObjectFunction<class_2804> skyDataGetter,
            Long2ObjectFunction<class_2804> blockDataGetter
        ) {
            super(level, skyLayerListener, blockLayerListener);
            this.skyDataGetter = skyDataGetter;
            this.blockDataGetter = blockDataGetter;
        }

        @Override
        protected void collectLightData(long ptr, long section) {
            collectCenter(ptr, section);

            for (SectionEdge i : SectionEdge.VALUES) {
                collectYZPlane(ptr, class_4076.method_18678(section, i.sectionOffset, 0, 0), i);
                collectXZPlane(ptr, class_4076.method_18678(section, 0, i.sectionOffset, 0), i);
                collectXYPlane(ptr, class_4076.method_18678(section, 0, 0, i.sectionOffset), i);

                for (SectionEdge j : SectionEdge.VALUES) {
                    collectXStrip(ptr, class_4076.method_18678(section, 0, i.sectionOffset, j.sectionOffset), i, j);
                    collectYStrip(ptr, class_4076.method_18678(section, i.sectionOffset, 0, j.sectionOffset), i, j);
                    collectZStrip(ptr, class_4076.method_18678(section, i.sectionOffset, j.sectionOffset, 0), i, j);
                }
            }

            collectCorners(ptr, section);
        }

        private void collectXStrip(long ptr, long section, SectionEdge y, SectionEdge z) {
            var blockData = blockDataGetter.get(section);
            var skyData = skyDataGetter.get(section);
            for (int x = 0; x < 16; x++) {
                write(ptr, x, y.relative, z.relative, blockData.method_12139(x, y.pos, z.pos), skyData.method_12139(x, y.pos, z.pos));
            }
        }

        private void collectYStrip(long ptr, long section, SectionEdge x, SectionEdge z) {
            var blockData = blockDataGetter.get(section);
            var skyData = skyDataGetter.get(section);
            for (int y = 0; y < 16; y++) {
                write(ptr, x.relative, y, z.relative, blockData.method_12139(x.pos, y, z.pos), skyData.method_12139(x.pos, y, z.pos));
            }
        }

        private void collectZStrip(long ptr, long section, SectionEdge x, SectionEdge y) {
            var blockData = blockDataGetter.get(section);
            var skyData = skyDataGetter.get(section);
            for (int z = 0; z < 16; z++) {
                write(ptr, x.relative, y.relative, z, blockData.method_12139(x.pos, y.pos, z), skyData.method_12139(x.pos, y.pos, z));
            }
        }

        private void collectYZPlane(long ptr, long section, SectionEdge x) {
            var blockData = blockDataGetter.get(section);
            var skyData = skyDataGetter.get(section);
            for (int y = 0; y < 16; y++) {
                for (int z = 0; z < 16; z++) {
                    write(ptr, x.relative, y, z, blockData.method_12139(x.pos, y, z), skyData.method_12139(x.pos, y, z));
                }
            }
        }

        private void collectXZPlane(long ptr, long section, SectionEdge y) {
            var blockData = blockDataGetter.get(section);
            var skyData = skyDataGetter.get(section);
            for (int z = 0; z < 16; z++) {
                for (int x = 0; x < 16; x++) {
                    write(ptr, x, y.relative, z, blockData.method_12139(x, y.pos, z), skyData.method_12139(x, y.pos, z));
                }
            }
        }

        private void collectXYPlane(long ptr, long section, SectionEdge z) {
            var blockData = blockDataGetter.get(section);
            var skyData = skyDataGetter.get(section);
            for (int y = 0; y < 16; y++) {
                for (int x = 0; x < 16; x++) {
                    write(ptr, x, y, z.relative, blockData.method_12139(x, y, z.pos), skyData.method_12139(x, y, z.pos));
                }
            }
        }

        private void collectCenter(long ptr, long section) {
            var blockData = blockDataGetter.get(section);
            var skyData = skyDataGetter.get(section);
            for (int y = 0; y < 16; y++) {
                for (int z = 0; z < 16; z++) {
                    for (int x = 0; x < 16; x++) {
                        write(ptr, x, y, z, blockData.method_12139(x, y, z), skyData.method_12139(x, y, z));
                    }
                }
            }
        }

        private void collectCorners(long ptr, long section) {
            var blockLayerListener = this.blockLayerListener;
            var skyLayerListener = this.skyLayerListener;

            var blockPos = new class_2338.class_2339();
            int xMin = class_4076.method_18688(class_4076.method_18686(section));
            int yMin = class_4076.method_18688(class_4076.method_18689(section));
            int zMin = class_4076.method_18688(class_4076.method_18690(section));

            for (SectionEdge y : SectionEdge.VALUES) {
                for (SectionEdge z : SectionEdge.VALUES) {
                    for (SectionEdge x : SectionEdge.VALUES) {
                        blockPos.method_10103(x.relative + xMin, y.relative + yMin, z.relative + zMin);
                        write(
                            ptr,
                            x.relative,
                            y.relative,
                            z.relative,
                            blockLayerListener.method_15543(blockPos),
                            skyLayerListener.method_15543(blockPos)
                        );
                    }
                }
            }
        }

        private enum SectionEdge {
            LOW(15, -1, -1),
            HIGH(0, 16, 1),
            ;

            public static final SectionEdge[] VALUES = values();

            /**
             * The position in the section to collect.
             */
            private final int pos;
            /**
             * The position relative to the main section.
             */
            private final int relative;
            /**
             * The offset to the neighboring section.
             */
            private final int sectionOffset;

            SectionEdge(int pos, int relative, int sectionOffset) {
                this.pos = pos;
                this.relative = relative;
                this.sectionOffset = sectionOffset;
            }
        }
    }

    private static class Slow extends LightDataCollector {
        public Slow(class_1936 level, class_3562 skyLayerListener, class_3562 blockLayerListener) {
            super(level, skyLayerListener, blockLayerListener);
        }

        @Override
        protected void collectLightData(long ptr, long section) {
            var blockLayerListener = this.blockLayerListener;
            var skyLayerListener = this.skyLayerListener;

            var blockPos = new class_2338.class_2339();
            int xMin = class_4076.method_18688(class_4076.method_18686(section));
            int yMin = class_4076.method_18688(class_4076.method_18689(section));
            int zMin = class_4076.method_18688(class_4076.method_18690(section));

            for (int y = -1; y < 17; y++) {
                for (int z = -1; z < 17; z++) {
                    for (int x = -1; x < 17; x++) {
                        blockPos.method_10103(xMin + x, yMin + y, zMin + z);
                        write(ptr, x, y, z, blockLayerListener.method_15543(blockPos), skyLayerListener.method_15543(blockPos));
                    }
                }
            }
        }
    }

    private static class ConstantDataLayer extends class_2804 {
        private final int value;

        private ConstantDataLayer(int value) {
            this.value = value;
        }

        @Override
        public int method_12139(int x, int y, int z) {
            return value;
        }
    }
}