package com.zurrtum.create.client.flywheel.lib.util;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;

import java.lang.ref.Cleaner;
import java.lang.ref.WeakReference;
import java.util.Iterator;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.function.Consumer;
import java.util.function.Function;
import net.minecraft.class_1936;

public final class LevelAttached<T> {
    private static final ConcurrentLinkedDeque<WeakReference<LevelAttached<?>>> ALL = new ConcurrentLinkedDeque<>();
    private static final Cleaner CLEANER = Cleaner.create();

    private final LoadingCache<class_1936, T> cache;

    public LevelAttached(Function<class_1936, T> factory, Consumer<T> finalizer) {
        WeakReference<LevelAttached<?>> thisRef = new WeakReference<>(this);
        ALL.add(thisRef);

        cache = CacheBuilder.newBuilder().<class_1936, T>removalListener(n -> finalizer.accept(n.getValue())).build(new CacheLoader<>() {
            @Override
            public T load(class_1936 key) {
                return factory.apply(key);
            }
        });

        CLEANER.register(this, new CleaningAction(thisRef, cache));
    }

    public LevelAttached(Function<class_1936, T> factory) {
        this(
            factory, t -> {
            }
        );
    }

    public static void invalidateLevel(class_1936 level) {
        Iterator<WeakReference<LevelAttached<?>>> iterator = ALL.iterator();
        while (iterator.hasNext()) {
            LevelAttached<?> attached = iterator.next().get();
            if (attached == null) {
                iterator.remove();
            } else {
                attached.remove(level);
            }
        }
    }

    public T get(class_1936 level) {
        return cache.getUnchecked(level);
    }

    public void remove(class_1936 level) {
        cache.invalidate(level);
    }

    public T refresh(class_1936 level) {
        remove(level);
        return get(level);
    }

    public void reset() {
        cache.invalidateAll();
    }

    private static class CleaningAction implements Runnable {
        private final WeakReference<LevelAttached<?>> ref;
        private final LoadingCache<class_1936, ?> cache;

        private CleaningAction(WeakReference<LevelAttached<?>> ref, LoadingCache<class_1936, ?> cache) {
            this.ref = ref;
            this.cache = cache;
        }

        @Override
        public void run() {
            ALL.remove(ref);
            cache.invalidateAll();
        }
    }
}
