package com.zurrtum.create.client.vanillin.config;

import com.zurrtum.create.client.flywheel.lib.visualization.SimpleEntityVisualizer;
import org.jetbrains.annotations.Nullable;

import java.util.Objects;
import java.util.function.Predicate;
import net.minecraft.class_1297;
import net.minecraft.class_1299;
import net.minecraft.class_897;

/**
 * An object to configure the visualizer for an entity.
 *
 * @param <T> The type of the entity.
 */
public final class EntityVisualizerBuilder<T extends class_1297> {
    private final Configurator configurator;
    private final class_1299<T> type;
    @Nullable
    private SimpleEntityVisualizer.Factory<T> visualFactory;
    @Nullable
    private Predicate<T> skipVanillaRender;

    public EntityVisualizerBuilder(Configurator configurator, class_1299<T> type) {
        this.configurator = configurator;
        this.type = type;
    }

    /**
     * Sets the visual factory for the entity.
     *
     * @param visualFactory The visual factory.
     * @return {@code this}
     */
    public EntityVisualizerBuilder<T> factory(SimpleEntityVisualizer.Factory<T> visualFactory) {
        this.visualFactory = visualFactory;
        return this;
    }

    /**
     * Sets a predicate to determine whether to skip rendering with the vanilla {@link class_897}.
     *
     * @param skipVanillaRender The predicate.
     * @return {@code this}
     */
    public EntityVisualizerBuilder<T> skipVanillaRender(Predicate<T> skipVanillaRender) {
        this.skipVanillaRender = skipVanillaRender;
        return this;
    }

    /**
     * Sets a predicate to always skip rendering with the vanilla {@link class_897}.
     *
     * @return {@code this}
     */
    public EntityVisualizerBuilder<T> neverSkipVanillaRender() {
        this.skipVanillaRender = entity -> false;
        return this;
    }

    /**
     * Constructs the entity visualizer and sets it for the entity type.
     *
     * @return The entity visualizer.
     */
    public SimpleEntityVisualizer<T> apply(boolean enabledByDefault) {
        Objects.requireNonNull(visualFactory, "Visual factory cannot be null!");
        if (skipVanillaRender == null) {
            skipVanillaRender = entity -> true;
        }

        SimpleEntityVisualizer<T> visualizer = new SimpleEntityVisualizer<>(visualFactory, skipVanillaRender);
        configurator.register(type, visualizer, enabledByDefault);

        return visualizer;
    }
}
