package com.zurrtum.create.api.packager;

import com.zurrtum.create.api.registry.SimpleRegistry;
import com.zurrtum.create.catnip.math.BlockFace;
import com.zurrtum.create.content.logistics.packager.AllInventoryIdentifiers;
import org.jetbrains.annotations.Nullable;

import java.util.Set;
import net.minecraft.class_1937;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2680;
import net.minecraft.class_3341;

/**
 * Interface identifying an inventory spanning multiple block faces.
 * This is used to avoid multiple packagers on the same inventory requesting items from each other.
 */
@FunctionalInterface
public interface InventoryIdentifier {
    SimpleRegistry<class_2248, Finder> REGISTRY = SimpleRegistry.create();

    /**
     * @return true if the given face is part of the inventory this identifier represents
     */
    boolean contains(BlockFace face);

    /**
     * Get the InventoryIdentifier for the given BlockFace, if present.
     */
    @Nullable
    static InventoryIdentifier get(class_1937 level, BlockFace face) {
        class_2680 state = level.method_8320(face.getPos());
        Finder finder = REGISTRY.get(state);
        Finder toQuery = finder != null ? finder : AllInventoryIdentifiers::fallback;
        return toQuery.find(level, state, face);
    }

    /**
     * Interface for finding an InventoryIdentifier for a block.
     */
    @FunctionalInterface
    interface Finder {
        /**
         * Attempt to find the InventoryIdentifier that contains the given BlockFace.
         *
         * @return the found identifier, or null if one isn't present
         */
        @Nullable InventoryIdentifier find(class_1937 level, class_2680 state, BlockFace face);
    }

    // common identifier implementations.

    record Single(class_2338 pos) implements InventoryIdentifier {
        @Override
        public boolean contains(BlockFace face) {
            return this.pos.equals(face.getPos());
        }
    }

    record Pair(class_2338 first, class_2338 second) implements InventoryIdentifier {
        @Override
        public boolean contains(BlockFace face) {
            class_2338 pos = face.getPos();
            return this.first.equals(pos) || this.second.equals(pos);
        }
    }

    record Bounds(class_3341 bounds) implements InventoryIdentifier {
        @Override
        public boolean contains(BlockFace face) {
            return this.bounds.method_14662(face.getPos());
        }
    }

    record MultiFace(class_2338 pos, Set<class_2350> sides) implements InventoryIdentifier {
        @Override
        public boolean contains(BlockFace face) {
            return this.pos.equals(face.getPos()) && this.sides.contains(face.getFace());
        }
    }
}
