package com.zurrtum.create.client.foundation.item;

import com.google.common.collect.ImmutableList;
import com.zurrtum.create.client.AllKeys;
import com.zurrtum.create.client.catnip.lang.FontHelper.Palette;
import com.zurrtum.create.client.foundation.utility.CreateLang;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.Supplier;
import net.minecraft.class_1074;
import net.minecraft.class_1657;
import net.minecraft.class_1792;
import net.minecraft.class_1935;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_5244;
import net.minecraft.class_5250;

import static net.minecraft.class_124.*;

public record ItemDescription(
    ImmutableList<class_2561> lines, ImmutableList<class_2561> linesOnShift, ImmutableList<class_2561> linesOnCtrl
) {
    private static final Map<class_1792, Supplier<String>> CUSTOM_TOOLTIP_KEYS = new IdentityHashMap<>();

    @Nullable
    public static ItemDescription create(class_1792 item, Palette palette) {
        return create(getTooltipTranslationKey(item), palette);
    }

    @Nullable
    public static ItemDescription create(String translationKey, Palette palette) {
        if (!canFillBuilder(translationKey + ".summary")) {
            return null;
        }

        Builder builder = new Builder(palette);
        fillBuilder(builder, translationKey);
        return builder.build();
    }

    public static boolean canFillBuilder(String translationKey) {
        return class_1074.method_4663(translationKey);
    }

    public static void fillBuilder(Builder builder, String translationKey) {
        // Summary
        String summaryKey = translationKey + ".summary";
        if (class_1074.method_4663(summaryKey)) {
            builder.addSummary(class_1074.method_4662(summaryKey));
        }

        // Behaviours
        for (int i = 1; i < 100; i++) {
            String conditionKey = translationKey + ".condition" + i;
            String behaviourKey = translationKey + ".behaviour" + i;
            if (!class_1074.method_4663(conditionKey))
                break;
            builder.addBehaviour(class_1074.method_4662(conditionKey), class_1074.method_4662(behaviourKey));
        }

        // Actions
        for (int i = 1; i < 100; i++) {
            String controlKey = translationKey + ".control" + i;
            String actionKey = translationKey + ".action" + i;
            if (!class_1074.method_4663(controlKey))
                break;
            builder.addAction(class_1074.method_4662(controlKey), class_1074.method_4662(actionKey));
        }
    }

    public static void useKey(class_1792 item, Supplier<String> supplier) {
        CUSTOM_TOOLTIP_KEYS.put(item, supplier);
    }

    public static void useKey(class_1935 item, String string) {
        useKey(item.method_8389(), () -> string);
    }

    public static void referKey(class_1935 item, Supplier<? extends class_1935> otherItem) {
        useKey(item.method_8389(), () -> otherItem.get().method_8389().method_7876());
    }

    public static String getTooltipTranslationKey(class_1792 item) {
        if (CUSTOM_TOOLTIP_KEYS.containsKey(item)) {
            return CUSTOM_TOOLTIP_KEYS.get(item).get() + ".tooltip";
        }
        return item.method_7876() + ".tooltip";
    }

    public ImmutableList<class_2561> getCurrentLines() {
        if (AllKeys.hasShiftDown()) {
            return linesOnShift;
        } else if (AllKeys.hasControlDown()) {
            return linesOnCtrl;
        } else {
            return lines;
        }
    }

    public static class Builder {
        protected final Palette palette;
        protected final List<String> summary = new ArrayList<>();
        protected final List<Pair<String, String>> behaviours = new ArrayList<>();
        protected final List<Pair<String, String>> actions = new ArrayList<>();

        public Builder(Palette palette) {
            this.palette = palette;
        }

        public Builder addSummary(String summaryLine) {
            summary.add(summaryLine);
            return this;
        }

        public Builder addBehaviour(String condition, String behaviour) {
            behaviours.add(Pair.of(condition, behaviour));
            return this;
        }

        public Builder addAction(String condition, String action) {
            actions.add(Pair.of(condition, action));
            return this;
        }

        public ItemDescription build() {
            List<class_2561> lines = new ArrayList<>();
            List<class_2561> linesOnShift = new ArrayList<>();
            List<class_2561> linesOnCtrl = new ArrayList<>();

            for (String summaryLine : summary) {
                linesOnShift.addAll(TooltipHelper.cutStringTextComponent(summaryLine, palette));
            }

            if (!behaviours.isEmpty()) {
                linesOnShift.add(class_5244.field_39003);
            }

            for (Pair<String, String> behaviourPair : behaviours) {
                String condition = behaviourPair.getLeft();
                String behaviour = behaviourPair.getRight();
                linesOnShift.add(class_2561.method_43470(condition).method_27692(field_1080));
                linesOnShift.addAll(TooltipHelper.cutStringTextComponent(behaviour, palette.primary(), palette.highlight(), 1));
            }

            for (Pair<String, String> actionPair : actions) {
                String condition = actionPair.getLeft();
                String action = actionPair.getRight();
                linesOnCtrl.add(class_2561.method_43470(condition).method_27692(field_1080));
                linesOnCtrl.addAll(TooltipHelper.cutStringTextComponent(action, palette.primary(), palette.highlight(), 1));
            }

            boolean hasDescription = !linesOnShift.isEmpty();
            boolean hasControls = !linesOnCtrl.isEmpty();

            if (hasDescription || hasControls) {
                String[] holdDesc = CreateLang.translateDirect("tooltip.holdForDescription", "$").getString().split("\\$");
                String[] holdCtrl = CreateLang.translateDirect("tooltip.holdForControls", "$").getString().split("\\$");
                class_5250 keyShift = CreateLang.translateDirect("tooltip.keyShift");
                class_5250 keyCtrl = CreateLang.translateDirect("tooltip.keyCtrl");
                for (List<class_2561> list : Arrays.asList(lines, linesOnShift, linesOnCtrl)) {
                    boolean shift = list == linesOnShift;
                    boolean ctrl = list == linesOnCtrl;

                    if (holdDesc.length != 2 || holdCtrl.length != 2) {
                        list.add(0, class_2561.method_43470("Invalid lang formatting!"));
                        continue;
                    }

                    if (hasControls) {
                        class_5250 tabBuilder = class_2561.method_43473();
                        tabBuilder.method_10852(class_2561.method_43470(holdCtrl[0]).method_27692(field_1063));
                        tabBuilder.method_10852(keyCtrl.method_27662().method_27692(ctrl ? field_1068 : field_1080));
                        tabBuilder.method_10852(class_2561.method_43470(holdCtrl[1]).method_27692(field_1063));
                        list.add(0, tabBuilder);
                    }

                    if (hasDescription) {
                        class_5250 tabBuilder = class_2561.method_43473();
                        tabBuilder.method_10852(class_2561.method_43470(holdDesc[0]).method_27692(field_1063));
                        tabBuilder.method_10852(keyShift.method_27662().method_27692(shift ? field_1068 : field_1080));
                        tabBuilder.method_10852(class_2561.method_43470(holdDesc[1]).method_27692(field_1063));
                        list.add(0, tabBuilder);
                    }

                    if (shift || ctrl)
                        list.add(hasDescription && hasControls ? 2 : 1, class_5244.field_39003);
                }
            }

            if (!hasDescription) {
                linesOnCtrl.clear();
                linesOnShift.addAll(lines);
            }
            if (!hasControls) {
                linesOnCtrl.clear();
                linesOnCtrl.addAll(lines);
            }

            return new ItemDescription(ImmutableList.copyOf(lines), ImmutableList.copyOf(linesOnShift), ImmutableList.copyOf(linesOnCtrl));
        }
    }

    public static class Modifier implements TooltipModifier {
        protected final class_1792 item;
        protected final Palette palette;
        protected String cachedLanguage;
        protected ItemDescription description;

        public Modifier(class_1792 item, Palette palette) {
            this.item = item;
            this.palette = palette;
        }

        @Override
        public void modify(List<class_2561> tooltip, class_1657 player) {
            if (checkLocale()) {
                description = create(item, palette);
            }
            if (description == null) {
                return;
            }
            tooltip.addAll(1, description.getCurrentLines());
        }

        protected boolean checkLocale() {
            String currentLanguage = class_310.method_1551().method_1526().method_4669();
            if (!currentLanguage.equals(cachedLanguage)) {
                cachedLanguage = currentLanguage;
                return true;
            }
            return false;
        }
    }
}
