package com.zurrtum.create.content.contraptions.bearing;

import com.zurrtum.create.AllAdvancements;
import com.zurrtum.create.AllBlockEntityTypes;
import com.zurrtum.create.AllClientHandle;
import com.zurrtum.create.catnip.math.AngleHelper;
import com.zurrtum.create.content.contraptions.AbstractContraptionEntity;
import com.zurrtum.create.content.contraptions.AssemblyException;
import com.zurrtum.create.content.contraptions.ControlledContraptionEntity;
import com.zurrtum.create.content.contraptions.bearing.ClockworkContraption.HandType;
import com.zurrtum.create.content.kinetics.base.KineticBlockEntity;
import com.zurrtum.create.foundation.advancement.CreateTrigger;
import com.zurrtum.create.foundation.blockEntity.behaviour.BlockEntityBehaviour;
import com.zurrtum.create.foundation.blockEntity.behaviour.scrollValue.ServerScrollOptionBehaviour;
import org.apache.commons.lang3.tuple.Pair;

import java.util.List;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2350.class_2351;
import net.minecraft.class_2680;
import net.minecraft.class_2741;
import net.minecraft.class_3532;

public class ClockworkBearingBlockEntity extends KineticBlockEntity implements IBearingBlockEntity {

    protected ControlledContraptionEntity hourHand;
    protected ControlledContraptionEntity minuteHand;
    protected float hourAngle;
    protected float minuteAngle;
    protected float clientHourAngleDiff;
    protected float clientMinuteAngleDiff;

    protected boolean running;
    protected boolean assembleNextTick;
    protected AssemblyException lastException;
    protected ServerScrollOptionBehaviour<ClockHands> operationMode;

    private float prevForcedAngle;

    public ClockworkBearingBlockEntity(class_2338 pos, class_2680 state) {
        super(AllBlockEntityTypes.CLOCKWORK_BEARING, pos, state);
        setLazyTickRate(3);
    }

    @Override
    public void addBehaviours(List<BlockEntityBehaviour<?>> behaviours) {
        super.addBehaviours(behaviours);
        operationMode = new ServerScrollOptionBehaviour<>(ClockHands.class, this);
        behaviours.add(operationMode);
    }

    @Override
    public List<CreateTrigger> getAwardables() {
        return List.of(AllAdvancements.CLOCKWORK_BEARING);
    }

    @Override
    public boolean isWoodenTop() {
        return false;
    }

    @Override
    public void tick() {
        super.tick();

        if (field_11863.method_8608()) {
            prevForcedAngle = hourAngle;
            clientMinuteAngleDiff /= 2;
            clientHourAngleDiff /= 2;
        }

        if (!field_11863.method_8608() && assembleNextTick) {
            assembleNextTick = false;
            if (running) {
                boolean canDisassemble = true;
                if (speed == 0 && (canDisassemble || hourHand == null || hourHand.getContraption().getBlocks().isEmpty())) {
                    if (hourHand != null)
                        hourHand.getContraption().stop(field_11863);
                    if (minuteHand != null)
                        minuteHand.getContraption().stop(field_11863);
                    disassemble();
                }
                return;
            } else
                assemble();
            return;
        }

        if (!running)
            return;

        if (!(hourHand != null && hourHand.isStalled())) {
            float newAngle = hourAngle + getHourArmSpeed();
            hourAngle = newAngle % 360;
        }

        if (!(minuteHand != null && minuteHand.isStalled())) {
            float newAngle = minuteAngle + getMinuteArmSpeed();
            minuteAngle = newAngle % 360;
        }

        applyRotations();
    }

    public AssemblyException getLastAssemblyException() {
        return lastException;
    }

    protected void applyRotations() {
        class_2680 blockState = method_11010();
        class_2351 axis = class_2351.field_11048;

        if (blockState.method_28498(class_2741.field_12525))
            axis = blockState.method_11654(class_2741.field_12525).method_10166();

        if (hourHand != null) {
            hourHand.setAngle(hourAngle);
            hourHand.setRotationAxis(axis);
        }
        if (minuteHand != null) {
            minuteHand.setAngle(minuteAngle);
            minuteHand.setRotationAxis(axis);
        }
    }

    @Override
    public void lazyTick() {
        super.lazyTick();
        if (hourHand != null && !field_11863.method_8608())
            sendData();
    }

    public float getHourArmSpeed() {
        float speed = getAngularSpeed() / 2f;

        if (speed != 0) {
            ClockHands mode = ClockHands.values()[operationMode.getValue()];
            float hourTarget = mode == ClockHands.HOUR_FIRST ? getHourTarget(false) : mode == ClockHands.MINUTE_FIRST ? getMinuteTarget() : getHourTarget(
                true);
            float shortestAngleDiff = AngleHelper.getShortestAngleDiff(hourAngle, hourTarget);
            if (shortestAngleDiff < 0) {
                speed = Math.max(speed, shortestAngleDiff);
            } else {
                speed = Math.min(-speed, shortestAngleDiff);
            }
        }

        return speed + clientHourAngleDiff / 3f;
    }

    public float getMinuteArmSpeed() {
        float speed = getAngularSpeed();

        if (speed != 0) {
            ClockHands mode = ClockHands.values()[operationMode.getValue()];
            float minuteTarget = mode == ClockHands.MINUTE_FIRST ? getHourTarget(false) : getMinuteTarget();
            float shortestAngleDiff = AngleHelper.getShortestAngleDiff(minuteAngle, minuteTarget);
            if (shortestAngleDiff < 0) {
                speed = Math.max(speed, shortestAngleDiff);
            } else {
                speed = Math.min(-speed, shortestAngleDiff);
            }
        }

        return speed + clientMinuteAngleDiff / 3f;
    }

    protected float getHourTarget(boolean cycle24) {
        boolean isNatural = field_11863.method_8597().comp_645();
        int dayTime = (int) ((field_11863.method_8532() * (isNatural ? 1 : 24)) % 24000);
        int hours = (dayTime / 1000 + 6) % 24;
        int offset = method_11010().method_11654(ClockworkBearingBlock.FACING).method_10171().method_10181();
        return offset * -360 / (cycle24 ? 24f : 12f) * (hours % (cycle24 ? 24 : 12));
    }

    protected float getMinuteTarget() {
        boolean isNatural = field_11863.method_8597().comp_645();
        int dayTime = (int) ((field_11863.method_8532() * (isNatural ? 1 : 24)) % 24000);
        int minutes = (dayTime % 1000) * 60 / 1000;
        int offset = method_11010().method_11654(ClockworkBearingBlock.FACING).method_10171().method_10181();
        return offset * -360 / 60f * (minutes);
    }

    public float getAngularSpeed() {
        float speed = -Math.abs(getSpeed() * 3 / 10f);
        if (field_11863.method_8608())
            speed *= AllClientHandle.INSTANCE.getServerSpeed();
        return speed;
    }

    public void assemble() {
        if (!(field_11863.method_8320(field_11867).method_26204() instanceof ClockworkBearingBlock))
            return;

        class_2350 direction = method_11010().method_11654(class_2741.field_12525);

        // Collect Construct
        Pair<ClockworkContraption, ClockworkContraption> contraption;
        try {
            contraption = ClockworkContraption.assembleClockworkAt(field_11863, field_11867, direction);
            lastException = null;
        } catch (AssemblyException e) {
            lastException = e;
            sendData();
            return;
        }
        if (contraption == null)
            return;
        if (contraption.getLeft() == null)
            return;
        if (contraption.getLeft().getBlocks().isEmpty())
            return;
        class_2338 anchor = field_11867.method_10093(direction);

        contraption.getLeft().removeBlocksFromWorld(field_11863, class_2338.field_10980);
        hourHand = ControlledContraptionEntity.create(field_11863, this, contraption.getLeft());
        hourHand.method_23327(anchor.method_10263(), anchor.method_10264(), anchor.method_10260());
        hourHand.setRotationAxis(direction.method_10166());
        field_11863.method_8649(hourHand);

        if (contraption.getLeft().containsBlockBreakers())
            award(AllAdvancements.CONTRAPTION_ACTORS);

        if (contraption.getRight() != null) {
            anchor = field_11867.method_10079(direction, contraption.getRight().offset + 1);
            contraption.getRight().removeBlocksFromWorld(field_11863, class_2338.field_10980);
            minuteHand = ControlledContraptionEntity.create(field_11863, this, contraption.getRight());
            minuteHand.method_23327(anchor.method_10263(), anchor.method_10264(), anchor.method_10260());
            minuteHand.setRotationAxis(direction.method_10166());
            field_11863.method_8649(minuteHand);

            if (contraption.getRight().containsBlockBreakers())
                award(AllAdvancements.CONTRAPTION_ACTORS);
        }

        award(AllAdvancements.CLOCKWORK_BEARING);

        // Run
        running = true;
        hourAngle = 0;
        minuteAngle = 0;
        sendData();
    }

    public void disassemble() {
        if (!running && hourHand == null && minuteHand == null)
            return;

        hourAngle = 0;
        minuteAngle = 0;
        applyRotations();

        if (hourHand != null) {
            hourHand.disassemble();
        }
        if (minuteHand != null)
            minuteHand.disassemble();

        hourHand = null;
        minuteHand = null;
        running = false;
        sendData();
    }

    @Override
    public void attach(ControlledContraptionEntity contraption) {
        if (!(contraption.getContraption() instanceof ClockworkContraption cc))
            return;

        method_5431();
        class_2350 facing = method_11010().method_11654(class_2741.field_12525);
        class_2338 anchor = field_11867.method_10079(facing, cc.offset + 1);
        if (cc.handType == HandType.HOUR) {
            this.hourHand = contraption;
            hourHand.method_23327(anchor.method_10263(), anchor.method_10264(), anchor.method_10260());
        } else {
            this.minuteHand = contraption;
            minuteHand.method_23327(anchor.method_10263(), anchor.method_10264(), anchor.method_10260());
        }
        if (!field_11863.method_8608()) {
            this.running = true;
            sendData();
        }
    }

    @Override
    public void write(class_11372 view, boolean clientPacket) {
        view.method_71472("Running", running);
        view.method_71464("HourAngle", hourAngle);
        view.method_71464("MinuteAngle", minuteAngle);
        AssemblyException.write(view, lastException);
        super.write(view, clientPacket);
    }

    @Override
    protected void read(class_11368 view, boolean clientPacket) {
        float hourAngleBefore = hourAngle;
        float minuteAngleBefore = minuteAngle;

        running = view.method_71433("Running", false);
        hourAngle = view.method_71423("HourAngle", 0);
        minuteAngle = view.method_71423("MinuteAngle", 0);
        lastException = AssemblyException.read(view);
        super.read(view, clientPacket);

        if (!clientPacket)
            return;

        if (running) {
            clientHourAngleDiff = AngleHelper.getShortestAngleDiff(hourAngleBefore, hourAngle);
            clientMinuteAngleDiff = AngleHelper.getShortestAngleDiff(minuteAngleBefore, minuteAngle);
            hourAngle = hourAngleBefore;
            minuteAngle = minuteAngleBefore;
        } else {
            hourHand = null;
            minuteHand = null;
        }
    }

    @Override
    public void onSpeedChanged(float prevSpeed) {
        super.onSpeedChanged(prevSpeed);
        assembleNextTick = true;
    }

    @Override
    public boolean isValid() {
        return !method_11015();
    }

    @Override
    public float getInterpolatedAngle(float partialTicks) {
        if (isVirtual())
            return class_3532.method_16439(partialTicks, prevForcedAngle, hourAngle);
        if (hourHand == null || hourHand.isStalled())
            partialTicks = 0;
        return class_3532.method_16439(partialTicks, hourAngle, hourAngle + getHourArmSpeed());
    }

    @Override
    public void onStall() {
        if (!field_11863.method_8608())
            sendData();
    }

    @Override
    public void remove() {
        if (!field_11863.method_8608())
            disassemble();
        super.remove();
    }

    @Override
    public boolean isAttachedTo(AbstractContraptionEntity contraption) {
        if (!(contraption.getContraption() instanceof ClockworkContraption cc))
            return false;
        if (cc.handType == HandType.HOUR)
            return this.hourHand == contraption;
        else
            return this.minuteHand == contraption;
    }

    public boolean isRunning() {
        return running;
    }

    public enum ClockHands {
        HOUR_FIRST,
        MINUTE_FIRST,
        HOUR_FIRST_24;
    }

    public void setAngle(float forcedAngle) {
        hourAngle = forcedAngle;
    }

    @Override
    public class_2338 getBlockPosition() {
        return field_11867;
    }
}