package com.zurrtum.create.content.kinetics.base;

import com.zurrtum.create.AllBlockEntityTypes;
import com.zurrtum.create.AllClientHandle;
import com.zurrtum.create.Create;
import com.zurrtum.create.api.stress.BlockStressValues;
import com.zurrtum.create.compat.computercraft.events.KineticsChangeEvent;
import com.zurrtum.create.content.kinetics.KineticNetwork;
import com.zurrtum.create.content.kinetics.RotationPropagator;
import com.zurrtum.create.content.kinetics.base.IRotate.SpeedLevel;
import com.zurrtum.create.content.kinetics.base.IRotate.StressImpact;
import com.zurrtum.create.content.kinetics.simpleRelays.ICogWheel;
import com.zurrtum.create.content.kinetics.transmission.sequencer.SequencedGearshiftBlockEntity.SequenceContext;
import com.zurrtum.create.foundation.blockEntity.SmartBlockEntity;
import com.zurrtum.create.foundation.blockEntity.behaviour.BlockEntityBehaviour;
import com.zurrtum.create.infrastructure.config.AllConfigs;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Objects;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1937;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2350.class_2351;
import net.minecraft.class_2350.class_2352;
import net.minecraft.class_2586;
import net.minecraft.class_2591;
import net.minecraft.class_2680;

public class KineticBlockEntity extends SmartBlockEntity {

    public @Nullable Long network;
    public @Nullable class_2338 source;
    public boolean networkDirty;
    public boolean updateSpeed;
    public int preventSpeedUpdate;
    public SequenceContext sequenceContext;
    public KineticEffectHandler effects;
    protected float speed;
    protected float capacity;
    protected float stress;
    protected boolean overStressed;
    protected boolean wasMoved;
    protected float lastStressApplied;
    protected float lastCapacityProvided;
    private int flickerTally;
    private int networkSize;
    private int validationCountdown;

    public KineticBlockEntity(class_2591<?> typeIn, class_2338 pos, class_2680 state) {
        super(typeIn, pos, state);
        effects = new KineticEffectHandler(this);
        updateSpeed = true;
    }

    public static KineticBlockEntity encased(class_2338 pos, class_2680 state) {
        return new KineticBlockEntity(AllBlockEntityTypes.ENCASED_SHAFT, pos, state);
    }

    public static void switchToBlockState(class_1937 world, class_2338 pos, class_2680 state) {
        if (world.method_8608())
            return;

        class_2586 blockEntity = world.method_8321(pos);
        class_2680 currentState = world.method_8320(pos);
        boolean isKinetic = blockEntity instanceof KineticBlockEntity;

        if (currentState == state)
            return;
        if (blockEntity == null || !isKinetic) {
            world.method_8652(pos, state, class_2248.field_31036);
            return;
        }

        KineticBlockEntity kineticBlockEntity = (KineticBlockEntity) blockEntity;
        if (state.method_26204() instanceof KineticBlock && !((KineticBlock) state.method_26204()).areStatesKineticallyEquivalent(currentState, state)) {
            if (kineticBlockEntity.hasNetwork())
                kineticBlockEntity.getOrCreateNetwork().remove(kineticBlockEntity);
            kineticBlockEntity.detachKinetics();
            kineticBlockEntity.removeSource();
        }

        if (blockEntity instanceof GeneratingKineticBlockEntity generatingBlockEntity) {
            generatingBlockEntity.reActivateSource = true;
        }

        world.method_8652(pos, state, class_2248.field_31036);
    }

    public static float convertToDirection(float axisSpeed, class_2350 d) {
        return d.method_10171() == class_2352.field_11056 ? axisSpeed : -axisSpeed;
    }

    public static float convertToLinear(float speed) {
        return speed / 512f;
    }

    public static float convertToAngular(float speed) {
        // speed (rpm) * 360 (revolution->deg) / 60 (min->sec) / 20 (sec->tick)
        // rpm -> deg/tick
        return speed * 360f / 60f / 20f;
    }

    @Override
    public void initialize() {
        if (hasNetwork() && !field_11863.method_8608()) {
            KineticNetwork network = getOrCreateNetwork();
            if (!network.initialized)
                network.initFromTE(capacity, stress, networkSize);
            network.addSilently(this, lastCapacityProvided, lastStressApplied);
        }

        super.initialize();
    }

    @Override
    public void tick() {
        if (!field_11863.method_8608() && needsSpeedUpdate())
            attachKinetics();

        super.tick();
        effects.tick();

        preventSpeedUpdate = 0;

        if (field_11863.method_8608()) {
            return;
        }

        if (validationCountdown-- <= 0) {
            validationCountdown = AllConfigs.server().kinetics.kineticValidationFrequency.get();
            validateKinetics();
        }

        if (getFlickerScore() > 0)
            flickerTally = getFlickerScore() - 1;

        if (networkDirty) {
            if (hasNetwork())
                getOrCreateNetwork().updateNetwork();
            networkDirty = false;
        }
    }

    private void validateKinetics() {
        if (hasSource()) {
            if (!hasNetwork()) {
                removeSource();
                return;
            }

            if (!field_11863.method_22340(source))
                return;

            class_2586 blockEntity = field_11863.method_8321(source);
            KineticBlockEntity sourceBE = blockEntity instanceof KineticBlockEntity ? (KineticBlockEntity) blockEntity : null;
            if (sourceBE == null || sourceBE.speed == 0) {
                removeSource();
                detachKinetics();
                return;
            }

            return;
        }

        if (speed != 0) {
            if (getGeneratedSpeed() == 0)
                speed = 0;
        }
    }

    public void updateFromNetwork(float maxStress, float currentStress, int networkSize) {
        networkDirty = false;
        this.capacity = maxStress;
        this.stress = currentStress;
        this.networkSize = networkSize;
        boolean overStressed = maxStress < currentStress && StressImpact.isEnabled();
        method_5431();

        if (overStressed != this.overStressed) {
            float prevSpeed = getSpeed();
            this.overStressed = overStressed;
            onSpeedChanged(prevSpeed);
            sendData();
        }
    }

    protected KineticsChangeEvent makeComputerKineticsChangeEvent() {
        return new KineticsChangeEvent(speed, capacity, stress, overStressed);
    }

    protected class_2248 getStressConfigKey() {
        return method_11010().method_26204();
    }

    public float calculateStressApplied() {
        float impact = (float) BlockStressValues.getImpact(getStressConfigKey());
        this.lastStressApplied = impact;
        return impact;
    }

    public float calculateAddedStressCapacity() {
        float capacity = (float) BlockStressValues.getCapacity(getStressConfigKey());
        this.lastCapacityProvided = capacity;
        return capacity;
    }

    public void onSpeedChanged(float previousSpeed) {
        boolean fromOrToZero = (previousSpeed == 0) != (getSpeed() == 0);
        boolean directionSwap = !fromOrToZero && Math.signum(previousSpeed) != Math.signum(getSpeed());
        if (fromOrToZero || directionSwap)
            flickerTally = getFlickerScore() + 5;
        method_5431();
    }

    @Override
    public void remove() {
        if (!field_11863.method_8608()) {
            if (hasNetwork())
                getOrCreateNetwork().remove(this);
            detachKinetics();
        }
        super.remove();
    }

    @Override
    protected void write(class_11372 view, boolean clientPacket) {
        view.method_71464("Speed", speed);
        if (sequenceContext != null && (!clientPacket || syncSequenceContext()))
            view.method_71468("Sequence", SequenceContext.CODEC, sequenceContext);

        if (needsSpeedUpdate())
            view.method_71472("NeedsSpeedUpdate", true);

        if (hasSource())
            view.method_71468("Source", class_2338.field_25064, source);

        if (hasNetwork()) {
            class_11372 networkTag = view.method_71461("Network");
            networkTag.method_71466("Id", network);
            networkTag.method_71464("Stress", stress);
            networkTag.method_71464("Capacity", capacity);
            networkTag.method_71465("Size", networkSize);

            if (lastStressApplied != 0)
                networkTag.method_71464("AddedStress", lastStressApplied);
            if (lastCapacityProvided != 0)
                networkTag.method_71464("AddedCapacity", lastCapacityProvided);
        }

        super.write(view, clientPacket);
    }

    public boolean needsSpeedUpdate() {
        return updateSpeed;
    }

    @Override
    protected void read(class_11368 view, boolean clientPacket) {
        boolean overStressedBefore = overStressed;
        clearKineticInformation();

        // DO NOT READ kinetic information when placed after movement
        if (wasMoved) {
            super.read(view, clientPacket);
            return;
        }

        speed = view.method_71423("Speed", 0);
        sequenceContext = view.method_71426("Sequence", SequenceContext.CODEC).orElse(null);

        source = view.method_71426("Source", class_2338.field_25064).orElse(null);

        view.method_71420("Network").ifPresent(networkTag -> {
            network = networkTag.method_71425("Id", 0);
            stress = networkTag.method_71423("Stress", 0);
            capacity = networkTag.method_71423("Capacity", 0);
            networkSize = networkTag.method_71424("Size", 0);
            lastStressApplied = networkTag.method_71423("AddedStress", 0);
            lastCapacityProvided = networkTag.method_71423("AddedCapacity", 0);
            overStressed = capacity < stress && StressImpact.isEnabled();
        });

        super.read(view, clientPacket);

        if (clientPacket && overStressedBefore != overStressed && speed != 0)
            effects.triggerOverStressedEffect();

        if (clientPacket)
            AllClientHandle.INSTANCE.queueUpdate(this);
    }

    public float getGeneratedSpeed() {
        return 0;
    }

    public boolean isSource() {
        return getGeneratedSpeed() != 0;
    }

    public void setSource(class_2338 source) {
        this.source = source;
        if (field_11863 == null || field_11863.method_8608())
            return;

        class_2586 blockEntity = field_11863.method_8321(source);
        if (!(blockEntity instanceof KineticBlockEntity sourceBE)) {
            removeSource();
            return;
        }

        setNetwork(sourceBE.network);
        copySequenceContextFrom(sourceBE);
    }

    public float getSpeed() {
        if (overStressed || (field_11863 != null && field_11863.method_54719().method_54754()))
            return 0;
        return getTheoreticalSpeed();
    }

    public void setSpeed(float speed) {
        this.speed = speed;
    }

    public float getTheoreticalSpeed() {
        return speed;
    }

    public boolean hasSource() {
        return source != null;
    }

    protected void copySequenceContextFrom(KineticBlockEntity sourceBE) {
        sequenceContext = sourceBE.sequenceContext;
    }

    public void removeSource() {
        float prevSpeed = getSpeed();

        speed = 0;
        source = null;
        setNetwork(null);
        sequenceContext = null;

        onSpeedChanged(prevSpeed);
    }

    public void setNetwork(@Nullable Long networkIn) {
        if (Objects.equals(network, networkIn))
            return;
        if (network != null)
            getOrCreateNetwork().remove(this);

        network = networkIn;
        method_5431();

        if (networkIn == null)
            return;

        network = networkIn;
        KineticNetwork network = getOrCreateNetwork();
        network.initialized = true;
        network.add(this);
    }

    public KineticNetwork getOrCreateNetwork() {
        return Create.TORQUE_PROPAGATOR.getOrCreateNetworkFor(this);
    }

    public boolean hasNetwork() {
        return network != null;
    }

    public void attachKinetics() {
        updateSpeed = false;
        RotationPropagator.handleAdded(field_11863, field_11867, this);
    }

    public void detachKinetics() {
        RotationPropagator.handleRemoved(field_11863, field_11867, this);
    }

    public boolean isSpeedRequirementFulfilled() {
        class_2680 state = method_11010();
        if (!(state.method_26204() instanceof IRotate def))
            return true;
        SpeedLevel minimumRequiredSpeedLevel = def.getMinimumRequiredSpeedLevel();
        return Math.abs(getSpeed()) >= minimumRequiredSpeedLevel.getSpeedValue();
    }

    @Override
    public void addBehaviours(List<BlockEntityBehaviour<?>> behaviours) {
    }

    public void clearKineticInformation() {
        speed = 0;
        source = null;
        network = null;
        overStressed = false;
        stress = 0;
        capacity = 0;
        lastStressApplied = 0;
        lastCapacityProvided = 0;
    }

    public void warnOfMovement() {
        wasMoved = true;
    }

    public int getFlickerScore() {
        return flickerTally;
    }

    public boolean isOverStressed() {
        return overStressed;
    }

    // Custom Propagation

    /**
     * Specify ratio of transferred rotation from this kinetic component to a
     * specific other.
     *
     * @param target           other Kinetic BE to transfer to
     * @param stateFrom        this BE's blockstate
     * @param stateTo          other BE's blockstate
     * @param diff             difference in position (to.pos - from.pos)
     * @param connectedViaAxes whether these kinetic blocks are connected via mutual
     *                         IRotate.hasShaftTowards()
     * @param connectedViaCogs whether these kinetic blocks are connected via mutual
     *                         IRotate.hasIntegratedCogwheel()
     * @return factor of rotation speed from this BE to other. 0 if no rotation is
     * transferred, or the standard rules apply (integrated shafts/cogs)
     */
    public float propagateRotationTo(
        KineticBlockEntity target,
        class_2680 stateFrom,
        class_2680 stateTo,
        class_2338 diff,
        boolean connectedViaAxes,
        boolean connectedViaCogs
    ) {
        return 0;
    }

    /**
     * Specify additional locations the rotation propagator should look for
     * potentially connected components. Neighbour list contains offset positions in
     * all 6 directions by default.
     *
     * @param block
     * @param state
     * @param neighbours
     * @return
     */
    public List<class_2338> addPropagationLocations(IRotate block, class_2680 state, List<class_2338> neighbours) {
        if (!canPropagateDiagonally(block, state))
            return neighbours;

        class_2351 axis = block.getRotationAxis(state);
        class_2338.method_20437(new class_2338(-1, -1, -1), new class_2338(1, 1, 1)).forEach(offset -> {
            if (axis.method_10173(offset.method_10263(), offset.method_10264(), offset.method_10260()) != 0)
                return;
            if (offset.method_10262(class_2338.field_11176) != 2)
                return;
            neighbours.add(field_11867.method_10081(offset));
        });
        return neighbours;
    }

    /**
     * Specify whether this component can propagate speed to the other in any
     * circumstance. Shaft and cogwheel connections are already handled by internal
     * logic. Does not have to be specified on both ends, it is assumed that this
     * relation is symmetrical.
     *
     * @param other
     * @param state
     * @param otherState
     * @return true if this and the other component should check their propagation
     * factor and are not already connected via integrated cogs or shafts
     */
    public boolean isCustomConnection(KineticBlockEntity other, class_2680 state, class_2680 otherState) {
        return false;
    }

    protected boolean canPropagateDiagonally(IRotate block, class_2680 state) {
        return ICogWheel.isSmallCog(state);
    }

    public boolean isNoisy() {
        return true;
    }

    public int getRotationAngleOffset(class_2351 axis) {
        return 0;
    }

    protected boolean syncSequenceContext() {
        return false;
    }

}