package com.zurrtum.create.content.kinetics;

import com.zurrtum.create.AllBlocks;
import com.zurrtum.create.catnip.data.Iterate;
import com.zurrtum.create.content.kinetics.base.DirectionalShaftHalvesBlockEntity;
import com.zurrtum.create.content.kinetics.base.IRotate;
import com.zurrtum.create.content.kinetics.base.KineticBlockEntity;
import com.zurrtum.create.content.kinetics.chainDrive.ChainDriveBlock;
import com.zurrtum.create.content.kinetics.gearbox.GearboxBlockEntity;
import com.zurrtum.create.content.kinetics.simpleRelays.CogWheelBlock;
import com.zurrtum.create.content.kinetics.simpleRelays.ICogWheel;
import com.zurrtum.create.content.kinetics.speedController.SpeedControllerBlock;
import com.zurrtum.create.content.kinetics.speedController.SpeedControllerBlockEntity;
import com.zurrtum.create.content.kinetics.transmission.SplitShaftBlockEntity;
import com.zurrtum.create.infrastructure.config.AllConfigs;
import java.util.LinkedList;
import java.util.List;
import net.minecraft.class_1937;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2350.class_2351;
import net.minecraft.class_2586;
import net.minecraft.class_2680;

import static net.minecraft.class_2741.field_12496;

public class RotationPropagator {

    private static final int MAX_FLICKER_SCORE = 128;

    /**
     * Determines the change in rotation between two attached kinetic entities. For
     * instance, an axis connection returns 1 while a 1-to-1 gear connection
     * reverses the rotation and therefore returns -1.
     *
     * @param from
     * @param to
     * @return
     */
    private static float getRotationSpeedModifier(KineticBlockEntity from, KineticBlockEntity to) {
        final class_2680 stateFrom = from.method_11010();
        final class_2680 stateTo = to.method_11010();

        class_2248 fromBlock = stateFrom.method_26204();
        class_2248 toBlock = stateTo.method_26204();
        if (!(fromBlock instanceof IRotate definitionFrom && toBlock instanceof IRotate definitionTo))
            return 0;

        final class_2338 diff = to.method_11016().method_10059(from.method_11016());
        final class_2350 direction = class_2350.method_10147(diff.method_10263(), diff.method_10264(), diff.method_10260());
        final class_1937 world = from.method_10997();

        boolean alignedAxes = true;
        for (class_2351 axis : class_2351.values())
            if (axis != direction.method_10166())
                if (axis.method_10173(diff.method_10263(), diff.method_10264(), diff.method_10260()) != 0)
                    alignedAxes = false;

        boolean connectedByAxis = alignedAxes && definitionFrom.hasShaftTowards(
            world,
            from.method_11016(),
            stateFrom,
            direction
        ) && definitionTo.hasShaftTowards(world, to.method_11016(), stateTo, direction.method_10153());

        boolean connectedByGears = ICogWheel.isSmallCog(stateFrom) && ICogWheel.isSmallCog(stateTo);

        float custom = from.propagateRotationTo(to, stateFrom, stateTo, diff, connectedByAxis, connectedByGears);
        if (custom != 0)
            return custom;

        // Axis <-> Axis
        if (connectedByAxis) {
            float axisModifier = getAxisModifier(to, direction.method_10153());
            if (axisModifier != 0)
                axisModifier = 1 / axisModifier;
            return getAxisModifier(from, direction) * axisModifier;
        }

        // Attached Encased Belts
        if (fromBlock instanceof ChainDriveBlock && toBlock instanceof ChainDriveBlock) {
            boolean connected = ChainDriveBlock.areBlocksConnected(stateFrom, stateTo, direction);
            return connected ? ChainDriveBlock.getRotationSpeedModifier(from, to) : 0;
        }

        // Large Gear <-> Large Gear
        if (isLargeToLargeGear(stateFrom, stateTo, diff)) {
            class_2351 sourceAxis = stateFrom.method_11654(field_12496);
            class_2351 targetAxis = stateTo.method_11654(field_12496);
            int sourceAxisDiff = sourceAxis.method_10173(diff.method_10263(), diff.method_10264(), diff.method_10260());
            int targetAxisDiff = targetAxis.method_10173(diff.method_10263(), diff.method_10264(), diff.method_10260());

            return sourceAxisDiff > 0 ^ targetAxisDiff > 0 ? -1 : 1;
        }

        // Gear <-> Large Gear
        if (ICogWheel.isLargeCog(stateFrom) && ICogWheel.isSmallCog(stateTo))
            if (isLargeToSmallCog(stateFrom, stateTo, definitionTo, diff))
                return -2f;
        if (ICogWheel.isLargeCog(stateTo) && ICogWheel.isSmallCog(stateFrom))
            if (isLargeToSmallCog(stateTo, stateFrom, definitionFrom, diff))
                return -.5f;

        // Gear <-> Gear
        if (connectedByGears) {
            if (diff.method_19455(class_2338.field_11176) != 1)
                return 0;
            if (ICogWheel.isLargeCog(stateTo))
                return 0;
            if (direction.method_10166() == definitionFrom.getRotationAxis(stateFrom))
                return 0;
            if (definitionFrom.getRotationAxis(stateFrom) == definitionTo.getRotationAxis(stateTo))
                return -1;
        }

        return 0;
    }

    private static float getConveyedSpeed(KineticBlockEntity from, KineticBlockEntity to) {
        final class_2680 stateFrom = from.method_11010();
        final class_2680 stateTo = to.method_11010();

        // Rotation Speed Controller <-> Large Gear
        if (isLargeCogToSpeedController(stateFrom, stateTo, to.method_11016().method_10059(from.method_11016())))
            return SpeedControllerBlockEntity.getConveyedSpeed(from, to, true);
        if (isLargeCogToSpeedController(stateTo, stateFrom, from.method_11016().method_10059(to.method_11016())))
            return SpeedControllerBlockEntity.getConveyedSpeed(to, from, false);

        float rotationSpeedModifier = getRotationSpeedModifier(from, to);
        return from.getTheoreticalSpeed() * rotationSpeedModifier;
    }

    private static boolean isLargeToLargeGear(class_2680 from, class_2680 to, class_2338 diff) {
        if (!ICogWheel.isLargeCog(from) || !ICogWheel.isLargeCog(to))
            return false;
        class_2351 fromAxis = from.method_11654(field_12496);
        class_2351 toAxis = to.method_11654(field_12496);
        if (fromAxis == toAxis)
            return false;
        for (class_2351 axis : class_2351.values()) {
            int axisDiff = axis.method_10173(diff.method_10263(), diff.method_10264(), diff.method_10260());
            if (axis == fromAxis || axis == toAxis) {
                if (axisDiff == 0)
                    return false;

            } else if (axisDiff != 0)
                return false;
        }
        return true;
    }

    private static float getAxisModifier(KineticBlockEntity be, class_2350 direction) {
        if (!(be.hasSource() || be.isSource()) || !(be instanceof DirectionalShaftHalvesBlockEntity))
            return 1;
        class_2350 source = ((DirectionalShaftHalvesBlockEntity) be).getSourceFacing();

        if (be instanceof GearboxBlockEntity)
            return direction.method_10166() == source.method_10166() ? direction == source ? 1 : -1 : direction.method_10171() == source.method_10171() ? -1 : 1;

        if (be instanceof SplitShaftBlockEntity)
            return ((SplitShaftBlockEntity) be).getRotationSpeedModifier(direction);

        return 1;
    }

    private static boolean isLargeToSmallCog(class_2680 from, class_2680 to, IRotate defTo, class_2338 diff) {
        class_2351 axisFrom = from.method_11654(field_12496);
        if (axisFrom != defTo.getRotationAxis(to))
            return false;
        if (axisFrom.method_10173(diff.method_10263(), diff.method_10264(), diff.method_10260()) != 0)
            return false;
        for (class_2351 axis : class_2351.values()) {
            if (axis == axisFrom)
                continue;
            if (Math.abs(axis.method_10173(diff.method_10263(), diff.method_10264(), diff.method_10260())) != 1)
                return false;
        }
        return true;
    }

    private static boolean isLargeCogToSpeedController(class_2680 from, class_2680 to, class_2338 diff) {
        if (!ICogWheel.isLargeCog(from) || !to.method_27852(AllBlocks.ROTATION_SPEED_CONTROLLER))
            return false;
        if (!diff.equals(class_2338.field_10980.method_10074()))
            return false;
        class_2351 axis = from.method_11654(CogWheelBlock.AXIS);
        if (axis.method_10178())
            return false;
        if (to.method_11654(SpeedControllerBlock.HORIZONTAL_AXIS) == axis)
            return false;
        return true;
    }

    /**
     * Insert the added position to the kinetic network.
     *
     * @param worldIn
     * @param pos
     */
    public static void handleAdded(class_1937 worldIn, class_2338 pos, KineticBlockEntity addedTE) {
        if (worldIn.method_8608())
            return;
        if (!worldIn.method_8477(pos))
            return;
        propagateNewSource(addedTE);
    }

    /**
     * Search for sourceless networks attached to the given entity and update them.
     *
     * @param currentTE
     */
    private static void propagateNewSource(KineticBlockEntity currentTE) {
        class_2338 pos = currentTE.method_11016();
        class_1937 world = currentTE.method_10997();

        for (KineticBlockEntity neighbourTE : getConnectedNeighbours(currentTE)) {
            float speedOfCurrent = currentTE.getTheoreticalSpeed();
            float speedOfNeighbour = neighbourTE.getTheoreticalSpeed();
            float newSpeed = getConveyedSpeed(currentTE, neighbourTE);
            float oppositeSpeed = getConveyedSpeed(neighbourTE, currentTE);

            if (newSpeed == 0 && oppositeSpeed == 0)
                continue;

            boolean incompatible = Math.signum(newSpeed) != Math.signum(speedOfNeighbour) && (newSpeed != 0 && speedOfNeighbour != 0);

            boolean tooFast = Math.abs(newSpeed) > AllConfigs.server().kinetics.maxRotationSpeed.get() || Math.abs(oppositeSpeed) > AllConfigs.server().kinetics.maxRotationSpeed.get();
            // Check for both the new speed and the opposite speed, just in case

            boolean speedChangedTooOften = currentTE.getFlickerScore() > MAX_FLICKER_SCORE;
            if (tooFast || speedChangedTooOften) {
                world.method_22352(pos, true);
                return;
            }

            // Opposite directions
            if (incompatible) {
                world.method_22352(pos, true);
                return;

                // Same direction: overpower the slower speed
            } else {

                // Neighbour faster, overpower the incoming tree
                if (Math.abs(oppositeSpeed) > Math.abs(speedOfCurrent)) {
                    float prevSpeed = currentTE.getSpeed();
                    currentTE.setSource(neighbourTE.method_11016());
                    currentTE.setSpeed(getConveyedSpeed(neighbourTE, currentTE));
                    currentTE.onSpeedChanged(prevSpeed);
                    currentTE.sendData();

                    propagateNewSource(currentTE);
                    return;
                }

                // Current faster, overpower the neighbours' tree
                if (Math.abs(newSpeed) >= Math.abs(speedOfNeighbour)) {

                    // Do not overpower you own network -> cycle
                    if (!currentTE.hasNetwork() || currentTE.network.equals(neighbourTE.network)) {
                        float epsilon = Math.abs(speedOfNeighbour) / 256f / 256f;
                        if (Math.abs(newSpeed) > Math.abs(speedOfNeighbour) + epsilon)
                            world.method_22352(pos, true);
                        continue;
                    }

                    if (currentTE.hasSource() && currentTE.source.equals(neighbourTE.method_11016()))
                        currentTE.removeSource();

                    float prevSpeed = neighbourTE.getSpeed();
                    neighbourTE.setSource(currentTE.method_11016());
                    neighbourTE.setSpeed(getConveyedSpeed(currentTE, neighbourTE));
                    neighbourTE.onSpeedChanged(prevSpeed);
                    neighbourTE.sendData();
                    propagateNewSource(neighbourTE);
                    continue;
                }
            }

            if (neighbourTE.getTheoreticalSpeed() == newSpeed)
                continue;

            float prevSpeed = neighbourTE.getSpeed();
            neighbourTE.setSpeed(newSpeed);
            neighbourTE.setSource(currentTE.method_11016());
            neighbourTE.onSpeedChanged(prevSpeed);
            neighbourTE.sendData();
            propagateNewSource(neighbourTE);

        }
    }

    /**
     * Remove the given entity from the network.
     *
     * @param worldIn
     * @param pos
     * @param removedBE
     */
    public static void handleRemoved(class_1937 worldIn, class_2338 pos, KineticBlockEntity removedBE) {
        if (worldIn.method_8608())
            return;
        if (removedBE == null)
            return;
        if (removedBE.getTheoreticalSpeed() == 0)
            return;

        for (class_2338 neighbourPos : getPotentialNeighbourLocations(removedBE)) {
            class_2680 neighbourState = worldIn.method_8320(neighbourPos);
            if (!(neighbourState.method_26204() instanceof IRotate))
                continue;
            class_2586 blockEntity = worldIn.method_8321(neighbourPos);
            if (!(blockEntity instanceof KineticBlockEntity neighbourBE))
                continue;

            if (!neighbourBE.hasSource() || !neighbourBE.source.equals(pos))
                continue;

            propagateMissingSource(neighbourBE);
        }

    }

    /**
     * Clear the entire subnetwork depending on the given entity and find a new
     * source
     *
     * @param updateTE
     */
    private static void propagateMissingSource(KineticBlockEntity updateTE) {
        final class_1937 world = updateTE.method_10997();

        List<KineticBlockEntity> potentialNewSources = new LinkedList<>();
        List<class_2338> frontier = new LinkedList<>();
        frontier.add(updateTE.method_11016());
        class_2338 missingSource = updateTE.hasSource() ? updateTE.source : null;

        while (!frontier.isEmpty()) {
            final class_2338 pos = frontier.remove(0);
            class_2586 blockEntity = world.method_8321(pos);
            if (!(blockEntity instanceof KineticBlockEntity currentBE))
                continue;

            currentBE.removeSource();
            currentBE.sendData();

            for (KineticBlockEntity neighbourBE : getConnectedNeighbours(currentBE)) {
                if (neighbourBE.method_11016().equals(missingSource))
                    continue;
                if (!neighbourBE.hasSource())
                    continue;

                if (!neighbourBE.source.equals(pos)) {
                    potentialNewSources.add(neighbourBE);
                    continue;
                }

                if (neighbourBE.isSource())
                    potentialNewSources.add(neighbourBE);

                frontier.add(neighbourBE.method_11016());
            }
        }

        for (KineticBlockEntity newSource : potentialNewSources) {
            if (newSource.hasSource() || newSource.isSource()) {
                propagateNewSource(newSource);
                return;
            }
        }
    }

    private static KineticBlockEntity findConnectedNeighbour(KineticBlockEntity currentTE, class_2338 neighbourPos) {
        class_2680 neighbourState = currentTE.method_10997().method_8320(neighbourPos);
        if (!(neighbourState.method_26204() instanceof IRotate))
            return null;
        if (!neighbourState.method_31709())
            return null;
        class_2586 neighbourBE = currentTE.method_10997().method_8321(neighbourPos);
        if (!(neighbourBE instanceof KineticBlockEntity neighbourKBE))
            return null;
        if (!(neighbourKBE.method_11010().method_26204() instanceof IRotate))
            return null;
        if (!isConnected(currentTE, neighbourKBE) && !isConnected(neighbourKBE, currentTE))
            return null;
        return neighbourKBE;
    }

    public static boolean isConnected(KineticBlockEntity from, KineticBlockEntity to) {
        final class_2680 stateFrom = from.method_11010();
        final class_2680 stateTo = to.method_11010();
        return isLargeCogToSpeedController(stateFrom, stateTo, to.method_11016().method_10059(from.method_11016())) || getRotationSpeedModifier(
            from,
            to
        ) != 0 || from.isCustomConnection(to, stateFrom, stateTo);
    }

    private static List<KineticBlockEntity> getConnectedNeighbours(KineticBlockEntity be) {
        List<KineticBlockEntity> neighbours = new LinkedList<>();
        for (class_2338 neighbourPos : getPotentialNeighbourLocations(be)) {
            final KineticBlockEntity neighbourBE = findConnectedNeighbour(be, neighbourPos);
            if (neighbourBE == null)
                continue;

            neighbours.add(neighbourBE);
        }
        return neighbours;
    }

    private static List<class_2338> getPotentialNeighbourLocations(KineticBlockEntity be) {
        List<class_2338> neighbours = new LinkedList<>();
        class_2338 blockPos = be.method_11016();
        class_1937 level = be.method_10997();

        if (!level.method_8477(blockPos))
            return neighbours;

        for (class_2350 facing : Iterate.directions) {
            class_2338 relative = blockPos.method_10093(facing);
            if (level.method_8477(relative))
                neighbours.add(relative);
        }

        class_2680 blockState = be.method_11010();
        if (!(blockState.method_26204() instanceof IRotate block))
            return neighbours;
        return be.addPropagationLocations(block, blockState, neighbours);
    }

}