package com.zurrtum.create.client.vanillin.elements;

import com.zurrtum.create.client.flywheel.api.material.Material;
import com.zurrtum.create.client.flywheel.api.model.Model;
import com.zurrtum.create.client.flywheel.api.vertex.MutableVertexList;
import com.zurrtum.create.client.flywheel.api.visual.DynamicVisual;
import com.zurrtum.create.client.flywheel.api.visualization.VisualizationContext;
import com.zurrtum.create.client.flywheel.lib.instance.InstanceTypes;
import com.zurrtum.create.client.flywheel.lib.instance.TransformedInstance;
import com.zurrtum.create.client.flywheel.lib.material.Materials;
import com.zurrtum.create.client.flywheel.lib.material.SimpleMaterial;
import com.zurrtum.create.client.flywheel.lib.model.QuadMesh;
import com.zurrtum.create.client.flywheel.lib.model.SingleMeshModel;
import com.zurrtum.create.client.flywheel.lib.util.RendererReloadCache;
import com.zurrtum.create.client.flywheel.lib.visual.AbstractVisual;
import com.zurrtum.create.client.flywheel.lib.visual.SimpleDynamicVisual;
import com.zurrtum.create.client.flywheel.lib.visual.util.SmartRecycler;
import net.minecraft.class_1058;
import net.minecraft.class_1088;
import net.minecraft.class_1297;
import net.minecraft.class_310;
import net.minecraft.class_3532;
import net.minecraft.class_4587;
import net.minecraft.class_4730;
import net.minecraft.class_765;
import net.minecraft.class_7833;
import org.joml.Vector4f;
import org.joml.Vector4fc;

/**
 * A component that uses instances to render the fire animation on an entity.
 */
public final class FireElement extends AbstractVisual implements SimpleDynamicVisual {
    private static final Material FIRE_MATERIAL = SimpleMaterial.builderOf(Materials.CUTOUT_UNSHADED_BLOCK)
        .backfaceCulling(false) // Disable backface because we want to be able to flip the model.
        .build();

    // Parameterize by the material instead of the sprite
    // because Material#sprite is a surprisingly heavy operation
    // and because sprites are invalidated after a resource reload.
    private static final RendererReloadCache<class_4730, Model> FIRE_MODELS = new RendererReloadCache<>(texture -> {
        return new SingleMeshModel(new FireMesh(class_310.method_1551().method_72703().method_73030(texture)), FIRE_MATERIAL);
    });

    private final class_1297 entity;
    private final class_4587 stack = new class_4587();

    private final SmartRecycler<Model, TransformedInstance> recycler;

    public FireElement(VisualizationContext ctx, class_1297 entity, float partialTick) {
        super(ctx, entity.method_73183(), partialTick);

        this.entity = entity;

        recycler = new SmartRecycler<>(this::createInstance);
    }

    private TransformedInstance createInstance(Model model) {
        TransformedInstance instance = visualizationContext.instancerProvider().instancer(InstanceTypes.TRANSFORMED, model).createInstance();
        instance.light(class_765.field_32769);
        instance.setChanged();
        return instance;
    }

    /**
     * Update the fire instances. You'd typically call this in your visual's
     * {@link com.zurrtum.create.client.flywheel.lib.visual.SimpleDynamicVisual#beginFrame(DynamicVisual.Context) beginFrame} method.
     *
     * @param context The frame context.
     */
    @Override
    public void beginFrame(DynamicVisual.Context context) {
        recycler.resetCount();

        if (entity.method_5862()) {
            setupInstances(context);
        }

        recycler.discardExtra();
    }

    private void setupInstances(DynamicVisual.Context context) {
        double entityX = class_3532.method_16436(context.partialTick(), entity.field_6038, entity.method_23317());
        double entityY = class_3532.method_16436(context.partialTick(), entity.field_5971, entity.method_23318());
        double entityZ = class_3532.method_16436(context.partialTick(), entity.field_5989, entity.method_23321());
        var renderOrigin = visualizationContext.renderOrigin();

        final float scale = entity.method_17681() * 1.4F;
        final float maxHeight = entity.method_17682() / scale;
        float width = 1;
        float y = 0;
        float z = 0;

        stack.method_34426();
        stack.method_22904(entityX - renderOrigin.method_10263(), entityY - renderOrigin.method_10264(), entityZ - renderOrigin.method_10260());
        stack.method_22905(scale, scale, scale);
        stack.method_22907(class_7833.field_40716.rotationDegrees(-context.camera().method_19330()));
        stack.method_46416(0.0F, 0.0F, -0.3F + (float) ((int) maxHeight) * 0.02F);

        for (int i = 0; y < maxHeight; ++i) {
            var instance = recycler.get(FIRE_MODELS.get(i % 2 == 0 ? class_1088.field_5397 : class_1088.field_5370)).setTransform(stack).scaleX(width)
                .translate(0, y, z);

            if (i / 2 % 2 == 0) {
                // Vanilla flips the uv directly, but it's easier for us to flip the whole model.
                instance.scaleX(-1);
            }

            instance.setChanged();

            y += 0.45F;
            // Get narrower as we go up.
            width *= 0.9F;
            // Offset each one so they don't z-fight.
            z += 0.03F;
        }
    }

    @Override
    public void _delete() {
        recycler.delete();
    }

    private record FireMesh(class_1058 sprite) implements QuadMesh {
        private static final Vector4fc BOUNDING_SPHERE = new Vector4f(0, 0.5f, 0, class_3532.field_15724 * 0.5f);

        @Override
        public int vertexCount() {
            return 4;
        }

        @Override
        public void write(MutableVertexList vertexList) {
            float u0 = sprite.method_4594();
            float v0 = sprite.method_4593();
            float u1 = sprite.method_4577();
            float v1 = sprite.method_4575();
            writeVertex(vertexList, 0, 0.5f, 0, u1, v1);
            writeVertex(vertexList, 1, -0.5f, 0, u0, v1);
            writeVertex(vertexList, 2, -0.5f, 1.4f, u0, v0);
            writeVertex(vertexList, 3, 0.5f, 1.4f, u1, v0);
        }

        // Magic numbers taken from:
        // net.minecraft.client.renderer.entity.EntityRenderDispatcher#fireVertex
        private static void writeVertex(MutableVertexList vertexList, int i, float x, float y, float u, float v) {
            vertexList.x(i, x);
            vertexList.y(i, y);
            vertexList.z(i, 0);
            vertexList.r(i, 1);
            vertexList.g(i, 1);
            vertexList.b(i, 1);
            vertexList.u(i, u);
            vertexList.v(i, v);
            vertexList.light(i, class_765.field_32769);
            vertexList.normalX(i, 0);
            vertexList.normalY(i, 1);
            vertexList.normalZ(i, 0);
        }

        @Override
        public Vector4fc boundingSphere() {
            return BOUNDING_SPHERE;
        }
    }
}
