package com.zurrtum.create.content.contraptions;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import com.mojang.serialization.DynamicOps;
import com.mojang.serialization.ListBuilder;
import com.mojang.serialization.MapLike;
import com.mojang.serialization.RecordBuilder;
import com.zurrtum.create.AllMountedItemStorageTypeTags;
import com.zurrtum.create.Create;
import com.zurrtum.create.api.contraption.storage.SyncedMountedStorage;
import com.zurrtum.create.api.contraption.storage.fluid.MountedFluidStorage;
import com.zurrtum.create.api.contraption.storage.fluid.MountedFluidStorageType;
import com.zurrtum.create.api.contraption.storage.fluid.MountedFluidStorageWrapper;
import com.zurrtum.create.api.contraption.storage.item.MountedItemStorage;
import com.zurrtum.create.api.contraption.storage.item.MountedItemStorageType;
import com.zurrtum.create.api.contraption.storage.item.MountedItemStorageWrapper;
import com.zurrtum.create.catnip.nbt.NBTHelper;
import com.zurrtum.create.foundation.codec.CreateCodecs;
import com.zurrtum.create.infrastructure.items.CombinedInvWrapper;
import com.zurrtum.create.infrastructure.packet.s2c.MountedStorageSyncPacket;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.Predicate;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1263;
import net.minecraft.class_1657;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2487;
import net.minecraft.class_2586;
import net.minecraft.class_2596;
import net.minecraft.class_2602;
import net.minecraft.class_2680;
import net.minecraft.class_3215;
import net.minecraft.class_3222;
import net.minecraft.class_3499.class_3501;
import net.minecraft.class_7225;

public class MountedStorageManager {
    // builders used during assembly, null afterward
    // ImmutableMap.Builder is not used because it will throw with duplicate keys, not override them
    private Map<class_2338, MountedItemStorage> itemsBuilder;
    private Map<class_2338, MountedFluidStorage> fluidsBuilder;
    private Map<class_2338, SyncedMountedStorage> syncedItemsBuilder;
    private Map<class_2338, SyncedMountedStorage> syncedFluidsBuilder;

    // built data structures after assembly, null before
    private ImmutableMap<class_2338, MountedItemStorage> allItemStorages;
    // different from allItemStorages, does not contain internal ones
    protected MountedItemStorageWrapper items;
    @Nullable
    protected MountedItemStorageWrapper fuelItems;
    protected MountedFluidStorageWrapper fluids;

    private ImmutableMap<class_2338, SyncedMountedStorage> syncedItems;
    private ImmutableMap<class_2338, SyncedMountedStorage> syncedFluids;

    private List<class_1263> externalHandlers;
    protected CombinedInvWrapper allItems;

    // ticks until storage can sync again
    private int syncCooldown;

    // client-side: not all storages are synced, this determines which interactions are valid
    private Set<class_2338> interactablePositions;

    public MountedStorageManager() {
        this.reset();
    }

    public void initialize() {
        if (this.isInitialized()) {
            // originally this threw an exception to try to catch mistakes.
            // however, in the case where a Contraption is deserialized before its Entity, that would also throw,
            // since both the deserialization and the onEntityCreated callback initialize the storage.
            // this case occurs when placing a picked up minecart contraption.
            // the reverse case is fine since deserialization also resets the manager first.
            return;
        }

        this.allItemStorages = ImmutableMap.copyOf(this.itemsBuilder);

        this.items = new MountedItemStorageWrapper(subMap(this.allItemStorages, this::isExposed));

        this.allItems = this.items;
        this.itemsBuilder = null;

        ImmutableMap<class_2338, MountedItemStorage> fuelMap = subMap(this.allItemStorages, this::canUseForFuel);
        this.fuelItems = fuelMap.isEmpty() ? null : new MountedItemStorageWrapper(fuelMap);

        ImmutableMap<class_2338, MountedFluidStorage> fluids = ImmutableMap.copyOf(this.fluidsBuilder);
        this.fluids = new MountedFluidStorageWrapper(fluids);
        this.fluidsBuilder = null;

        this.syncedItems = ImmutableMap.copyOf(this.syncedItemsBuilder);
        this.syncedItemsBuilder = null;
        this.syncedFluids = ImmutableMap.copyOf(this.syncedFluidsBuilder);
        this.syncedFluidsBuilder = null;
    }

    private boolean isExposed(MountedItemStorage storage) {
        return !storage.type.is(AllMountedItemStorageTypeTags.INTERNAL);
    }

    private boolean canUseForFuel(MountedItemStorage storage) {
        return this.isExposed(storage) && !storage.type.is(AllMountedItemStorageTypeTags.FUEL_BLACKLIST);
    }

    private boolean isInitialized() {
        return this.itemsBuilder == null;
    }

    private void assertInitialized() {
        if (!this.isInitialized()) {
            throw new IllegalStateException("MountedStorageManager is uninitialized");
        }
    }

    protected void reset() {
        this.allItemStorages = null;
        this.items = null;
        this.fuelItems = null;
        this.fluids = null;
        this.externalHandlers = new ArrayList<>();
        this.allItems = null;
        this.itemsBuilder = new HashMap<>();
        this.fluidsBuilder = new HashMap<>();
        this.syncedItemsBuilder = new HashMap<>();
        this.syncedFluidsBuilder = new HashMap<>();
        // interactablePositions intentionally not reset
    }

    public void addBlock(class_1937 level, class_2680 state, class_2338 globalPos, class_2338 localPos, @Nullable class_2586 be) {
        MountedItemStorageType<?> itemType = MountedItemStorageType.REGISTRY.get(state.method_26204());
        if (itemType != null) {
            MountedItemStorage storage = itemType.mount(level, state, globalPos, be);
            if (storage != null) {
                this.addStorage(storage, localPos);
            }
        }

        MountedFluidStorageType<?> fluidType = MountedFluidStorageType.REGISTRY.get(state.method_26204());
        if (fluidType != null) {
            MountedFluidStorage storage = fluidType.mount(level, state, globalPos, be);
            if (storage != null) {
                this.addStorage(storage, localPos);
            }
        }
    }

    public void unmount(class_1937 level, class_3501 info, class_2338 globalPos, @Nullable class_2586 be) {
        class_2338 localPos = info.comp_1341();
        class_2680 state = info.comp_1342();

        MountedItemStorage itemStorage = this.getAllItemStorages().get(localPos);
        if (itemStorage != null) {
            MountedItemStorageType<?> expectedType = MountedItemStorageType.REGISTRY.get(state.method_26204());
            if (itemStorage.type == expectedType) {
                itemStorage.unmount(level, state, globalPos, be);
            }
        }

        MountedFluidStorage fluidStorage = this.getFluids().storages.get(localPos);
        if (fluidStorage != null) {
            MountedFluidStorageType<?> expectedType = MountedFluidStorageType.REGISTRY.get(state.method_26204());
            if (fluidStorage.type == expectedType) {
                fluidStorage.unmount(level, state, globalPos, be);
            }
        }
    }

    public void tick(AbstractContraptionEntity entity) {
        if (syncCooldown > 0) {
            syncCooldown--;
            return;
        }

        Map<class_2338, MountedItemStorage> items = new HashMap<>();
        Map<class_2338, MountedFluidStorage> fluids = new HashMap<>();
        syncedItems.forEach((pos, storage) -> {
            if (storage.isDirty()) {
                items.put(pos, (MountedItemStorage) storage);
                storage.markClean();
            }
        });
        syncedFluids.forEach((pos, storage) -> {
            if (storage.isDirty()) {
                fluids.put(pos, (MountedFluidStorage) storage);
                storage.markClean();
            }
        });

        if (!items.isEmpty() || !fluids.isEmpty()) {
            class_2596<class_2602> packet = new MountedStorageSyncPacket(entity.method_5628(), items, fluids);
            ((class_3215) entity.method_73183().method_8398()).method_18754(entity, packet);
            syncCooldown = 8;
        }
    }

    public void handleSync(MountedStorageSyncPacket packet, AbstractContraptionEntity entity) {
        // packet only contains changed storages, grab existing ones before resetting
        ImmutableMap<class_2338, MountedItemStorage> items = this.getAllItemStorages();
        MountedFluidStorageWrapper fluids = this.getFluids();
        this.reset();

        // track freshly synced storages
        Map<SyncedMountedStorage, class_2338> syncedStorages = new IdentityHashMap<>();

        try {
            // re-add existing ones
            this.itemsBuilder.putAll(items);
            this.fluidsBuilder.putAll(fluids.storages);
            // add newly synced ones, overriding existing ones if present
            packet.items().forEach((pos, storage) -> {
                this.itemsBuilder.put(pos, storage);
                syncedStorages.put((SyncedMountedStorage) storage, pos);
            });
            packet.fluids().forEach((pos, storage) -> {
                this.fluidsBuilder.put(pos, storage);
                syncedStorages.put((SyncedMountedStorage) storage, pos);
            });
        } catch (Throwable t) {
            // an exception will leave the manager in an invalid state
            Create.LOGGER.error("An error occurred while syncing a MountedStorageManager", t);
        }

        this.initialize();

        // call all afterSync methods
        Contraption contraption = entity.getContraption();
        syncedStorages.forEach((storage, pos) -> storage.afterSync(contraption, pos));
    }

    // contraption is provided on the client for initial afterSync storage callbacks
    public void read(class_11368 view, boolean clientPacket, @Nullable Contraption contraption) {
        reset();

        try {
            view.method_71438("items").forEach(item -> {
                class_2338 pos = item.method_71426("pos", class_2338.field_25064).orElseThrow();
                MountedItemStorage storage = item.method_71426("storage", MountedItemStorage.CODEC).orElseThrow();
                addStorage(storage, pos);
            });

            view.method_71438("fluids").forEach(fluid -> {
                class_2338 pos = fluid.method_71426("pos", class_2338.field_25064).orElseThrow();
                MountedFluidStorage storage = fluid.method_71426("storage", MountedFluidStorage.CODEC).orElseThrow();
                addStorage(storage, pos);
            });

            view.method_71426("interactable_positions", CreateCodecs.BLOCK_POS_LIST_CODEC).ifPresent(list -> interactablePositions = new HashSet<>(list));
        } catch (Throwable t) {
            Create.LOGGER.error("Error deserializing mounted storage", t);
            // an exception will leave the manager in an invalid state, initialize must be called
        }

        initialize();
        afterSync(clientPacket, contraption);
    }

    public <T> void read(final DynamicOps<T> ops, MapLike<T> map, boolean clientPacket, @Nullable Contraption contraption) {
        reset();

        try {
            ops.getList(map.get("items")).getOrThrow().accept(item -> {
                MapLike<T> data = ops.getMap(item).getOrThrow();
                class_2338 pos = class_2338.field_25064.parse(ops, data.get("pos")).getOrThrow();
                MountedItemStorage storage = MountedItemStorage.CODEC.parse(ops, data.get("storage")).getOrThrow();
                addStorage(storage, pos);
            });

            ops.getList(map.get("fluids")).getOrThrow().accept(fluid -> {
                MapLike<T> data = ops.getMap(fluid).getOrThrow();
                class_2338 pos = class_2338.field_25064.parse(ops, data.get("pos")).getOrThrow();
                MountedFluidStorage storage = MountedFluidStorage.CODEC.parse(ops, data.get("storage")).getOrThrow();
                addStorage(storage, pos);
            });

            CreateCodecs.BLOCK_POS_LIST_CODEC.parse(ops, map.get("interactable_positions"))
                .ifSuccess(list -> interactablePositions = new HashSet<>(list));
        } catch (Throwable t) {
            Create.LOGGER.error("Error deserializing mounted storage", t);
            // an exception will leave the manager in an invalid state, initialize must be called
        }

        initialize();
        afterSync(clientPacket, contraption);
    }

    private void afterSync(boolean clientPacket, @Nullable Contraption contraption) {
        // for client sync, run initial afterSync callbacks
        if (!clientPacket || contraption == null)
            return;

        getAllItemStorages().forEach((pos, storage) -> {
            if (storage instanceof SyncedMountedStorage synced) {
                synced.afterSync(contraption, pos);
            }
        });
        getFluids().storages.forEach((pos, storage) -> {
            if (storage instanceof SyncedMountedStorage synced) {
                synced.afterSync(contraption, pos);
            }
        });
    }

    public void write(class_11372 view, boolean clientPacket) {
        class_11372.class_11374 items = view.method_71476("items");
        getAllItemStorages().forEach((pos, storage) -> {
            if (!clientPacket || storage instanceof SyncedMountedStorage) {
                class_11372 item = items.method_71480();
                item.method_71468("pos", class_2338.field_25064, pos);
                item.method_71468("storage", MountedItemStorage.CODEC, storage);
            }
        });

        class_11372.class_11374 fluids = view.method_71476("fluids");
        getFluids().storages.forEach((pos, storage) -> {
            if (!clientPacket || storage instanceof SyncedMountedStorage) {
                class_11372 fluid = fluids.method_71480();
                fluid.method_71468("pos", class_2338.field_25064, pos);
                fluid.method_71468("storage", MountedFluidStorage.CODEC, storage);
            }
        });

        if (clientPacket) {
            // let the client know of all non-synced ones too
            List<class_2338> list = Sets.union(this.getAllItemStorages().keySet(), getFluids().storages.keySet()).stream().toList();
            view.method_71468("interactable_positions", CreateCodecs.BLOCK_POS_LIST_CODEC, list);
        }
    }

    public <T> void write(final DynamicOps<T> ops, final T empty, RecordBuilder<T> map, boolean clientPacket) {
        ListBuilder<T> items = ops.listBuilder();
        getAllItemStorages().forEach((pos, storage) -> {
            if (!clientPacket || storage instanceof SyncedMountedStorage) {
                RecordBuilder<T> item = ops.mapBuilder();
                item.add("pos", pos, class_2338.field_25064);
                item.add("storage", storage, MountedItemStorage.CODEC);
                items.add(item.build(empty));
            }
        });
        map.add("items", items.build(empty));

        ListBuilder<T> fluids = ops.listBuilder();
        getFluids().storages.forEach((pos, storage) -> {
            if (!clientPacket || storage instanceof SyncedMountedStorage) {
                RecordBuilder<T> fluid = ops.mapBuilder();
                fluid.add("pos", pos, class_2338.field_25064);
                fluid.add("storage", storage, MountedFluidStorage.CODEC);
                fluids.add(fluid.build(empty));
            }
        });
        map.add("fluids", fluids.build(empty));

        if (clientPacket) {
            // let the client know of all non-synced ones too
            List<class_2338> list = Sets.union(this.getAllItemStorages().keySet(), getFluids().storages.keySet()).stream().toList();
            map.add("interactable_positions", list, CreateCodecs.BLOCK_POS_LIST_CODEC);
        }
    }

    public void attachExternal(class_1263 externalStorage) {
        this.externalHandlers.add(externalStorage);
        int size = externalHandlers.size();
        class_1263[] all = new class_1263[size + 1];
        all[0] = this.items;
        for (int i = 0; i < size; i++) {
            all[i + 1] = externalHandlers.get(i);
        }

        this.allItems = new CombinedInvWrapper(all);
    }

    /**
     * The primary way to access a contraption's inventory. Includes all
     * non-internal mounted storages as well as all external storage.
     */
    public CombinedInvWrapper getAllItems() {
        this.assertInitialized();
        return this.allItems;
    }

    /**
     * Gets a map of all MountedItemStorages in the contraption, irrelevant of them being internal or providing fuel.
     */
    public ImmutableMap<class_2338, MountedItemStorage> getAllItemStorages() {
        this.assertInitialized();
        return this.allItemStorages;
    }

    /**
     * Gets an item handler wrapping all non-internal mounted storages. This is not
     * the whole contraption inventory as it does not include external storages.
     * Most often, you want {@link #getAllItems()}, which does.
     */
    public MountedItemStorageWrapper getMountedItems() {
        this.assertInitialized();
        return this.items;
    }

    /**
     * Gets an item handler wrapping all non-internal mounted storages that provide fuel.
     * May be null if none are present.
     */
    @Nullable
    public MountedItemStorageWrapper getFuelItems() {
        this.assertInitialized();
        return this.fuelItems;
    }

    /**
     * Gets a fluid handler wrapping all mounted fluid storages.
     */
    public MountedFluidStorageWrapper getFluids() {
        this.assertInitialized();
        return this.fluids;
    }

    public boolean handlePlayerStorageInteraction(Contraption contraption, class_1657 player, class_2338 localPos) {
        if (!(player instanceof class_3222 serverPlayer)) {
            return this.interactablePositions != null && this.interactablePositions.contains(localPos);
        }

        class_3501 info = contraption.getBlocks().get(localPos);
        if (info == null)
            return false;

        MountedStorageManager storageManager = contraption.getStorage();
        MountedItemStorage storage = storageManager.getAllItemStorages().get(localPos);

        if (storage != null) {
            return storage.handleInteraction(serverPlayer, contraption, info);
        } else {
            return false;
        }
    }

    private void readLegacy(class_7225.class_7874 registries, class_2487 nbt) {
        NBTHelper.iterateCompoundList(
            nbt.method_68569("Storage"), tag -> {
                class_2338 pos = NBTHelper.readBlockPos(tag, "Pos");
                class_2487 data = tag.method_68568("Data");

                //TODO
                //                if (data.contains("Toolbox")) {
                //                    this.addStorage(ToolboxMountedStorage.fromLegacy(registries, data), pos);
                //                } else if (data.contains("NoFuel")) {
                //                    this.addStorage(ItemVaultMountedStorage.fromLegacy(registries, data), pos);
                //                } else if (data.contains("Bottomless")) {
                //                    ItemStack supplied = data.getCompound("ProvidedStack").flatMap(c -> ItemStack.fromNbt(registries, c)).orElse(ItemStack.EMPTY);
                //                    this.addStorage(new CreativeCrateMountedStorage(supplied), pos);
                //                } else if (data.contains("Synced")) {
                //                    this.addStorage(DepotMountedStorage.fromLegacy(registries, data), pos);
                //                } else {
                //                    // we can create a fallback storage safely, it will be validated before unmounting
                //                    //                    ItemStackHandler handler = new ItemStackHandler();
                //                    //                    handler.deserializeNBT(registries, data);
                //                    this.addStorage(new FallbackMountedStorage(new Object()), pos);
                //                }
            }
        );

        NBTHelper.iterateCompoundList(
            nbt.method_68569("FluidStorage"), tag -> {
                class_2338 pos = NBTHelper.readBlockPos(tag, "Pos");
                class_2487 data = tag.method_68568("Data");

                //TODO
                //                if (data.contains("Bottomless")) {
                //                    this.addStorage(CreativeFluidTankMountedStorage.fromLegacy(registries, data), pos);
                //                } else {
                //                    this.addStorage(FluidTankMountedStorage.fromLegacy(registries, data), pos);
                //                }
            }
        );
    }

    private void addStorage(MountedItemStorage storage, class_2338 pos) {
        this.itemsBuilder.put(pos, storage);
        if (storage instanceof SyncedMountedStorage synced)
            this.syncedItemsBuilder.put(pos, synced);
    }

    private void addStorage(MountedFluidStorage storage, class_2338 pos) {
        this.fluidsBuilder.put(pos, storage);
        if (storage instanceof SyncedMountedStorage synced)
            this.syncedFluidsBuilder.put(pos, synced);
    }

    private static <K, V> ImmutableMap<K, V> subMap(Map<K, V> map, Predicate<V> predicate) {
        ImmutableMap.Builder<K, V> builder = ImmutableMap.builder();
        map.forEach((key, value) -> {
            if (predicate.test(value)) {
                builder.put(key, value);
            }
        });
        return builder.build();
    }
}
