package com.zurrtum.create.content.contraptions.bearing;

import com.zurrtum.create.AllAdvancements;
import com.zurrtum.create.AllBlockEntityTypes;
import com.zurrtum.create.AllClientHandle;
import com.zurrtum.create.AllSoundEvents;
import com.zurrtum.create.catnip.math.AngleHelper;
import com.zurrtum.create.content.contraptions.AbstractContraptionEntity;
import com.zurrtum.create.content.contraptions.AssemblyException;
import com.zurrtum.create.content.contraptions.ControlledContraptionEntity;
import com.zurrtum.create.content.kinetics.base.GeneratingKineticBlockEntity;
import com.zurrtum.create.content.kinetics.transmission.sequencer.SequencerInstructions;
import com.zurrtum.create.foundation.advancement.CreateTrigger;
import com.zurrtum.create.foundation.blockEntity.behaviour.BlockEntityBehaviour;
import com.zurrtum.create.foundation.blockEntity.behaviour.scrollValue.ServerScrollOptionBehaviour;
import java.util.List;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2591;
import net.minecraft.class_2680;
import net.minecraft.class_2741;
import net.minecraft.class_3532;

public class MechanicalBearingBlockEntity extends GeneratingKineticBlockEntity implements IBearingBlockEntity {

    protected ServerScrollOptionBehaviour<RotationMode> movementMode;
    protected ControlledContraptionEntity movedContraption;
    protected float angle;
    protected boolean running;
    protected boolean assembleNextTick;
    protected float clientAngleDiff;
    public AssemblyException lastException;
    protected double sequencedAngleLimit;

    private float prevAngle;

    public MechanicalBearingBlockEntity(class_2591<?> type, class_2338 pos, class_2680 state) {
        super(type, pos, state);
        setLazyTickRate(3);
        sequencedAngleLimit = -1;
    }

    public MechanicalBearingBlockEntity(class_2338 pos, class_2680 state) {
        this(AllBlockEntityTypes.MECHANICAL_BEARING, pos, state);
    }

    @Override
    public boolean isWoodenTop() {
        return false;
    }

    @Override
    protected boolean syncSequenceContext() {
        return true;
    }

    @Override
    public void addBehaviours(List<BlockEntityBehaviour<?>> behaviours) {
        super.addBehaviours(behaviours);
        movementMode = new ServerScrollOptionBehaviour<>(RotationMode.class, this);
        behaviours.add(movementMode);
    }

    @Override
    public List<CreateTrigger> getAwardables() {
        return List.of(AllAdvancements.CONTRAPTION_ACTORS);
    }

    @Override
    public void remove() {
        if (!field_11863.method_8608())
            disassemble();
        super.remove();
    }

    @Override
    public void write(class_11372 view, boolean clientPacket) {
        view.method_71472("Running", running);
        view.method_71464("Angle", angle);
        if (sequencedAngleLimit >= 0)
            view.method_71463("SequencedAngleLimit", sequencedAngleLimit);
        if (lastException != null) {
            view.method_71468("LastException", AssemblyException.CODEC, lastException);
        }
        super.write(view, clientPacket);
    }

    @Override
    protected void read(class_11368 view, boolean clientPacket) {
        if (wasMoved) {
            super.read(view, clientPacket);
            return;
        }

        float angleBefore = angle;
        running = view.method_71433("Running", false);
        angle = view.method_71423("Angle", 0);
        sequencedAngleLimit = view.method_71422("SequencedAngleLimit", -1);
        lastException = view.method_71426("LastException", AssemblyException.CODEC).orElse(null);
        super.read(view, clientPacket);
        if (!clientPacket)
            return;
        if (running) {
            if (movedContraption == null || !movedContraption.isStalled()) {
                clientAngleDiff = AngleHelper.getShortestAngleDiff(angleBefore, angle);
                angle = angleBefore;
            }
        } else
            movedContraption = null;
    }

    @Override
    public float getInterpolatedAngle(float partialTicks) {
        if (isVirtual())
            return class_3532.method_16439(partialTicks + .5f, prevAngle, angle);
        if (movedContraption == null || movedContraption.isStalled() || !running)
            partialTicks = 0;
        float angularSpeed = getAngularSpeed();
        if (sequencedAngleLimit >= 0)
            angularSpeed = (float) class_3532.method_15350(angularSpeed, -sequencedAngleLimit, sequencedAngleLimit);
        return class_3532.method_16439(partialTicks, angle, angle + angularSpeed);
    }

    @Override
    public void onSpeedChanged(float prevSpeed) {
        super.onSpeedChanged(prevSpeed);
        assembleNextTick = true;
        sequencedAngleLimit = -1;

        if (movedContraption != null && Math.signum(prevSpeed) != Math.signum(getSpeed()) && prevSpeed != 0) {
            if (!movedContraption.isStalled()) {
                angle = Math.round(angle);
                applyRotation();
            }
            movedContraption.getContraption().stop(field_11863);
        }

        if (!isWindmill() && sequenceContext != null && sequenceContext.instruction() == SequencerInstructions.TURN_ANGLE)
            sequencedAngleLimit = sequenceContext.getEffectiveValue(getTheoreticalSpeed());
    }

    public float getAngularSpeed() {
        float speed = convertToAngular(isWindmill() ? getGeneratedSpeed() : getSpeed());
        if (getSpeed() == 0)
            speed = 0;
        if (field_11863.method_8608()) {
            speed *= AllClientHandle.INSTANCE.getServerSpeed();
            speed += clientAngleDiff / 3f;
        }
        return speed;
    }

    public AssemblyException getLastAssemblyException() {
        return lastException;
    }

    public boolean isWindmill() {
        return false;
    }

    public void assemble() {
        if (!(field_11863.method_8320(field_11867).method_26204() instanceof BearingBlock))
            return;

        class_2350 direction = method_11010().method_11654(BearingBlock.FACING);
        BearingContraption contraption = new BearingContraption(isWindmill(), direction);
        try {
            if (!contraption.assemble(field_11863, field_11867))
                return;

            lastException = null;
        } catch (AssemblyException e) {
            lastException = e;
            sendData();
            return;
        }

        if (isWindmill())
            award(AllAdvancements.WINDMILL);
        if (contraption.getSailBlocks() >= 16 * 8)
            award(AllAdvancements.WINDMILL_MAXED);

        contraption.removeBlocksFromWorld(field_11863, class_2338.field_10980);
        movedContraption = ControlledContraptionEntity.create(field_11863, this, contraption);
        class_2338 anchor = field_11867.method_10093(direction);
        movedContraption.method_5814(anchor.method_10263(), anchor.method_10264(), anchor.method_10260());
        movedContraption.setRotationAxis(direction.method_10166());
        field_11863.method_8649(movedContraption);

        AllSoundEvents.CONTRAPTION_ASSEMBLE.playOnServer(field_11863, field_11867);

        if (contraption.containsBlockBreakers())
            award(AllAdvancements.CONTRAPTION_ACTORS);

        running = true;
        angle = 0;
        sendData();
        updateGeneratedRotation();
    }

    public void disassemble() {
        if (!running && movedContraption == null)
            return;
        angle = 0;
        sequencedAngleLimit = -1;
        if (isWindmill())
            applyRotation();
        if (movedContraption != null) {
            movedContraption.disassemble();
            AllSoundEvents.CONTRAPTION_DISASSEMBLE.playOnServer(field_11863, field_11867);
        }

        movedContraption = null;
        running = false;
        updateGeneratedRotation();
        assembleNextTick = false;
        sendData();
    }

    @Override
    public void tick() {
        super.tick();

        prevAngle = angle;
        if (field_11863.method_8608())
            clientAngleDiff /= 2;

        if (!field_11863.method_8608() && assembleNextTick) {
            assembleNextTick = false;
            if (running) {
                boolean canDisassemble = movementMode.get() == RotationMode.ROTATE_PLACE || (isNearInitialAngle() && movementMode.get() == RotationMode.ROTATE_PLACE_RETURNED);
                if (speed == 0 && (canDisassemble || movedContraption == null || movedContraption.getContraption().getBlocks().isEmpty())) {
                    if (movedContraption != null)
                        movedContraption.getContraption().stop(field_11863);
                    disassemble();
                    return;
                }
            } else {
                if (speed == 0 && !isWindmill())
                    return;
                assemble();
            }
        }

        if (!running)
            return;

        if (!(movedContraption != null && movedContraption.isStalled())) {
            float angularSpeed = getAngularSpeed();
            if (sequencedAngleLimit >= 0) {
                angularSpeed = (float) class_3532.method_15350(angularSpeed, -sequencedAngleLimit, sequencedAngleLimit);
                sequencedAngleLimit = Math.max(0, sequencedAngleLimit - Math.abs(angularSpeed));
            }
            angle = (angle + angularSpeed) % 360;
        }

        applyRotation();
    }

    public boolean isNearInitialAngle() {
        return Math.abs(angle) < 22.5 || Math.abs(angle) > 360 - 22.5;
    }

    @Override
    public void lazyTick() {
        super.lazyTick();
        if (movedContraption != null && !field_11863.method_8608())
            sendData();
    }

    protected void applyRotation() {
        if (movedContraption == null)
            return;
        movedContraption.setAngle(angle);
        class_2680 blockState = method_11010();
        if (blockState.method_28498(class_2741.field_12525))
            movedContraption.setRotationAxis(blockState.method_11654(class_2741.field_12525).method_10166());
    }

    @Override
    public void attach(ControlledContraptionEntity contraption) {
        class_2680 blockState = method_11010();
        if (!(contraption.getContraption() instanceof BearingContraption))
            return;
        if (!blockState.method_28498(BearingBlock.FACING))
            return;

        this.movedContraption = contraption;
        method_5431();
        class_2338 anchor = field_11867.method_10093(blockState.method_11654(BearingBlock.FACING));
        movedContraption.method_5814(anchor.method_10263(), anchor.method_10264(), anchor.method_10260());
        if (!field_11863.method_8608()) {
            this.running = true;
            sendData();
        }
    }

    @Override
    public void onStall() {
        if (!field_11863.method_8608())
            sendData();
    }

    @Override
    public boolean isValid() {
        return !method_11015();
    }

    @Override
    public boolean isAttachedTo(AbstractContraptionEntity contraption) {
        return movedContraption == contraption;
    }

    public boolean isRunning() {
        return running;
    }

    public void setAngle(float forcedAngle) {
        angle = forcedAngle;
    }

    public ControlledContraptionEntity getMovedContraption() {
        return movedContraption;
    }

    @Override
    public class_2338 getBlockPosition() {
        return field_11867;
    }
}
