package com.zurrtum.create.foundation.recipe.trie;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.zurrtum.create.Create;
import com.zurrtum.create.content.processing.basin.BasinRecipe;
import com.zurrtum.create.content.processing.recipe.SizedIngredient;
import com.zurrtum.create.foundation.fluid.FluidIngredient;
import com.zurrtum.create.infrastructure.fluids.FluidInventory;
import com.zurrtum.create.infrastructure.fluids.FluidStack;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.concurrent.ExecutionException;
import net.minecraft.class_1263;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1856;
import net.minecraft.class_1860;
import net.minecraft.class_1867;
import net.minecraft.class_1869;
import net.minecraft.class_3611;

public class RecipeTrie<R extends class_1860<?>> {
    private static final int MAX_CACHE_SIZE = Integer.getInteger("create.recipe_trie.max_cache_size", 512);

    private final IntArrayTrie<R> trie;
    private final Object2IntMap<AbstractVariant> variantToId;
    private final Int2ObjectMap<IntSet> variantToIngredients;
    private final int universalIngredientId;

    private final Cache<Set<AbstractVariant>, IntSet> ingredientCache = CacheBuilder.newBuilder().maximumSize(MAX_CACHE_SIZE).build();

    private RecipeTrie(
        IntArrayTrie<R> trie,
        Object2IntMap<AbstractVariant> variantToId,
        Int2ObjectMap<IntSet> variantToIngredients,
        int universalIngredientId
    ) {
        this.trie = trie;
        this.variantToId = variantToId;
        this.variantToIngredients = variantToIngredients;
        this.universalIngredientId = universalIngredientId;
    }

    public static @NotNull Set<AbstractVariant> getVariants(@Nullable class_1263 itemStorage, @Nullable FluidInventory fluidStorage) {
        Set<AbstractVariant> variants = new HashSet<>();

        if (itemStorage != null) {
            for (int slot = 0, size = itemStorage.method_5439(); slot < size; slot++) {
                class_1799 item = itemStorage.method_5438(slot);
                if (item.method_7960())
                    continue;

                variants.add(new AbstractVariant.AbstractItem(item.method_7909()));
            }
        }

        if (fluidStorage != null) {
            for (int tank = 0, size = fluidStorage.size(); tank < size; tank++) {
                FluidStack fluid = fluidStorage.getStack(tank);
                if (fluid.isEmpty())
                    continue;

                variants.add(new AbstractVariant.AbstractFluid(fluid.getFluid()));
            }
        }

        return variants;
    }

    private IntSet getAvailableIngredients(@NotNull Set<AbstractVariant> pool) {
        pool.retainAll(variantToId.keySet());

        try {
            return ingredientCache.get(
                Set.copyOf(pool), () -> {
                    IntSet ingredients = new IntOpenHashSet();
                    ingredients.add(universalIngredientId);

                    for (AbstractVariant variant : pool) {
                        int id = variantToId.getInt(variant);
                        if (id >= 0) {
                            var ingredientIds = variantToIngredients.get(id);
                            if (ingredientIds != null) {
                                ingredients.addAll(ingredientIds);
                            }
                        }
                    }

                    return ingredients;
                }
            );
        } catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Look up all recipes that can be made with (a subset of) the given pool of resources.
     *
     * @param pool the set of allowable variants. It will be modified to only contain known variants.
     * @return all recipes that can be made with the given pool of resources.
     */
    public @NotNull List<R> lookup(@NotNull Set<AbstractVariant> pool) {
        return trie.lookup(getAvailableIngredients(pool));
    }

    public static <R extends class_1860<?>> Builder<R> builder() {
        return new Builder<>();
    }

    public static class Builder<R extends class_1860<?>> {
        private final IntArrayTrie<R> trie = new IntArrayTrie<>();

        private final Map<Object, AbstractVariant> variantCache = new HashMap<>();
        private final Object2IntOpenHashMap<AbstractVariant> variantToId = new Object2IntOpenHashMap<>();
        private int nextVariantId = 0;

        private final Object2IntMap<AbstractIngredient> ingredientToId = new Object2IntOpenHashMap<>();
        private int nextIngredientId = 0;
        private final int universalIngredientId;

        private final Int2ObjectOpenHashMap<IntSet> variantToIngredients = new Int2ObjectOpenHashMap<>();

        private Builder() {
            variantToId.defaultReturnValue(-1);
            ingredientToId.defaultReturnValue(-1);

            universalIngredientId = getOrAssignId(AbstractIngredient.Universal.INSTANCE);
        }

        private int getOrAssignId(AbstractIngredient ingredient) {
            return ingredientToId.computeIfAbsent(
                ingredient, $ -> {
                    int id = nextIngredientId++;
                    for (AbstractVariant variant : ingredient.variants) {
                        variantToIngredients.computeIfAbsent(getOrAssignId(variant), $1 -> new IntOpenHashSet()).add(id);
                    }
                    return id;
                }
            );
        }

        private int getOrAssignId(AbstractVariant variant) {
            return variantToId.computeIfAbsent(variant, $ -> nextVariantId++);
        }

        private AbstractVariant getOrAssignVariant(class_1792 item) {
            AbstractVariant variant = variantCache.computeIfAbsent(item, $ -> new AbstractVariant.AbstractItem(item));
            getOrAssignId(variant);
            return variant;
        }

        private AbstractVariant getOrAssignVariant(class_3611 fluid) {
            AbstractVariant variant = variantCache.computeIfAbsent(fluid, $ -> new AbstractVariant.AbstractFluid(fluid));
            getOrAssignId(variant);
            return variant;
        }

        private void insert(AbstractRecipe<? extends R> recipe) {
            int[] key = new int[recipe.ingredients.size()];
            int i = 0;
            for (AbstractIngredient ingredient : recipe.ingredients) {
                key[i++] = getOrAssignId(ingredient);
            }
            Arrays.sort(key);
            trie.insert(key, recipe.recipe);
        }

        /**
         * Insert a recipe into the trie.
         * <br/>
         * Will handle item ingredients for all recipes, and fluid ingredients for {@link BasinRecipe}s.
         */
        public <R1 extends R> void insert(R1 recipe) {
            insert(createRecipe(recipe));
        }

        @SuppressWarnings("deprecation")
        private <R1 extends R> AbstractRecipe<R1> createRecipe(R1 recipe) {
            Set<AbstractIngredient> ingredients = new HashSet<>();

            Iterator<class_1856> items = null;
            List<FluidIngredient> fluids = null;
            if (recipe instanceof class_1869 shapedRecipe) {
                items = shapedRecipe.method_61693().stream().filter(Optional::isPresent).map(Optional::get).iterator();
            } else if (recipe instanceof class_1867 shapelessRecipe) {
                items = shapelessRecipe.field_9047.iterator();
            } else if (recipe instanceof BasinRecipe basinRecipe) {
                items = basinRecipe.getIngredients().stream().map(SizedIngredient::getIngredient).iterator();
                fluids = basinRecipe.getFluidIngredients();
            }
            if (items != null) {
                while (items.hasNext()) {
                    class_1856 ingredient = items.next();
                    if (ingredient.method_65799()) {
                        ingredients.add(AbstractIngredient.Universal.INSTANCE);
                        continue;
                    }
                    Set<AbstractVariant> variants = new HashSet<>();
                    ingredient.method_8105().forEach(entry -> variants.add(getOrAssignVariant(entry.comp_349())));
                    ingredients.add(new AbstractIngredient(variants));
                }
            }

            if (fluids != null) {
                for (FluidIngredient ingredient : fluids) {
                    if (ingredient.amount() == 0) {
                        ingredients.add(AbstractIngredient.Universal.INSTANCE);
                        continue;
                    }
                    Set<AbstractVariant> variants = new HashSet<>();
                    for (class_3611 fluid : ingredient.getMatchingFluids()) {
                        variants.add(getOrAssignVariant(fluid));
                    }
                    ingredients.add(new AbstractIngredient(variants));
                }
            }

            return new AbstractRecipe<>(recipe, ingredients);
        }

        public RecipeTrie<R> build() {
            variantToId.trim();
            variantToIngredients.trim();
            Create.LOGGER.info(
                "RecipeTrie of depth {} with {} nodes built with {} variants, {} ingredients, and {} recipes",
                trie.getMaxDepth(),
                trie.getNodeCount(),
                variantToId.size(),
                ingredientToId.size(),
                trie.getValueCount()
            );
            return new RecipeTrie<>(trie, variantToId, variantToIngredients, universalIngredientId);
        }
    }
}