package com.zurrtum.create.api.contraption;

import com.zurrtum.create.api.registry.SimpleRegistry;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.function.Supplier;
import net.minecraft.class_1937;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2680;
import net.minecraft.class_3499;

/**
 * Defines whether a block is movable by contraptions.
 * This is used as a fallback check for {@link BlockMovementChecks#isMovementAllowed(class_2680, class_1937, class_2338)}.
 * The registry uses suppliers, so the setting of a block can change. This is useful for config options.
 */
public enum ContraptionMovementSetting {
    /**
     * Block is fully movable with no restrictions.
     */
    MOVABLE,
    /**
     * Block can be mounted and moved, but if it's on a minecart contraption, the contraption cannot be picked up.
     */
    NO_PICKUP,
    /**
     * Block cannot ever be moved by a contraption.
     */
    UNMOVABLE;

    public static final SimpleRegistry<class_2248, Supplier<ContraptionMovementSetting>> REGISTRY = SimpleRegistry.create();

    /**
     * Shortcut that gets the block of the given state.
     */
    @Nullable
    public static ContraptionMovementSetting get(class_2680 state) {
        return get(state.method_26204());
    }

    /**
     * Get the current movement setting of the given block.
     */
    @Nullable
    public static ContraptionMovementSetting get(class_2248 block) {
        if (block instanceof MovementSettingProvider provider)
            return provider.getContraptionMovementSetting();
        Supplier<ContraptionMovementSetting> supplier = REGISTRY.get(block);
        return supplier == null ? null : supplier.get();
    }

    /**
     * Check if any of the blocks in the collection match the given setting.
     */
    public static boolean anyAre(Collection<class_3499.class_3501> blocks, ContraptionMovementSetting setting) {
        return blocks.stream().anyMatch(b -> get(b.comp_1342().method_26204()) == setting);
    }

    /**
     * Check if any of the blocks in the collection forbid pickup.
     */
    public static boolean isNoPickup(Collection<class_3499.class_3501> blocks) {
        return anyAre(blocks, ContraptionMovementSetting.NO_PICKUP);
    }

    /**
     * Interface that may optionally be implemented on a Block implementation which will be queried instead of the registry.
     */
    public interface MovementSettingProvider {
        ContraptionMovementSetting getContraptionMovementSetting();
    }
}