package com.zurrtum.create.client.ponder.foundation;


import com.zurrtum.create.catnip.math.Pointing;
import com.zurrtum.create.catnip.math.VecHelper;
import com.zurrtum.create.catnip.theme.Color;
import com.zurrtum.create.client.ponder.api.ParticleEmitter;
import com.zurrtum.create.client.ponder.api.PonderPalette;
import com.zurrtum.create.client.ponder.api.element.*;
import com.zurrtum.create.client.ponder.api.element.MinecartElement.MinecartConstructor;
import com.zurrtum.create.client.ponder.api.level.PonderLevel;
import com.zurrtum.create.client.ponder.api.scene.*;
import com.zurrtum.create.client.ponder.foundation.element.*;
import com.zurrtum.create.client.ponder.foundation.instruction.*;
import net.minecraft.class_1297;
import net.minecraft.class_1542;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2350.class_2351;
import net.minecraft.class_238;
import net.minecraft.class_2382;
import net.minecraft.class_2390;
import net.minecraft.class_2394;
import net.minecraft.class_243;
import net.minecraft.class_2459;
import net.minecraft.class_2487;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_2741;
import net.minecraft.class_2769;
import net.minecraft.class_2960;
import net.minecraft.class_7225;
import net.minecraft.util.math.*;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;

/**
 * Enqueue instructions to the schedule via this object's methods.
 */
public class PonderSceneBuilder implements SceneBuilder {

    private final OverlayInstructions overlay;
    private final WorldInstructions world;
    private final DebugInstructions debug;
    private final EffectInstructions effects;
    private final SpecialInstructions special;

    protected final PonderScene scene;

    public PonderSceneBuilder(PonderScene ponderScene) {
        scene = ponderScene;
        overlay = new PonderOverlayInstructions();
        special = new PonderSpecialInstructions();
        world = new PonderWorldInstructions();
        debug = new PonderDebugInstructions();
        effects = new PonderEffectInstructions();
    }

    @Override
    public OverlayInstructions overlay() {
        return overlay;
    }

    @Override
    public WorldInstructions world() {
        return world;
    }

    @Override
    public DebugInstructions debug() {
        return debug;
    }

    @Override
    public EffectInstructions effects() {
        return effects;
    }

    @Override
    public SpecialInstructions special() {
        return special;
    }

    @Override
    public PonderScene getScene() {
        return scene;
    }

    // General

    @Override
    public void title(String sceneId, String title) {
        scene.sceneId = class_2960.method_60655(scene.getNamespace(), sceneId);
        scene.localization.registerSpecific(scene.sceneId, PonderScene.TITLE_KEY, title);
    }

    @Override
    public void configureBasePlate(int xOffset, int zOffset, int basePlateSize) {
        scene.basePlateOffsetX = xOffset;
        scene.basePlateOffsetZ = zOffset;
        scene.basePlateSize = basePlateSize;
    }

    @Override
    public void scaleSceneView(float factor) {
        scene.scaleFactor = factor;
    }

    @Override
    public void removeShadow() {
        scene.hidePlatformShadow = true;
    }

    @Override
    public void setSceneOffsetY(float yOffset) {
        scene.yOffset = yOffset;
    }

    @Override
    public void showBasePlate() {
        world.showSection(
            scene.getSceneBuildingUtil().select().cuboid(
                new class_2338(scene.getBasePlateOffsetX(), 0, scene.getBasePlateOffsetZ()),
                new class_2382(scene.getBasePlateSize() - 1, 0, scene.getBasePlateSize() - 1)
            ), class_2350.field_11036
        );
    }

    @Override
    public void addInstruction(PonderInstruction instruction) {
        scene.schedule.add(instruction);
    }

    @Override
    public void addInstruction(Consumer<PonderScene> callback) {
        addInstruction(PonderInstruction.simple(callback));
    }

    @Override
    public void idle(int ticks) {
        addInstruction(new DelayInstruction(ticks));
    }

    @Override
    public void idleSeconds(int seconds) {
        idle(seconds * 20);
    }

    @Override
    public void markAsFinished() {
        addInstruction(new MarkAsFinishedInstruction());
    }

    @Override
    public void setNextUpEnabled(boolean isEnabled) {
        addInstruction(scene -> scene.setNextUpEnabled(isEnabled));
    }

    @Override
    public void rotateCameraY(float degrees) {
        addInstruction(new RotateSceneInstruction(0, degrees, true));
    }

    @Override
    public void addKeyframe() {
        addInstruction(KeyframeInstruction.IMMEDIATE);
    }

    @Override
    public void addLazyKeyframe() {
        addInstruction(KeyframeInstruction.DELAYED);
    }

    public class PonderEffectInstructions implements EffectInstructions {

        @Override
        public void emitParticles(class_243 location, ParticleEmitter emitter, float amountPerCycle, int cycles) {
            addInstruction(new EmitParticlesInstruction(location, emitter, amountPerCycle, cycles));
        }

        @Override
        public <T extends class_2394> ParticleEmitter simpleParticleEmitter(T data, class_243 motion) {
            return (w, x, y, z) -> w.method_8406(data, x, y, z, motion.field_1352, motion.field_1351, motion.field_1350);
        }

        @Override
        public <T extends class_2394> ParticleEmitter particleEmitterWithinBlockSpace(T data, class_243 motion) {
            return (w, x, y, z) -> w.method_8406(
                data,
                Math.floor(x) + w.field_9229.method_43057(),
                Math.floor(y) + w.field_9229.method_43057(),
                Math.floor(z) + w.field_9229.method_43057(),
                motion.field_1352,
                motion.field_1351,
                motion.field_1350
            );
        }

        @Override
        public void indicateRedstone(class_2338 pos) {
            createRedstoneParticles(pos, 0xFF0000, 10);
        }

        @Override
        public void indicateSuccess(class_2338 pos) {
            createRedstoneParticles(pos, 0x80FFaa, 10);
        }

        @Override
        public void createRedstoneParticles(class_2338 pos, int color, int amount) {
            int rgb = new Color(color).getRGB();
            addInstruction(new EmitParticlesInstruction(
                VecHelper.getCenterOf(pos),
                effects().particleEmitterWithinBlockSpace(new class_2390(rgb, 1), class_243.field_1353),
                amount,
                2
            ));
        }

    }

    public class PonderOverlayInstructions implements OverlayInstructions {

        @Override
        public TextElementBuilder showText(int duration) {
            TextWindowElement textWindowElement = new TextWindowElement();
            addInstruction(new TextInstruction(textWindowElement, duration));
            return textWindowElement.builder(scene);
        }

        @Override
        public TextElementBuilder showOutlineWithText(Selection selection, int duration) {
            TextWindowElement textWindowElement = new TextWindowElement();
            addInstruction(new TextInstruction(textWindowElement, duration, selection));
            return textWindowElement.builder(scene).pointAt(selection.getCenter());
        }

        @Override
        public InputElementBuilder showControls(class_243 sceneSpace, Pointing direction, int duration) {
            InputWindowElement inputWindowElement = new InputWindowElement(sceneSpace, direction);
            addInstruction(new ShowInputInstruction(inputWindowElement, duration));
            return inputWindowElement.builder();
        }

        @Override
        public void chaseBoundingBoxOutline(PonderPalette color, Object slot, class_238 boundingBox, int duration) {
            addInstruction(new ChaseAABBInstruction(color, slot, boundingBox, duration));
        }

        @Override
        public void showCenteredScrollInput(class_2338 pos, class_2350 side, int duration) {
            showScrollInput(scene.getSceneBuildingUtil().vector().blockSurface(pos, side), side, duration);
        }

        @Override
        public void showScrollInput(class_243 location, class_2350 side, int duration) {
            class_2351 axis = side.method_10166();
            float s = 1 / 16f;
            float q = 1 / 4f;
            class_243 expands = new class_243(axis == class_2351.field_11048 ? s : q, axis == class_2351.field_11052 ? s : q, axis == class_2351.field_11051 ? s : q);
            addInstruction(new HighlightValueBoxInstruction(location, expands, duration));
        }

        @Override
        public void showRepeaterScrollInput(class_2338 pos, int duration) {
            float s = 1 / 16f;
            float q = 1 / 6f;
            class_243 expands = new class_243(q, s, q);
            addInstruction(new HighlightValueBoxInstruction(
                scene.getSceneBuildingUtil().vector().blockSurface(pos, class_2350.field_11033)
                    .method_1031(0, 3 / 16f, 0), expands, duration
            ));
        }

        @Override
        public void showFilterSlotInput(class_243 location, int duration) {
            float s = .1f;
            class_243 expands = new class_243(s, s, s);
            addInstruction(new HighlightValueBoxInstruction(location, expands, duration));
        }

        @Override
        public void showFilterSlotInput(class_243 location, class_2350 side, int duration) {
            location = location.method_1019(class_243.method_24954(side.method_62675()).method_1021(-3 / 128f));
            class_243 expands = VecHelper.axisAlingedPlaneOf(side).method_1021(11 / 128f);
            addInstruction(new HighlightValueBoxInstruction(location, expands, duration));
        }

        @Override
        public void showLine(PonderPalette color, class_243 start, class_243 end, int duration) {
            addInstruction(new LineInstruction(color, start, end, duration, false));
        }

        @Override
        public void showBigLine(PonderPalette color, class_243 start, class_243 end, int duration) {
            addInstruction(new LineInstruction(color, start, end, duration, true));
        }

        @Override
        public void showOutline(PonderPalette color, Object slot, Selection selection, int duration) {
            addInstruction(new OutlineSelectionInstruction(color, slot, selection, duration));
        }

    }

    public class PonderSpecialInstructions implements SpecialInstructions {

        @Override
        public ElementLink<ParrotElement> createBirb(class_243 location, Supplier<? extends ParrotPose> pose) {
            ElementLink<ParrotElement> link = new ElementLinkImpl<>(ParrotElement.class);
            ParrotElement parrot = ParrotElementImpl.create(location, pose);
            addInstruction(new CreateParrotInstruction(10, class_2350.field_11033, parrot));
            addInstruction(scene -> scene.linkElement(parrot, link));
            return link;
        }

        @Override
        public void changeBirbPose(ElementLink<ParrotElement> birb, Supplier<? extends ParrotPose> pose) {
            addInstruction(scene -> scene.resolveOptional(birb).ifPresent(safeBirb -> safeBirb.setPose(pose.get())));
        }

        @Override
        public void movePointOfInterest(class_243 location) {
            addInstruction(new MovePoiInstruction(location));
        }

        @Override
        public void movePointOfInterest(class_2338 location) {
            movePointOfInterest(VecHelper.getCenterOf(location));
        }

        @Override
        public void rotateParrot(ElementLink<ParrotElement> link, double xRotation, double yRotation, double zRotation, int duration) {
            addInstruction(AnimateParrotInstruction.rotate(link, new class_243(xRotation, yRotation, zRotation), duration));
        }

        @Override
        public void moveParrot(ElementLink<ParrotElement> link, class_243 offset, int duration) {
            addInstruction(AnimateParrotInstruction.move(link, offset, duration));
        }

        @Override
        public ElementLink<MinecartElement> createCart(class_243 location, float angle, MinecartConstructor type) {
            ElementLink<MinecartElement> link = new ElementLinkImpl<>(MinecartElement.class);
            MinecartElement cart = new MinecartElementImpl(location, angle, type);
            addInstruction(new CreateMinecartInstruction(10, class_2350.field_11033, cart));
            addInstruction(scene -> scene.linkElement(cart, link));
            return link;
        }

        @Override
        public void rotateCart(ElementLink<MinecartElement> link, float yRotation, int duration) {
            addInstruction(AnimateMinecartInstruction.rotate(link, yRotation, duration));
        }

        @Override
        public void moveCart(ElementLink<MinecartElement> link, class_243 offset, int duration) {
            addInstruction(AnimateMinecartInstruction.move(link, offset, duration));
        }

        @Override
        public <T extends AnimatedSceneElement> void hideElement(ElementLink<T> link, class_2350 direction) {
            addInstruction(new FadeOutOfSceneInstruction<>(15, direction, link));
        }

    }

    public class PonderWorldInstructions implements WorldInstructions {
        @Override
        public class_7225.class_7874 getHolderLookupProvider() {
            return scene.getWorld().method_30349();
        }

        @Override
        public void incrementBlockBreakingProgress(class_2338 pos) {
            addInstruction(scene -> {
                PonderLevel world = scene.getWorld();
                int progress = world.getBlockBreakingProgressions().getOrDefault(pos, -1) + 1;
                if (progress == 9) {
                    world.addBlockDestroyEffects(pos, world.method_8320(pos));
                    world.method_22352(pos, false);
                    world.setBlockBreakingProgress(pos, 0);
                    scene.forEach(WorldSectionElement.class, WorldSectionElement::queueRedraw);
                } else
                    world.setBlockBreakingProgress(pos, progress + 1);
            });
        }

        @Override
        public void showSection(Selection selection, class_2350 fadeInDirection) {
            addInstruction(new DisplayWorldSectionInstruction(15, fadeInDirection, selection, scene::getBaseWorldSection));
        }

        @Override
        public void showSectionAndMerge(Selection selection, class_2350 fadeInDirection, ElementLink<WorldSectionElement> link) {
            addInstruction(new DisplayWorldSectionInstruction(15, fadeInDirection, selection, () -> scene.resolve(link)));
        }

        @Override
        public void glueBlockOnto(class_2338 position, class_2350 fadeInDirection, ElementLink<WorldSectionElement> link) {
            addInstruction(new DisplayWorldSectionInstruction(
                15,
                fadeInDirection,
                scene.getSceneBuildingUtil().select().position(position),
                () -> scene.resolve(link),
                position
            ));
        }

        @Override
        public ElementLink<WorldSectionElement> showIndependentSection(Selection selection, class_2350 fadeInDirection) {
            DisplayWorldSectionInstruction instruction = new DisplayWorldSectionInstruction(15, fadeInDirection, selection, null);
            addInstruction(instruction);
            return instruction.createLink(scene);
        }

        @Override
        public ElementLink<WorldSectionElement> showIndependentSectionImmediately(Selection selection) {
            DisplayWorldSectionInstruction instruction = new DisplayWorldSectionInstruction(0, class_2350.field_11033, selection, null);
            addInstruction(instruction);
            return instruction.createLink(scene);
        }

        @Override
        public void hideSection(Selection selection, class_2350 fadeOutDirection) {
            WorldSectionElement worldSectionElement = new WorldSectionElementImpl(selection);
            ElementLink<WorldSectionElement> elementLink = new ElementLinkImpl<>(WorldSectionElement.class);

            addInstruction(scene -> {
                scene.getBaseWorldSection().erase(selection);
                scene.linkElement(worldSectionElement, elementLink);
                scene.addElement(worldSectionElement);
                worldSectionElement.queueRedraw();
            });

            hideIndependentSection(elementLink, fadeOutDirection);
        }

        @Override
        public void hideIndependentSection(ElementLink<WorldSectionElement> link, class_2350 fadeOutDirection) {
            addInstruction(new FadeOutOfSceneInstruction<>(15, fadeOutDirection, link));
        }

        @Override
        public void restoreBlocks(Selection selection) {
            addInstruction(scene -> scene.getWorld().restoreBlocks(selection));
        }

        @Override
        public ElementLink<WorldSectionElement> makeSectionIndependent(Selection selection) {
            WorldSectionElementImpl worldSectionElement = new WorldSectionElementImpl(selection);
            ElementLink<WorldSectionElement> elementLink = new ElementLinkImpl<>(WorldSectionElement.class);

            addInstruction(scene -> {
                scene.getBaseWorldSection().erase(selection);
                scene.linkElement(worldSectionElement, elementLink);
                scene.addElement(worldSectionElement);
                worldSectionElement.queueRedraw();
                worldSectionElement.resetAnimatedTransform();
                worldSectionElement.setVisible(true);
                worldSectionElement.forceApplyFade(1);
            });

            return elementLink;
        }

        @Override
        public void rotateSection(ElementLink<WorldSectionElement> link, double xRotation, double yRotation, double zRotation, int duration) {
            addInstruction(AnimateWorldSectionInstruction.rotate(link, new class_243(xRotation, yRotation, zRotation), duration));
        }

        @Override
        public void configureCenterOfRotation(ElementLink<WorldSectionElement> link, class_243 anchor) {
            addInstruction(scene -> scene.resolveOptional(link).ifPresent(safe -> safe.setCenterOfRotation(anchor)));
        }

        @Override
        public void configureStabilization(ElementLink<WorldSectionElement> link, class_243 anchor) {
            addInstruction(scene -> scene.resolveOptional(link).ifPresent(safe -> safe.stabilizeRotation(anchor)));
        }

        @Override
        public void moveSection(ElementLink<WorldSectionElement> link, class_243 offset, int duration) {
            addInstruction(AnimateWorldSectionInstruction.move(link, offset, duration));
        }

        @Override
        public void setBlocks(Selection selection, class_2680 state, boolean spawnParticles) {
            addInstruction(new ReplaceBlocksInstruction(selection, $ -> state, true, spawnParticles));
        }

        @Override
        public void destroyBlock(class_2338 pos) {
            setBlock(pos, class_2246.field_10124.method_9564(), true);
        }

        @Override
        public void setBlock(class_2338 pos, class_2680 state, boolean spawnParticles) {
            setBlocks(scene.getSceneBuildingUtil().select().position(pos), state, spawnParticles);
        }

        @Override
        public void replaceBlocks(Selection selection, class_2680 state, boolean spawnParticles) {
            modifyBlocks(selection, $ -> state, spawnParticles);
        }

        @Override
        public void modifyBlock(class_2338 pos, UnaryOperator<class_2680> stateFunc, boolean spawnParticles) {
            modifyBlocks(scene.getSceneBuildingUtil().select().position(pos), stateFunc, spawnParticles);
        }

        @Override
        public void cycleBlockProperty(class_2338 pos, class_2769<?> property) {
            modifyBlocks(scene.getSceneBuildingUtil().select().position(pos), s -> s.method_28498(property) ? s.method_28493(property) : s, false);
        }

        @Override
        public void modifyBlocks(Selection selection, UnaryOperator<class_2680> stateFunc, boolean spawnParticles) {
            addInstruction(new ReplaceBlocksInstruction(selection, stateFunc, false, spawnParticles));
        }

        @Override
        public void toggleRedstonePower(Selection selection) {
            modifyBlocks(
                selection, s -> {
                    if (s.method_28498(class_2741.field_12511))
                        s = s.method_11657(class_2741.field_12511, s.method_11654(class_2741.field_12511) == 0 ? 15 : 0);
                    if (s.method_28498(class_2741.field_12484))
                        s = s.method_28493(class_2741.field_12484);
                    if (s.method_28498(class_2459.field_11446))
                        s = s.method_28493(class_2459.field_11446);
                    return s;
                }, false
            );
        }

        @Override
        public <T extends class_1297> void modifyEntities(Class<T> entityClass, Consumer<T> entityCallBack) {
            addInstruction(scene -> scene.forEachWorldEntity(entityClass, entityCallBack));
        }

        @Override
        public <T extends class_1297> void modifyEntitiesInside(Class<T> entityClass, Selection area, Consumer<T> entityCallBack) {
            addInstruction(scene -> scene.forEachWorldEntity(
                entityClass, e -> {
                    if (area.test(e.method_24515()))
                        entityCallBack.accept(e);
                }
            ));
        }

        @Override
        public void modifyEntity(ElementLink<EntityElement> link, Consumer<class_1297> entityCallBack) {
            addInstruction(scene -> {
                EntityElement resolve = scene.resolve(link);
                if (resolve != null)
                    resolve.ifPresent(entityCallBack);
            });
        }

        @Override
        public ElementLink<EntityElement> createEntity(Function<class_1937, class_1297> factory) {
            ElementLink<EntityElement> link = new ElementLinkImpl<>(EntityElement.class, UUID.randomUUID());
            addInstruction(scene -> {
                PonderLevel world = scene.getWorld();
                class_1297 entity = factory.apply(world);
                EntityElement handle = new EntityElementImpl(entity);
                scene.addElement(handle);
                scene.linkElement(handle, link);
                world.method_8649(entity);
            });
            return link;
        }

        @Override
        public ElementLink<EntityElement> createItemEntity(class_243 location, class_243 motion, class_1799 stack) {
            return createEntity(world -> {
                class_1542 itemEntity = new class_1542(world, location.field_1352, location.field_1351, location.field_1350, stack);
                itemEntity.method_18799(motion);
                return itemEntity;
            });
        }

        @Override
        public void modifyBlockEntityNBT(Selection selection, Class<? extends class_2586> beType, Consumer<class_2487> consumer) {
            modifyBlockEntityNBT(selection, beType, consumer, false);
        }

        @Override
        public <T extends class_2586> void modifyBlockEntity(class_2338 position, Class<T> beType, Consumer<T> consumer) {
            addInstruction(scene -> {
                class_2586 blockEntity = scene.getWorld().method_8321(position);
                if (beType.isInstance(blockEntity))
                    consumer.accept(beType.cast(blockEntity));
            });
        }

        @Override
        public void modifyBlockEntityNBT(
            Selection selection,
            Class<? extends class_2586> teType,
            Consumer<class_2487> consumer,
            boolean reDrawBlocks
        ) {
            addInstruction(new BlockEntityDataInstruction(
                selection, teType, nbt -> {
                consumer.accept(nbt);
                return nbt;
            }, reDrawBlocks
            ));
        }
    }

    public class PonderDebugInstructions implements DebugInstructions {

        @Override
        public void debugSchematic() {
            addInstruction(scene -> scene.addElement(new WorldSectionElementImpl(scene.getSceneBuildingUtil().select().everywhere())));
        }

        @Override
        public void addInstructionInstance(PonderInstruction instruction) {
            addInstruction(instruction);
        }

        @Override
        public void enqueueCallback(Consumer<PonderScene> callback) {
            addInstruction(callback);
        }

    }

}