/*
 * Copyright (c) Forge Development LLC and contributors
 * SPDX-License-Identifier: LGPL-2.1-only
 */

package com.zurrtum.create.client.model.obj;

import com.google.common.collect.Maps;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.zurrtum.create.client.model.StandardModelParameters;
import com.zurrtum.create.client.model.UnbakedModelLoader;

import java.io.FileNotFoundException;
import java.util.Map;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3298;
import net.minecraft.class_3300;
import net.minecraft.class_3518;
import net.minecraft.class_4013;

/**
 * A loader for {@link ObjModel OBJ models}.
 * <p>
 * Allows the user to enable automatic face culling, toggle quad shading, flip UVs, render emissively and specify a
 * {@link ObjMaterialLibrary material library} override.
 */
public class ObjLoader implements UnbakedModelLoader<ObjModel>, class_4013 {
    public static ObjLoader INSTANCE = new ObjLoader();

    private final Map<ObjGeometry.Settings, ObjGeometry> geometryCache = Maps.newConcurrentMap();
    private final Map<class_2960, ObjMaterialLibrary> materialCache = Maps.newConcurrentMap();

    private final class_3300 manager = class_310.method_1551().method_1478();

    @Override
    public void method_14491(class_3300 resourceManager) {
        geometryCache.clear();
        materialCache.clear();
    }

    @Override
    public ObjModel read(JsonObject jsonObject, JsonDeserializationContext jsonDeserializationContext) throws JsonParseException {
        if (!jsonObject.has("model"))
            throw new JsonParseException("OBJ Loader requires a 'model' key that points to a valid .OBJ model.");

        String modelLocation = jsonObject.get("model").getAsString();

        boolean automaticCulling = class_3518.method_15258(jsonObject, "automatic_culling", true);
        boolean shadeQuads = class_3518.method_15258(jsonObject, "shade_quads", true);
        boolean flipV = class_3518.method_15258(jsonObject, "flip_v", false);
        boolean emissiveAmbient = class_3518.method_15258(jsonObject, "emissive_ambient", true);
        String mtlOverride = class_3518.method_15253(jsonObject, "mtl_override", null);
        StandardModelParameters parameters = StandardModelParameters.parse(jsonObject, jsonDeserializationContext);

        var geometry = loadGeometry(new ObjGeometry.Settings(
            class_2960.method_60654(modelLocation),
            automaticCulling,
            shadeQuads,
            flipV,
            emissiveAmbient,
            mtlOverride,
            parameters
        ));
        return new ObjModel(parameters, geometry);
    }

    public ObjGeometry loadGeometry(ObjGeometry.Settings settings) {
        return geometryCache.computeIfAbsent(
            settings, (data) -> {
                class_3298 resource = manager.method_14486(settings.modelLocation()).orElseThrow();
                try (ObjTokenizer tokenizer = new ObjTokenizer(resource.method_14482())) {
                    return ObjGeometry.parse(tokenizer, data);
                } catch (FileNotFoundException e) {
                    throw new RuntimeException("Could not find OBJ model", e);
                } catch (Exception e) {
                    throw new RuntimeException("Could not read OBJ model", e);
                }
            }
        );
    }

    public ObjMaterialLibrary loadMaterialLibrary(class_2960 materialLocation) {
        return materialCache.computeIfAbsent(
            materialLocation, (location) -> {
                class_3298 resource = manager.method_14486(location).orElseThrow();
                try (ObjTokenizer rdr = new ObjTokenizer(resource.method_14482())) {
                    return new ObjMaterialLibrary(rdr);
                } catch (FileNotFoundException e) {
                    throw new RuntimeException("Could not find OBJ material library", e);
                } catch (Exception e) {
                    throw new RuntimeException("Could not read OBJ material library", e);
                }
            }
        );
    }
}
