package com.zurrtum.create.content.trains.track;

import com.zurrtum.create.Create;
import com.zurrtum.create.catnip.data.Iterate;
import com.zurrtum.create.content.contraptions.StructureTransform;
import com.zurrtum.create.content.trains.graph.*;
import com.zurrtum.create.content.trains.signal.SingleBlockEntityEdgePoint;
import com.zurrtum.create.content.trains.signal.TrackEdgePoint;
import com.zurrtum.create.foundation.blockEntity.SmartBlockEntity;
import com.zurrtum.create.foundation.blockEntity.behaviour.BehaviourType;
import com.zurrtum.create.foundation.blockEntity.behaviour.BlockEntityBehaviour;
import com.zurrtum.create.infrastructure.component.BezierTrackPointLocation;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.UUID;
import net.minecraft.class_11352;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350.class_2351;
import net.minecraft.class_2350.class_2352;
import net.minecraft.class_243;
import net.minecraft.class_2487;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_3532;
import net.minecraft.class_4844;
import net.minecraft.class_8942;

public class TrackTargetingBehaviour<T extends TrackEdgePoint> extends BlockEntityBehaviour<SmartBlockEntity> {

    public static final BehaviourType<TrackTargetingBehaviour<?>> TYPE = new BehaviourType<>();

    private class_2338 targetTrack;
    private BezierTrackPointLocation targetBezier;
    private class_2352 targetDirection;
    private UUID id;

    private class_243 prevDirection;
    private class_243 rotatedDirection;

    private class_2487 migrationData;
    private EdgePointType<T> edgePointType;
    private T edgePoint;
    private boolean orthogonal;

    public TrackTargetingBehaviour(SmartBlockEntity be, EdgePointType<T> edgePointType) {
        super(be);
        this.edgePointType = edgePointType;
        targetDirection = class_2352.field_11056;
        targetTrack = class_2338.field_10980;
        id = UUID.randomUUID();
        migrationData = null;
        orthogonal = false;
    }

    @Override
    public boolean isSafeNBT() {
        return true;
    }

    @Override
    public void write(class_11372 view, boolean clientPacket) {
        view.method_71468("Id", class_4844.field_25122, id);
        view.method_71468("TargetTrack", class_2338.field_25064, targetTrack);
        view.method_71472("Ortho", orthogonal);
        view.method_71472("TargetDirection", targetDirection == class_2352.field_11056);
        if (rotatedDirection != null)
            view.method_71468("RotatedAxis", class_243.field_38277, rotatedDirection);
        if (prevDirection != null)
            view.method_71468("PrevAxis", class_243.field_38277, prevDirection);
        if (migrationData != null && !clientPacket)
            view.method_71468("Migrate", class_2487.field_25128, migrationData);
        if (targetBezier != null) {
            class_11372 bezier = view.method_71461("Bezier");
            bezier.method_71465("Segment", targetBezier.segment());
            bezier.method_71468("Key", class_2338.field_25064, targetBezier.curveTarget().method_10059(getPos()));
        }
        super.write(view, clientPacket);
    }

    @Override
    public void read(class_11368 view, boolean clientPacket) {
        id = view.method_71426("Id", class_4844.field_25122).orElseGet(UUID::randomUUID);
        targetTrack = view.method_71426("TargetTrack", class_2338.field_25064).orElse(class_2338.field_10980);
        targetDirection = view.method_71433("TargetDirection", false) ? class_2352.field_11056 : class_2352.field_11060;
        orthogonal = view.method_71433("Ortho", false);
        view.method_71426("RotatedAxis", class_243.field_38277).ifPresent(rotated -> rotatedDirection = rotated);
        view.method_71426("PrevAxis", class_243.field_38277).ifPresent(prev -> prevDirection = prev);
        view.method_71426("Migrate", class_2487.field_25128).ifPresent(migration -> migrationData = migration);
        if (clientPacket)
            edgePoint = null;
        view.method_71420("Bezier").ifPresent(bezier -> {
            class_2338 key = bezier.method_71426("Key", class_2338.field_25064).orElse(class_2338.field_10980);
            targetBezier = new BezierTrackPointLocation(key.method_10081(getPos()), bezier.method_71424("Segment", 0));
        });
        super.read(view, clientPacket);
    }

    @Nullable
    public T getEdgePoint() {
        return edgePoint;
    }

    public void invalidateEdgePoint(class_2487 migrationData) {
        this.migrationData = migrationData;
        edgePoint = null;
        blockEntity.sendData();
    }

    @Override
    public void tick() {
        super.tick();
        if (edgePoint == null)
            edgePoint = createEdgePoint();
    }

    @SuppressWarnings("unchecked")
    public T createEdgePoint() {
        class_1937 level = getLevel();
        boolean isClientSide = level.method_8608();
        if (migrationData == null || isClientSide)
            for (TrackGraph trackGraph : Create.RAILWAYS.sided(level).trackNetworks.values()) {
                T point = trackGraph.getPoint(edgePointType, id);
                if (point == null)
                    continue;
                return point;
            }

        if (isClientSide)
            return null;
        if (!hasValidTrack())
            return null;
        TrackGraphLocation loc = determineGraphLocation();
        if (loc == null)
            return null;

        TrackGraph graph = loc.graph;
        TrackNode node1 = graph.locateNode(loc.edge.getFirst());
        TrackNode node2 = graph.locateNode(loc.edge.getSecond());
        TrackEdge edge = graph.getConnectionsFrom(node1).get(node2);
        if (edge == null)
            return null;

        T point = edgePointType.create();
        boolean front = getTargetDirection() == class_2352.field_11056;

        prevDirection = edge.getDirectionAt(loc.position).method_1021(front ? -1 : 1);

        if (rotatedDirection != null) {
            double dot = prevDirection.method_1026(rotatedDirection);
            if (dot < -.85f) {
                rotatedDirection = null;
                targetDirection = targetDirection.method_26424();
                return null;
            }

            rotatedDirection = null;
        }

        double length = edge.getLength();
        class_2487 data = migrationData;
        migrationData = null;

        {
            orthogonal = targetBezier == null;
            class_243 direction = edge.getDirection(true);
            int nonZeroComponents = 0;
            for (class_2351 axis : Iterate.axes)
                nonZeroComponents += direction.method_18043(axis) != 0 ? 1 : 0;
            orthogonal &= nonZeroComponents <= 1;
        }

        EdgeData signalData = edge.getEdgeData();
        if (signalData.hasPoints()) {
            for (EdgePointType<?> otherType : EdgePointType.TYPES.values()) {
                TrackEdgePoint otherPoint = signalData.get(otherType, loc.position);
                if (otherPoint == null)
                    continue;
                if (otherType != edgePointType) {
                    if (!otherPoint.canCoexistWith(edgePointType, front))
                        return null;
                    continue;
                }
                if (!otherPoint.canMerge())
                    return null;
                otherPoint.blockEntityAdded(blockEntity, front);
                id = otherPoint.getId();
                blockEntity.notifyUpdate();
                return (T) otherPoint;
            }
        }

        if (data != null) {
            try (class_8942.class_11340 logging = new class_8942.class_11340(blockEntity.method_71402(), Create.LOGGER)) {
                class_11368 view = class_11352.method_71417(logging, level.method_30349(), data);
                DimensionPalette dimensions = view.method_71426("DimensionPalette", DimensionPalette.CODEC).orElseThrow();
                point.read(view, true, dimensions);
            }
        }

        point.setId(id);
        boolean reverseEdge = front || point instanceof SingleBlockEntityEdgePoint;
        point.setLocation(reverseEdge ? loc.edge : loc.edge.swap(), reverseEdge ? loc.position : length - loc.position);
        point.blockEntityAdded(blockEntity, front);
        loc.graph.addPoint(level.method_8503(), edgePointType, point);
        blockEntity.sendData();
        return point;
    }

    @Override
    public void destroy() {
        super.destroy();
        if (edgePoint != null) {
            class_1937 world = getLevel();
            if (!world.method_8608()) {
                edgePoint.blockEntityRemoved(world.method_8503(), getPos(), getTargetDirection() == class_2352.field_11056);
            }
        }
    }

    @Override
    public BehaviourType<?> getType() {
        return TYPE;
    }

    public boolean isOnCurve() {
        return targetBezier != null;
    }

    public boolean isOrthogonal() {
        return orthogonal;
    }

    public boolean hasValidTrack() {
        return getTrackBlockState().method_26204() instanceof ITrackBlock;
    }

    public ITrackBlock getTrack() {
        return (ITrackBlock) getTrackBlockState().method_26204();
    }

    public class_2680 getTrackBlockState() {
        return getLevel().method_8320(getGlobalPosition());
    }

    public class_2338 getGlobalPosition() {
        return targetTrack.method_10081(blockEntity.method_11016());
    }

    public class_2338 getPositionForMapMarker() {
        class_2338 target = targetTrack.method_10081(blockEntity.method_11016());
        if (targetBezier != null && getLevel().method_8321(target) instanceof TrackBlockEntity tbe) {
            BezierConnection bc = tbe.getConnections().get(targetBezier.curveTarget());
            if (bc == null)
                return target;
            double length = class_3532.method_15357(bc.getLength() * 2);
            int seg = targetBezier.segment() + 1;
            double t = seg / length;
            return class_2338.method_49638(bc.getPosition(t));
        }
        return target;
    }

    public class_2352 getTargetDirection() {
        return targetDirection;
    }

    public BezierTrackPointLocation getTargetBezier() {
        return targetBezier;
    }

    public TrackGraphLocation determineGraphLocation() {
        class_1937 level = getLevel();
        class_2338 pos = getGlobalPosition();
        class_2680 state = getTrackBlockState();
        ITrackBlock track = getTrack();
        List<class_243> trackAxes = track.getTrackAxes(level, pos, state);
        class_2352 targetDirection = getTargetDirection();

        return targetBezier != null ? TrackGraphHelper.getBezierGraphLocationAt(
            level,
            pos,
            targetDirection,
            targetBezier
        ) : TrackGraphHelper.getGraphLocationAt(level, pos, targetDirection, trackAxes.getFirst());
    }

    public enum RenderedTrackOverlayType {
        STATION,
        SIGNAL,
        DUAL_SIGNAL,
        OBSERVER;
    }

    public void transform(class_2586 be, StructureTransform transform) {
        id = UUID.randomUUID();
        targetTrack = transform.applyWithoutOffset(targetTrack);
        if (prevDirection != null)
            rotatedDirection = transform.applyWithoutOffsetUncentered(prevDirection);
        if (targetBezier != null)
            targetBezier = new BezierTrackPointLocation(
                transform.applyWithoutOffset(targetBezier.curveTarget().method_10059(getPos())).method_10081(getPos()),
                targetBezier.segment()
            );
        blockEntity.notifyUpdate();
    }

}
