package com.zurrtum.create.content.schematics;

import com.mojang.serialization.Codec;
import com.zurrtum.create.AllBlocks;
import com.zurrtum.create.AllDataComponents;
import com.zurrtum.create.Create;
import com.zurrtum.create.api.contraption.BlockMovementChecks;
import com.zurrtum.create.catnip.levelWrappers.SchematicLevel;
import com.zurrtum.create.catnip.math.BBHelper;
import com.zurrtum.create.content.contraptions.StructureTransform;
import com.zurrtum.create.content.schematics.cannon.MaterialChecklist;
import com.zurrtum.create.content.schematics.requirement.ItemRequirement;
import com.zurrtum.create.foundation.blockEntity.IMergeableBE;
import com.zurrtum.create.foundation.codec.CreateCodecs;
import com.zurrtum.create.foundation.utility.BlockHelper;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import net.minecraft.class_11352;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1297;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2343;
import net.minecraft.class_2350;
import net.minecraft.class_2487;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_2741;
import net.minecraft.class_2742;
import net.minecraft.class_2756;
import net.minecraft.class_3341;
import net.minecraft.class_3492;
import net.minecraft.class_3499;
import net.minecraft.class_3542;
import net.minecraft.class_3612;
import net.minecraft.class_8942;

public class SchematicPrinter {

    public enum PrintStage implements class_3542 {
        BLOCKS,
        DEFERRED_BLOCKS,
        ENTITIES;

        public static final Codec<PrintStage> CODEC = class_3542.method_28140(PrintStage::values);

        @Override
        public String method_15434() {
            return name().toLowerCase(Locale.ROOT);
        }
    }

    private boolean schematicLoaded;
    private boolean isErrored;
    private SchematicLevel blockReader;
    private class_2338 schematicAnchor;

    private class_2338 currentPos;
    private int printingEntityIndex;
    private PrintStage printStage;
    private List<class_2338> deferredBlocks;

    public SchematicPrinter() {
        printingEntityIndex = -1;
        printStage = PrintStage.BLOCKS;
        deferredBlocks = new LinkedList<>();
    }

    public void read(class_11368 view, boolean clientPacket) {
        view.method_71426("CurrentPos", class_2338.field_25064).ifPresent(pos -> currentPos = pos);
        if (clientPacket) {
            schematicLoaded = false;
            view.method_71426("Anchor", class_2338.field_25064).ifPresent(pos -> {
                schematicAnchor = pos;
                schematicLoaded = true;
            });
        }

        printingEntityIndex = view.method_71424("EntityProgress", 0);
        printStage = view.method_71426("PrintStage", PrintStage.CODEC).orElse(PrintStage.BLOCKS);
        deferredBlocks.clear();
        view.method_71426("DeferredBlocks", CreateCodecs.BLOCK_POS_LIST_CODEC).ifPresent(deferredBlocks::addAll);
    }

    public void write(class_11372 view) {
        if (currentPos != null)
            view.method_71468("CurrentPos", class_2338.field_25064, currentPos);
        if (schematicAnchor != null)
            view.method_71468("Anchor", class_2338.field_25064, schematicAnchor);
        view.method_71465("EntityProgress", printingEntityIndex);
        view.method_71468("PrintStage", PrintStage.CODEC, printStage);
        view.method_71468("DeferredBlocks", CreateCodecs.BLOCK_POS_LIST_CODEC, deferredBlocks);
    }

    public void loadSchematic(class_1799 blueprint, class_1937 originalWorld, boolean processNBT) {
        if (!blueprint.method_57826(AllDataComponents.SCHEMATIC_ANCHOR) || !blueprint.method_57826(AllDataComponents.SCHEMATIC_DEPLOYED))
            return;

        class_3499 activeTemplate = SchematicItem.loadSchematic(originalWorld, blueprint);
        class_3492 settings = SchematicItem.getSettings(blueprint, processNBT);

        schematicAnchor = blueprint.method_58694(AllDataComponents.SCHEMATIC_ANCHOR);
        blockReader = new SchematicLevel(schematicAnchor, originalWorld);

        try {
            activeTemplate.method_15172(blockReader, schematicAnchor, schematicAnchor, settings, blockReader.method_8409(), class_2248.field_31028);
        } catch (Exception e) {
            Create.LOGGER.error("Failed to load Schematic for Printing", e);
            schematicLoaded = true;
            isErrored = true;
            return;
        }

        class_2338 extraBounds = class_3499.method_15171(settings, new class_2338(activeTemplate.method_15160()).method_10069(-1, -1, -1));
        blockReader.setBounds(BBHelper.encapsulate(blockReader.getBounds(), extraBounds));

        StructureTransform transform = new StructureTransform(settings.method_15134(), class_2350.class_2351.field_11052, settings.method_15113(), settings.method_15114());
        for (class_2586 be : blockReader.getBlockEntities())
            transform.apply(be);

        printingEntityIndex = -1;
        printStage = PrintStage.BLOCKS;
        deferredBlocks.clear();
        class_3341 bounds = blockReader.getBounds();
        currentPos = new class_2338(bounds.method_35415() - 1, bounds.method_35416(), bounds.method_35417());
        schematicLoaded = true;
    }

    public void resetSchematic() {
        schematicLoaded = false;
        schematicAnchor = null;
        isErrored = false;
        currentPos = null;
        blockReader = null;
        printingEntityIndex = -1;
        printStage = PrintStage.BLOCKS;
        deferredBlocks.clear();
    }

    public boolean isLoaded() {
        return schematicLoaded;
    }

    public boolean isErrored() {
        return isErrored;
    }

    public class_2338 getCurrentTarget() {
        if (!isLoaded() || isErrored())
            return null;
        return schematicAnchor.method_10081(currentPos);
    }

    public PrintStage getPrintStage() {
        return printStage;
    }

    public class_2338 getAnchor() {
        return schematicAnchor;
    }

    public boolean isWorldEmpty() {
        return blockReader.getAllPositions().isEmpty();
        //return blockReader.getBounds().getLength().equals(new Vector3i(0,0,0));
    }

    @FunctionalInterface
    public interface BlockTargetHandler {
        void handle(class_2338 target, class_2680 blockState, class_2586 blockEntity);
    }

    @FunctionalInterface
    public interface EntityTargetHandler {
        void handle(class_2338 target, class_1297 entity);
    }

    public void handleCurrentTarget(BlockTargetHandler blockHandler, EntityTargetHandler entityHandler) {
        class_2338 target = getCurrentTarget();

        if (printStage == PrintStage.ENTITIES) {
            class_1297 entity = blockReader.getEntityList().get(printingEntityIndex);
            entityHandler.handle(target, entity);
        } else {
            class_2680 blockState = BlockHelper.setZeroAge(blockReader.method_8320(target));
            class_2586 blockEntity = blockReader.method_8321(target);
            blockHandler.handle(target, blockState, blockEntity);
        }
    }

    @FunctionalInterface
    public interface PlacementPredicate {
        boolean shouldPlace(
            class_2338 target,
            class_2680 blockState,
            class_2586 blockEntity,
            class_2680 toReplace,
            class_2680 toReplaceOther,
            boolean isNormalCube
        );
    }

    public boolean shouldPlaceCurrent(class_1937 world) {
        return shouldPlaceCurrent(world, (a, b, c, d, e, f) -> true);
    }

    public boolean shouldPlaceCurrent(class_1937 world, PlacementPredicate predicate) {
        if (world == null)
            return false;

        if (printStage == PrintStage.ENTITIES)
            return true;

        return shouldPlaceBlock(world, predicate, getCurrentTarget());
    }

    public boolean shouldPlaceBlock(class_1937 world, PlacementPredicate predicate, class_2338 pos) {
        class_2680 state = BlockHelper.setZeroAge(blockReader.method_8320(pos));
        class_2586 blockEntity = blockReader.method_8321(pos);

        class_2680 toReplace = world.method_8320(pos);
        class_2586 toReplaceBE = world.method_8321(pos);
        class_2680 toReplaceOther = null;

        if (state.method_28498(class_2741.field_12483) && state.method_28498(class_2741.field_12481) && state.method_11654(class_2741.field_12483) == class_2742.field_12557)
            toReplaceOther = world.method_8320(pos.method_10093(state.method_11654(class_2741.field_12481)));
        if (state.method_28498(class_2741.field_12533) && state.method_11654(class_2741.field_12533) == class_2756.field_12607)
            toReplaceOther = world.method_8320(pos.method_10084());

        boolean mergeTEs = blockEntity != null && toReplaceBE instanceof IMergeableBE && toReplaceBE.method_11017().equals(blockEntity.method_11017());

        if (!world.method_8477(pos))
            return false;
        if (!world.method_8621().method_11952(pos))
            return false;
        if (toReplace == state && !mergeTEs)
            return false;
        if (toReplace.method_26214(world, pos) == -1 || (toReplaceOther != null && toReplaceOther.method_26214(world, pos) == -1))
            return false;

        boolean isNormalCube = state.method_26212(blockReader, currentPos);
        return predicate.shouldPlace(pos, state, blockEntity, toReplace, toReplaceOther, isNormalCube);
    }

    public ItemRequirement getCurrentRequirement() {
        if (printStage == PrintStage.ENTITIES)
            return ItemRequirement.of(blockReader.getEntityList().get(printingEntityIndex));

        class_2338 target = getCurrentTarget();
        class_2680 blockState = BlockHelper.setZeroAge(blockReader.method_8320(target));
        class_2586 blockEntity = null;
        if (blockState.method_31709()) {
            blockEntity = ((class_2343) blockState.method_26204()).method_10123(target, blockState);
            class_2487 data = BlockHelper.prepareBlockEntityData(blockReader, blockState, blockEntity);
            if (blockEntity != null && data != null) {
                try (class_8942.class_11340 logging = new class_8942.class_11340(blockEntity.method_71402(), Create.LOGGER)) {
                    blockEntity.method_58690(class_11352.method_71417(logging, blockReader.method_30349(), data));
                }
            }
        }
        return ItemRequirement.of(blockState, blockEntity);
    }

    public int markAllBlockRequirements(MaterialChecklist checklist, class_1937 world, PlacementPredicate predicate) {
        int blocksToPlace = 0;
        for (class_2338 pos : blockReader.getAllPositions()) {
            class_2338 relPos = pos.method_10081(schematicAnchor);
            class_2680 required = blockReader.method_8320(relPos);
            class_2586 requiredBE = blockReader.method_8321(relPos);

            if (!world.method_8477(pos.method_10081(schematicAnchor))) {
                checklist.warnBlockNotLoaded();
                continue;
            }
            if (!shouldPlaceBlock(world, predicate, relPos))
                continue;
            ItemRequirement requirement = ItemRequirement.of(required, requiredBE);
            if (requirement.isEmpty())
                continue;
            if (requirement.isInvalid())
                continue;
            checklist.require(requirement);
            blocksToPlace++;
        }
        return blocksToPlace;
    }

    public void markAllEntityRequirements(MaterialChecklist checklist) {
        for (class_1297 entity : blockReader.getEntityList()) {
            ItemRequirement requirement = ItemRequirement.of(entity);
            if (requirement.isEmpty())
                return;
            if (requirement.isInvalid())
                return;
            checklist.require(requirement);
        }
    }

    public boolean advanceCurrentPos() {
        List<class_1297> entities = blockReader.getEntityList();

        do {
            if (printStage == PrintStage.BLOCKS) {
                while (tryAdvanceCurrentPos()) {
                    deferredBlocks.add(currentPos);
                }
            }

            if (printStage == PrintStage.DEFERRED_BLOCKS) {
                if (deferredBlocks.isEmpty()) {
                    printStage = PrintStage.ENTITIES;
                } else {
                    currentPos = deferredBlocks.remove(0);
                }
            }

            if (printStage == PrintStage.ENTITIES) {
                if (printingEntityIndex + 1 < entities.size()) {
                    printingEntityIndex++;
                    currentPos = entities.get(printingEntityIndex).method_24515().method_10059(schematicAnchor);
                } else {
                    // Reached end of printing
                    return false;
                }
            }
        } while (!blockReader.getBounds().method_14662(currentPos));

        // More things available to print
        return true;
    }

    public boolean tryAdvanceCurrentPos() {
        currentPos = currentPos.method_10093(class_2350.field_11034);
        class_3341 bounds = blockReader.getBounds();
        class_2338 posInBounds = currentPos.method_10069(-bounds.method_35415(), -bounds.method_35416(), -bounds.method_35417());

        if (posInBounds.method_10263() > bounds.method_35414())
            currentPos = new class_2338(bounds.method_35415(), currentPos.method_10264(), currentPos.method_10260() + 1).method_10067();
        if (posInBounds.method_10260() > bounds.method_14663())
            currentPos = new class_2338(currentPos.method_10263(), currentPos.method_10264() + 1, bounds.method_35417()).method_10067();

        // End of blocks reached
        if (currentPos.method_10264() > bounds.method_14660()) {
            printStage = PrintStage.DEFERRED_BLOCKS;
            return false;
        }

        return shouldDeferBlock(blockReader.method_8320(getCurrentTarget()));
    }

    public static boolean shouldDeferBlock(class_2680 state) {
        return state.method_27852(AllBlocks.GANTRY_CARRIAGE) || state.method_27852(AllBlocks.MECHANICAL_ARM) || BlockMovementChecks.isBrittle(state);
    }

    public void sendBlockUpdates(class_1937 level) {
        class_3341 bounds = blockReader.getBounds();
        class_2338.method_23627(bounds.method_35410(1)).filter(pos -> !bounds.method_14662(pos))
            .filter(pos -> level.method_8477(pos.method_10081(schematicAnchor)) && level.method_8316(pos.method_10081(schematicAnchor)).method_39360(class_3612.field_15910))
            .forEach(pos -> level.method_64312(pos.method_10081(schematicAnchor), class_3612.field_15910, class_3612.field_15910.method_15789(level)));
    }

}
