package com.zurrtum.create.content.fluids;

import com.zurrtum.create.catnip.math.BlockFace;
import com.zurrtum.create.foundation.blockEntity.SmartBlockEntity;
import com.zurrtum.create.foundation.blockEntity.behaviour.BlockEntityBehaviour;
import com.zurrtum.create.foundation.fluid.FluidHelper;
import com.zurrtum.create.infrastructure.fluids.FluidInventory;
import com.zurrtum.create.infrastructure.fluids.FluidStack;
import org.jetbrains.annotations.Nullable;

import java.lang.ref.WeakReference;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.function.Supplier;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2586;
import net.minecraft.class_3218;

public abstract class FlowSource {
    BlockFace location;

    public FlowSource(BlockFace location) {
        this.location = location;
    }

    public FluidStack provideFluid(Predicate<FluidStack> extractionPredicate) {
        return Optional.ofNullable(provideHandler())
            .flatMap(tank -> tank.stream(location.getOppositeFace()).filter(stack -> !stack.isEmpty() && extractionPredicate.test(stack)).findFirst()
                .map(stack -> stack.copyWithAmount(1))).orElse(FluidStack.EMPTY);
    }

    // Layer III. PFIs need active attention to prevent them from disengaging early
    public void keepAlive() {
    }

    public abstract boolean isEndpoint();

    public void manageSource(class_1937 world, class_2586 networkBE) {
    }

    public void whileFlowPresent(class_1937 world, boolean pulling) {
    }

    public @Nullable FluidInventory provideHandler() {
        return null;
    }

    public static class FluidHandler extends FlowSource {
        @Nullable Supplier<FluidInventory> fluidHandlerCache;

        public FluidHandler(BlockFace location) {
            super(location);
        }

        public void manageSource(class_1937 world, class_2586 networkBE) {
            if (fluidHandlerCache == null) {
                class_2338 pos = location.getConnectedPos();
                class_2586 blockEntity = world.method_8321(pos);
                if (blockEntity != null) {
                    class_2350 side = location.getOppositeFace();
                    if (world instanceof class_3218 serverWorld) {
                        fluidHandlerCache = FluidHelper.getFluidInventoryCache(serverWorld, pos, side);
                    } else if (networkBE instanceof SmartBlockEntity smartBE && smartBE.isVirtual()) {
                        fluidHandlerCache = () -> FluidHelper.getFluidInventory(world, pos, null, blockEntity, side);
                    }
                }
            }
        }

        @Override
        @Nullable
        public FluidInventory provideHandler() {
            return fluidHandlerCache != null ? fluidHandlerCache.get() : null;
        }

        @Override
        public boolean isEndpoint() {
            return true;
        }
    }

    public static class OtherPipe extends FlowSource {
        WeakReference<FluidTransportBehaviour> cached;

        public OtherPipe(BlockFace location) {
            super(location);
        }

        @Override
        public void manageSource(class_1937 world, class_2586 networkBE) {
            if (cached != null && cached.get() != null && !cached.get().blockEntity.method_11015())
                return;
            cached = null;
            FluidTransportBehaviour fluidTransportBehaviour = BlockEntityBehaviour.get(
                world,
                location.getConnectedPos(),
                FluidTransportBehaviour.TYPE
            );
            if (fluidTransportBehaviour != null)
                cached = new WeakReference<>(fluidTransportBehaviour);
        }

        @Override
        public FluidStack provideFluid(Predicate<FluidStack> extractionPredicate) {
            if (cached == null || cached.get() == null)
                return FluidStack.EMPTY;
            FluidTransportBehaviour behaviour = cached.get();
            FluidStack providedOutwardFluid = behaviour.getProvidedOutwardFluid(location.getOppositeFace());
            return extractionPredicate.test(providedOutwardFluid) ? providedOutwardFluid : FluidStack.EMPTY;
        }

        @Override
        public boolean isEndpoint() {
            return false;
        }

    }

    public static class Blocked extends FlowSource {

        public Blocked(BlockFace location) {
            super(location);
        }

        @Override
        public boolean isEndpoint() {
            return false;
        }

    }

}
