package com.zurrtum.create.client.content.trains.schedule;

import com.google.common.collect.ImmutableList;
import com.zurrtum.create.AllDataComponents;
import com.zurrtum.create.AllSchedules;
import com.zurrtum.create.catnip.animation.LerpedFloat;
import com.zurrtum.create.catnip.animation.LerpedFloat.Chaser;
import com.zurrtum.create.catnip.data.IntAttached;
import com.zurrtum.create.catnip.data.Pair;
import com.zurrtum.create.client.AllKeys;
import com.zurrtum.create.client.AllScheduleRenders;
import com.zurrtum.create.client.Create;
import com.zurrtum.create.client.catnip.animation.AnimationTickHolder;
import com.zurrtum.create.client.catnip.gui.UIRenderHelper;
import com.zurrtum.create.client.catnip.gui.element.GuiGameElement;
import com.zurrtum.create.client.catnip.gui.widget.ElementWidget;
import com.zurrtum.create.client.foundation.gui.AllGuiTextures;
import com.zurrtum.create.client.foundation.gui.AllIcons;
import com.zurrtum.create.client.foundation.gui.ModularGuiLine;
import com.zurrtum.create.client.foundation.gui.ModularGuiLineBuilder;
import com.zurrtum.create.client.foundation.gui.menu.AbstractSimiContainerScreen;
import com.zurrtum.create.client.foundation.gui.widget.IconButton;
import com.zurrtum.create.client.foundation.gui.widget.Indicator;
import com.zurrtum.create.client.foundation.gui.widget.Indicator.State;
import com.zurrtum.create.client.foundation.gui.widget.Label;
import com.zurrtum.create.client.foundation.gui.widget.SelectionScrollInput;
import com.zurrtum.create.client.foundation.utility.CreateLang;
import com.zurrtum.create.content.trains.GlobalRailwayManager;
import com.zurrtum.create.content.trains.graph.EdgePointType;
import com.zurrtum.create.content.trains.graph.TrackGraph;
import com.zurrtum.create.content.trains.schedule.Schedule;
import com.zurrtum.create.content.trains.schedule.ScheduleDataEntry;
import com.zurrtum.create.content.trains.schedule.ScheduleEntry;
import com.zurrtum.create.content.trains.schedule.ScheduleMenu;
import com.zurrtum.create.content.trains.schedule.condition.ScheduleWaitCondition;
import com.zurrtum.create.content.trains.schedule.destination.DestinationInstruction;
import com.zurrtum.create.content.trains.schedule.destination.ScheduleInstruction;
import com.zurrtum.create.content.trains.station.GlobalStation;
import com.zurrtum.create.foundation.gui.menu.MenuType;
import com.zurrtum.create.infrastructure.packet.c2s.GhostItemSubmitPacket;
import com.zurrtum.create.infrastructure.packet.c2s.ScheduleEditPacket;
import net.minecraft.ChatFormatting;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.client.gui.components.EditBox;
import net.minecraft.client.gui.components.Renderable;
import net.minecraft.client.input.KeyEvent;
import net.minecraft.client.input.MouseButtonEvent;
import net.minecraft.client.renderer.Rect2i;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.network.RegistryFriendlyByteBuf;
import net.minecraft.network.chat.CommonComponents;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.MutableComponent;
import net.minecraft.resources.Identifier;
import net.minecraft.util.FormattedCharSequence;
import net.minecraft.util.Mth;
import net.minecraft.util.ProblemReporter;
import net.minecraft.world.entity.player.Inventory;
import net.minecraft.world.inventory.Slot;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.Items;
import net.minecraft.world.level.storage.TagValueInput;
import net.minecraft.world.level.storage.ValueInput;
import net.minecraft.world.phys.Vec3;
import org.jetbrains.annotations.Nullable;
import org.joml.Matrix3x2fStack;

import java.util.*;
import java.util.function.Consumer;
import java.util.function.Function;

import static com.zurrtum.create.Create.LOGGER;

public class ScheduleScreen extends AbstractSimiContainerScreen<ScheduleMenu> {
    private static final int CARD_HEADER = 22;
    private static final int CARD_WIDTH = 195;

    private List<Rect2i> extraAreas = Collections.emptyList();

    private List<LerpedFloat> horizontalScrolls = new ArrayList<>();
    private LerpedFloat scroll = LerpedFloat.linear().startWithValue(0);
    private ElementWidget renderedItem;

    private Schedule schedule;

    private IconButton confirmButton;
    private IconButton cyclicButton;
    private Indicator cyclicIndicator;

    private IconButton resetProgress;
    private IconButton skipProgress;

    private ScheduleInstruction editingDestination;
    private ScheduleWaitCondition editingCondition;
    private SelectionScrollInput scrollInput;
    private Label scrollInputLabel;
    private IconButton editorConfirm, editorDelete;
    private ModularGuiLine editorSubWidgets;
    private Consumer<Boolean> onEditorClose;

    private DestinationSuggestions destinationSuggestions;

    public ScheduleScreen(ScheduleMenu menu, Inventory inv, Component title) {
        super(menu, inv, title);
        schedule = new Schedule();
        CompoundTag tag = menu.contentHolder.get(AllDataComponents.TRAIN_SCHEDULE);
        if (tag != null && !tag.isEmpty()) {
            try (ProblemReporter.ScopedCollector logging = new ProblemReporter.ScopedCollector(() -> "ScheduleScreen", LOGGER)) {
                ValueInput view = TagValueInput.create(logging, menu.player.registryAccess(), tag);
                schedule = Schedule.read(view);
            }
        }
        menu.slotsActive = false;
        editorSubWidgets = new ModularGuiLine();
    }

    public static ScheduleScreen create(
        Minecraft mc,
        MenuType<ItemStack> type,
        int syncId,
        Inventory inventory,
        Component title,
        RegistryFriendlyByteBuf extraData
    ) {
        return type.create(ScheduleScreen::new, syncId, inventory, title, getStack(extraData));
    }

    @Override
    protected void init() {
        AllGuiTextures bg = AllGuiTextures.SCHEDULE;
        setWindowSize(bg.getWidth(), bg.getHeight());
        super.init();
        clearWidgets();

        confirmButton = new IconButton(leftPos + bg.getWidth() - 42, topPos + bg.getHeight() - 30, AllIcons.I_CONFIRM);
        confirmButton.withCallback(() -> minecraft.player.closeContainer());
        addRenderableWidget(confirmButton);

        cyclicIndicator = new Indicator(leftPos + 21, topPos + 196, CommonComponents.EMPTY);
        cyclicIndicator.state = schedule.cyclic ? State.ON : State.OFF;

        List<Component> tip = new ArrayList<>();
        tip.add(CreateLang.translateDirect("schedule.loop"));
        tip.add(CreateLang.translateDirect("gui.schematicannon.optionDisabled").withStyle(ChatFormatting.RED));
        tip.add(CreateLang.translateDirect("schedule.loop1").withStyle(ChatFormatting.GRAY));
        tip.add(CreateLang.translateDirect("schedule.loop2").withStyle(ChatFormatting.GRAY));

        List<Component> tipEnabled = new ArrayList<>(tip);
        tipEnabled.set(1, CreateLang.translateDirect("gui.schematicannon.optionEnabled").withStyle(ChatFormatting.DARK_GREEN));

        cyclicButton = new IconButton(leftPos + 21, topPos + 196, AllIcons.I_REFRESH);
        cyclicButton.withCallback(() -> {
            schedule.cyclic = !schedule.cyclic;
            cyclicButton.green = schedule.cyclic;
            cyclicButton.getToolTip().clear();
            cyclicButton.getToolTip().addAll(schedule.cyclic ? tipEnabled : tip);
        });
        cyclicButton.green = schedule.cyclic;
        cyclicButton.getToolTip().clear();
        cyclicButton.getToolTip().addAll(schedule.cyclic ? tipEnabled : tip);

        addRenderableWidget(cyclicButton);

        resetProgress = new IconButton(leftPos + 45, topPos + 196, AllIcons.I_PRIORITY_VERY_HIGH);
        resetProgress.withCallback(() -> {
            schedule.savedProgress = 0;
            resetProgress.active = false;
        });
        resetProgress.active = schedule.savedProgress > 0 && !schedule.entries.isEmpty();
        resetProgress.setToolTip(CreateLang.translateDirect("schedule.reset"));
        addRenderableWidget(resetProgress);

        skipProgress = new IconButton(leftPos + 63, topPos + 196, AllIcons.I_PRIORITY_LOW);
        skipProgress.withCallback(() -> {
            schedule.savedProgress++;
            schedule.savedProgress %= schedule.entries.size();
            resetProgress.active = schedule.savedProgress > 0;
        });
        skipProgress.active = schedule.entries.size() > 1;
        skipProgress.setToolTip(CreateLang.translateDirect("schedule.skip"));
        addRenderableWidget(skipProgress);

        stopEditing();
        extraAreas = ImmutableList.of(new Rect2i(leftPos + bg.getWidth(), topPos + bg.getHeight() - 56, 48, 48));
        horizontalScrolls.clear();
        for (int i = 0; i < schedule.entries.size(); i++)
            horizontalScrolls.add(LerpedFloat.linear().startWithValue(0));

        renderedItem = new ElementWidget(
            leftPos + AllGuiTextures.SCHEDULE.getWidth(),
            topPos + AllGuiTextures.SCHEDULE.getHeight() - 56
        ).showingElement(GuiGameElement.of(menu.contentHolder).scale(3));
        addRenderableWidget(renderedItem);
    }

    @Override
    public void onClose() {
        super.onClose();
        renderedItem.getRenderElement().clear();
    }

    public static <T> List<MutableComponent> getTypeOptions(List<Pair<Identifier, T>> list) {
        String langSection = list.equals(AllSchedules.INSTRUCTION_TYPES) ? "instruction." : "condition.";
        return list.stream().map(Pair::getFirst).map(rl -> rl.getNamespace() + ".schedule." + langSection + rl.getPath()).map(Component::translatable)
            .toList();
    }

    @SuppressWarnings("unchecked")
    protected <T extends ScheduleDataEntry> void startEditing(IScheduleInput<T> field, T input, Consumer<Boolean> onClose, boolean allowDeletion) {
        onEditorClose = onClose;
        confirmButton.visible = false;
        cyclicButton.visible = false;
        cyclicIndicator.visible = false;
        skipProgress.visible = false;
        resetProgress.visible = false;

        scrollInput = new SelectionScrollInput(leftPos + 56, topPos + 65, 143, 16);
        scrollInputLabel = new Label(leftPos + 59, topPos + 69, CommonComponents.EMPTY).withShadow();
        editorConfirm = new IconButton(leftPos + 56 + 168, topPos + 65 + 22, AllIcons.I_CONFIRM);
        if (allowDeletion)
            editorDelete = new IconButton(leftPos + 56 - 45, topPos + 65 + 22, AllIcons.I_TRASH);
        menu.slotsActive = true;
        menu.targetSlotsActive = field.slotsTargeted();

        for (int i = 0; i < field.slotsTargeted(); i++) {
            ItemStack item = field.getItem(input, i);
            menu.ghostInventory.setItem(i, item);
            minecraft.player.connection.send(new GhostItemSubmitPacket(item, i));
        }

        if (input instanceof ScheduleInstruction instruction) {
            int startIndex = 0;
            for (int i = 0; i < AllSchedules.INSTRUCTION_TYPES.size(); i++)
                if (AllSchedules.INSTRUCTION_TYPES.get(i).getFirst().equals(instruction.getId()))
                    startIndex = i;
            editingDestination = instruction;
            updateEditorSubwidgets((IScheduleInput<ScheduleInstruction>) field, editingDestination);
            scrollInput.forOptions(getTypeOptions(AllSchedules.INSTRUCTION_TYPES)).titled(CreateLang.translateDirect("schedule.instruction_type"))
                .writingTo(scrollInputLabel).calling(index -> {
                    Pair<Identifier, Function<Identifier, ? extends ScheduleInstruction>> pair = AllSchedules.INSTRUCTION_TYPES.get(index);
                    ScheduleInstruction newlyCreated = pair.getSecond().apply(pair.getFirst());
                    if (editingDestination.getId().equals(newlyCreated.getId()))
                        return;
                    editingDestination = newlyCreated;
                    updateEditorSubwidgets(AllScheduleRenders.get(newlyCreated), editingDestination);
                }).setState(startIndex);
        }

        if (input instanceof ScheduleWaitCondition cond) {
            int startIndex = 0;
            for (int i = 0; i < AllSchedules.CONDITION_TYPES.size(); i++)
                if (AllSchedules.CONDITION_TYPES.get(i).getFirst().equals(cond.getId()))
                    startIndex = i;
            editingCondition = cond;
            updateEditorSubwidgets((IScheduleInput<ScheduleWaitCondition>) field, editingCondition);
            scrollInput.forOptions(getTypeOptions(AllSchedules.CONDITION_TYPES)).titled(CreateLang.translateDirect("schedule.condition_type"))
                .writingTo(scrollInputLabel).calling(index -> {
                    Pair<Identifier, Function<Identifier, ? extends ScheduleWaitCondition>> pair = AllSchedules.CONDITION_TYPES.get(index);
                    ScheduleWaitCondition newlyCreated = pair.getSecond().apply(pair.getFirst());
                    if (editingCondition.getId().equals(newlyCreated.getId()))
                        return;
                    editingCondition = newlyCreated;
                    updateEditorSubwidgets(AllScheduleRenders.get(newlyCreated), editingCondition);
                }).setState(startIndex);
        }

        addRenderableWidget(scrollInput);
        addRenderableWidget(scrollInputLabel);
        addRenderableWidget(editorConfirm);
        if (allowDeletion)
            addRenderableWidget(editorDelete);
    }

    private void onDestinationEdited(String text) {
        if (destinationSuggestions != null)
            destinationSuggestions.updateCommandInfo();
    }

    protected void stopEditing() {
        confirmButton.visible = true;
        cyclicButton.visible = true;
        cyclicIndicator.visible = true;
        skipProgress.visible = true;
        resetProgress.visible = true;

        if (editingCondition == null && editingDestination == null)
            return;

        destinationSuggestions = null;

        removeWidget(scrollInput);
        removeWidget(scrollInputLabel);
        removeWidget(editorConfirm);
        removeWidget(editorDelete);

        ScheduleDataEntry input = editingCondition == null ? editingDestination : editingCondition;
        IScheduleInput<ScheduleDataEntry> editing = AllScheduleRenders.get(input);
        for (int i = 0; i < editing.slotsTargeted(); i++) {
            editing.setItem(input, i, menu.ghostInventory.getItem(i));
            minecraft.player.connection.send(new GhostItemSubmitPacket(ItemStack.EMPTY, i));
        }

        editorSubWidgets.saveValues(input.getData());
        editorSubWidgets.forEach(this::removeWidget);
        editorSubWidgets.clear();

        editingCondition = null;
        editingDestination = null;
        editorConfirm = null;
        editorDelete = null;
        menu.slotsActive = false;
        renderedItem.getRenderElement().clear();
        init();
    }

    protected <T extends ScheduleDataEntry> void updateEditorSubwidgets(IScheduleInput<T> field, T input) {
        destinationSuggestions = null;
        menu.targetSlotsActive = field.slotsTargeted();

        editorSubWidgets.forEach(this::removeWidget);
        editorSubWidgets.clear();
        field.initConfigurationWidgets(input, new ModularGuiLineBuilder(font, editorSubWidgets, leftPos + 77, topPos + 92).speechBubble());
        editorSubWidgets.loadValues(input.getData(), this::addRenderableWidget, this::addRenderableOnly);

        if (!(input instanceof DestinationInstruction destinationInstruction))
            return;

        editorSubWidgets.forEach(e -> {
            if (!(e instanceof EditBox destinationBox))
                return;
            destinationSuggestions = new DestinationSuggestions(
                minecraft,
                this,
                destinationBox,
                font,
                getViableStations(destinationInstruction),
                false,
                topPos + 33
            );
            destinationSuggestions.setAllowSuggestions(true);
            destinationSuggestions.updateCommandInfo();
            destinationBox.setResponder(this::onDestinationEdited);
        });
    }

    private List<IntAttached<String>> getViableStations(DestinationInstruction field) {
        GlobalRailwayManager railwayManager = Create.RAILWAYS;
        Set<TrackGraph> viableGraphs = new HashSet<>(railwayManager.trackNetworks.values());

        for (ScheduleEntry entry : schedule.entries) {
            if (!(entry.instruction instanceof DestinationInstruction destination))
                continue;
            if (destination == field)
                continue;
            String filter = destination.getFilterForRegex();
            if (filter.isBlank())
                continue;
            Graphs:
            for (Iterator<TrackGraph> iterator = viableGraphs.iterator(); iterator.hasNext(); ) {
                TrackGraph trackGraph = iterator.next();
                for (GlobalStation station : trackGraph.getPoints(EdgePointType.STATION)) {
                    if (station.name.matches(filter))
                        continue Graphs;
                }
                iterator.remove();
            }
        }

        if (viableGraphs.isEmpty())
            viableGraphs = new HashSet<>(railwayManager.trackNetworks.values());

        Vec3 position = minecraft.player.position();
        Set<String> visited = new HashSet<>();

        return viableGraphs.stream().flatMap(g -> g.getPoints(EdgePointType.STATION).stream()).filter(station -> station.blockEntityPos != null)
            .filter(station -> visited.add(station.name))
            .map(station -> IntAttached.with((int) Vec3.atBottomCenterOf(station.blockEntityPos).distanceTo(position), station.name)).toList();
    }

    @Override
    protected void containerTick() {
        super.containerTick();
        scroll.tickChaser();
        for (LerpedFloat lerpedFloat : horizontalScrolls)
            lerpedFloat.tickChaser();

        if (destinationSuggestions != null)
            destinationSuggestions.tick();

        schedule.savedProgress = schedule.entries.isEmpty() ? 0 : Mth.clamp(schedule.savedProgress, 0, schedule.entries.size() - 1);
        resetProgress.active = schedule.savedProgress > 0;
        skipProgress.active = schedule.entries.size() > 1;
    }

    @Override
    public void render(GuiGraphics graphics, int mouseX, int mouseY, float partialTicks) {
        partialTicks = minecraft.getDeltaTracker().getGameTimeDeltaPartialTick(false);

        if (menu.slotsActive)
            super.render(graphics, mouseX, mouseY, partialTicks);
        else {
            for (Renderable widget : renderables)
                widget.render(graphics, mouseX, mouseY, partialTicks);
            renderForeground(graphics, mouseX, mouseY, partialTicks);
        }
    }

    protected void renderSchedule(GuiGraphics graphics, int mouseX, int mouseY, float partialTicks) {
        Matrix3x2fStack matrixStack = graphics.pose();
        UIRenderHelper.drawStretched(graphics, leftPos + 33, topPos + 16, 3, 173, AllGuiTextures.SCHEDULE_STRIP_DARK);

        int yOffset = 25;
        List<ScheduleEntry> entries = schedule.entries;
        float scrollOffset = -scroll.getValue(partialTicks);

        graphics.enableScissor(leftPos, topPos + 16, leftPos + 236, topPos + 189);

        for (int i = 0; i <= entries.size(); i++) {

            if (schedule.savedProgress == i && !schedule.entries.isEmpty()) {
                matrixStack.pushMatrix();
                float expectedY = scrollOffset + topPos + yOffset + 4;
                float actualY = Mth.clamp(expectedY, topPos + 18, topPos + 170);
                matrixStack.translate(0, actualY);
                (expectedY == actualY ? AllGuiTextures.SCHEDULE_POINTER : AllGuiTextures.SCHEDULE_POINTER_OFFSCREEN).render(graphics, leftPos, 0);
                matrixStack.popMatrix();
            }

            matrixStack.pushMatrix();
            matrixStack.translate(0, scrollOffset);
            if (i == 0 || entries.size() == 0)
                UIRenderHelper.drawStretched(graphics, leftPos + 33, topPos + 16, 3, 10, AllGuiTextures.SCHEDULE_STRIP_LIGHT);

            if (i == entries.size()) {
                if (i > 0)
                    yOffset += 9;
                AllGuiTextures.SCHEDULE_STRIP_END.render(graphics, leftPos + 29, topPos + yOffset);
                AllGuiTextures.SCHEDULE_CARD_NEW.render(graphics, leftPos + 43, topPos + yOffset);
                matrixStack.popMatrix();
                break;
            }

            ScheduleEntry scheduleEntry = entries.get(i);
            int cardY = yOffset;
            int cardHeight = renderScheduleEntry(graphics, scheduleEntry, cardY, mouseX, mouseY, partialTicks);
            yOffset += cardHeight;

            if (i + 1 < entries.size()) {
                AllGuiTextures.SCHEDULE_STRIP_DOTTED.render(graphics, leftPos + 29, topPos + yOffset - 3);
                yOffset += 10;
            }

            matrixStack.popMatrix();

            if (!scheduleEntry.instruction.supportsConditions())
                continue;

            float h = cardHeight - 26;
            float y1 = cardY + 24 + scrollOffset;
            float y2 = y1 + h;
            if (y2 > 189)
                h -= y2 - 189;
            if (y1 < 16) {
                float correction = 16 - y1;
                y1 += correction;
                h -= correction;
            }

            if (h <= 0)
                continue;

            graphics.enableScissor(leftPos + 43, 0, leftPos + 204, 400);
            matrixStack.pushMatrix();
            matrixStack.translate(0, scrollOffset);
            renderScheduleConditions(graphics, scheduleEntry, cardY, mouseX, mouseY, partialTicks, cardHeight, i);
            matrixStack.popMatrix();
            graphics.disableScissor();

            if (isConditionAreaScrollable(scheduleEntry)) {
                matrixStack.pushMatrix();
                matrixStack.translate(0, scrollOffset);
                int center = (cardHeight - 8 + CARD_HEADER) / 2;
                float chaseTarget = horizontalScrolls.get(i).getChaseTarget();
                if (!Mth.equal(chaseTarget, 0))
                    AllGuiTextures.SCHEDULE_SCROLL_LEFT.render(graphics, leftPos + 40, topPos + cardY + center);
                if (!Mth.equal(chaseTarget, scheduleEntry.conditions.size() - 1))
                    AllGuiTextures.SCHEDULE_SCROLL_RIGHT.render(graphics, leftPos + 203, topPos + cardY + center);
                matrixStack.popMatrix();
            }
        }

        graphics.disableScissor();

        graphics.fillGradient(leftPos + 16, topPos + 16, leftPos + 16 + 220, topPos + 16 + 10, 0x77000000, 0x00000000);
        graphics.fillGradient(leftPos + 16, topPos + 179, leftPos + 16 + 220, topPos + 179 + 10, 0x00000000, 0x77000000);
    }

    public int renderScheduleEntry(GuiGraphics graphics, ScheduleEntry entry, int yOffset, int mouseX, int mouseY, float partialTicks) {
        int zLevel = 100;
        AllGuiTextures light = AllGuiTextures.SCHEDULE_CARD_LIGHT;
        AllGuiTextures medium = AllGuiTextures.SCHEDULE_CARD_MEDIUM;
        AllGuiTextures dark = AllGuiTextures.SCHEDULE_CARD_DARK;

        int cardWidth = CARD_WIDTH;
        int cardHeader = CARD_HEADER;
        int maxRows = 0;
        for (List<ScheduleWaitCondition> list : entry.conditions)
            maxRows = Math.max(maxRows, list.size());
        ScheduleInstruction instruction = entry.instruction;
        boolean supportsConditions = instruction.supportsConditions();
        int cardHeight = cardHeader + (supportsConditions ? 24 + maxRows * 18 : 4);

        Matrix3x2fStack matrixStack = graphics.pose();
        matrixStack.pushMatrix();
        matrixStack.translate(leftPos + 25, topPos + yOffset);

        UIRenderHelper.drawStretched(graphics, 0, 1, cardWidth, cardHeight - 2, light);
        UIRenderHelper.drawStretched(graphics, 1, 0, cardWidth - 2, cardHeight, light);
        UIRenderHelper.drawStretched(graphics, 1, 1, cardWidth - 2, cardHeight - 2, dark);
        UIRenderHelper.drawStretched(graphics, 2, 2, cardWidth - 4, cardHeight - 4, medium);
        UIRenderHelper.drawStretched(graphics, 2, 2, cardWidth - 4, cardHeader, supportsConditions ? light : medium);

        AllGuiTextures.SCHEDULE_CARD_REMOVE.render(graphics, cardWidth - 14, 2);
        AllGuiTextures.SCHEDULE_CARD_DUPLICATE.render(graphics, cardWidth - 14, cardHeight - 14);

        int i = schedule.entries.indexOf(entry);
        if (i > 0)
            AllGuiTextures.SCHEDULE_CARD_MOVE_UP.render(graphics, cardWidth, cardHeader - 14);
        if (i < schedule.entries.size() - 1)
            AllGuiTextures.SCHEDULE_CARD_MOVE_DOWN.render(graphics, cardWidth, cardHeader);

        UIRenderHelper.drawStretched(graphics, 8, 0, 3, cardHeight + 10, AllGuiTextures.SCHEDULE_STRIP_LIGHT);
        (supportsConditions ? AllGuiTextures.SCHEDULE_STRIP_TRAVEL : AllGuiTextures.SCHEDULE_STRIP_ACTION).render(graphics, 4, 6);

        if (supportsConditions)
            AllGuiTextures.SCHEDULE_STRIP_WAIT.render(graphics, 4, 28);

        IScheduleInput<ScheduleInstruction> scheduleInput = AllScheduleRenders.get(instruction);
        Pair<ItemStack, Component> destination = scheduleInput.getSummary(instruction);
        renderInput(graphics, destination, 26, 5, false, 100);
        scheduleInput.renderSpecialIcon(instruction, graphics, 30, 5);

        matrixStack.popMatrix();

        return cardHeight;
    }

    public void renderScheduleConditions(
        GuiGraphics graphics,
        ScheduleEntry entry,
        int yOffset,
        int mouseX,
        int mouseY,
        float partialTicks,
        int cardHeight,
        int entryIndex
    ) {
        int cardWidth = CARD_WIDTH;
        int cardHeader = CARD_HEADER;

        Matrix3x2fStack matrixStack = graphics.pose();
        matrixStack.pushMatrix();
        matrixStack.translate(leftPos + 25, topPos + yOffset);
        int xOffset = 26;
        float scrollOffset = getConditionScroll(entry, partialTicks, entryIndex);

        matrixStack.pushMatrix();
        matrixStack.translate(-scrollOffset, 0);

        for (List<ScheduleWaitCondition> list : entry.conditions) {
            int maxWidth = getConditionColumnWidth(list);
            for (int i = 0; i < list.size(); i++) {
                ScheduleWaitCondition scheduleWaitCondition = list.get(i);
                IScheduleInput<ScheduleWaitCondition> scheduleInput = AllScheduleRenders.get(scheduleWaitCondition);
                Math.max(maxWidth, renderInput(graphics, scheduleInput.getSummary(scheduleWaitCondition), xOffset, 29 + i * 18, i != 0, maxWidth));
                scheduleInput.renderSpecialIcon(scheduleWaitCondition, graphics, xOffset + 4, 29 + i * 18);
            }

            AllGuiTextures.SCHEDULE_CONDITION_APPEND.render(graphics, xOffset + (maxWidth - 10) / 2, 29 + list.size() * 18);
            xOffset += maxWidth + 10;
        }

        AllGuiTextures.SCHEDULE_CONDITION_NEW.render(graphics, xOffset - 3, 29);
        matrixStack.popMatrix();

        if (xOffset + 16 > cardWidth - 26) {
            matrixStack.rotate(Mth.DEG_TO_RAD * -90);
            graphics.fillGradient(-cardHeight + 2, 18, -2 - cardHeader, 28, 0x44000000, 0x00000000);
            graphics.fillGradient(-cardHeight + 2, cardWidth - 26, -2 - cardHeader, cardWidth - 16, 0x00000000, 0x44000000);
        }

        matrixStack.popMatrix();
    }

    private boolean isConditionAreaScrollable(ScheduleEntry entry) {
        int xOffset = 26;
        for (List<ScheduleWaitCondition> list : entry.conditions)
            xOffset += getConditionColumnWidth(list) + 10;
        return xOffset + 16 > CARD_WIDTH - 26;
    }

    private float getConditionScroll(ScheduleEntry entry, float partialTicks, int entryIndex) {
        float scrollOffset = 0;
        float scrollIndex = horizontalScrolls.get(entryIndex).getValue(partialTicks);
        for (List<ScheduleWaitCondition> list : entry.conditions) {
            int maxWidth = getConditionColumnWidth(list);
            float partialOfThisColumn = Math.min(1, scrollIndex);
            scrollOffset += (maxWidth + 10) * partialOfThisColumn;
            scrollIndex -= partialOfThisColumn;
        }
        return scrollOffset;
    }

    private int getConditionColumnWidth(List<ScheduleWaitCondition> list) {
        int maxWidth = 0;
        for (ScheduleWaitCondition scheduleWaitCondition : list) {
            IScheduleInput<ScheduleWaitCondition> scheduleInput = AllScheduleRenders.get(scheduleWaitCondition);
            maxWidth = Math.max(maxWidth, getFieldSize(32, scheduleInput.getSummary(scheduleWaitCondition)));
        }
        return maxWidth;
    }

    protected int renderInput(GuiGraphics graphics, Pair<ItemStack, Component> pair, int x, int y, boolean clean, int minSize) {
        ItemStack stack = pair.getFirst();
        Component text = pair.getSecond();
        boolean hasItem = !stack.isEmpty();
        int fieldSize = Math.min(getFieldSize(minSize, pair), 150);
        Matrix3x2fStack matrixStack = graphics.pose();
        matrixStack.pushMatrix();

        AllGuiTextures left = clean ? AllGuiTextures.SCHEDULE_CONDITION_LEFT_CLEAN : AllGuiTextures.SCHEDULE_CONDITION_LEFT;
        AllGuiTextures middle = AllGuiTextures.SCHEDULE_CONDITION_MIDDLE;
        AllGuiTextures item = AllGuiTextures.SCHEDULE_CONDITION_ITEM;
        AllGuiTextures right = AllGuiTextures.SCHEDULE_CONDITION_RIGHT;

        matrixStack.translate(x, y);
        UIRenderHelper.drawStretched(graphics, 0, 0, fieldSize, 16, middle);
        left.render(graphics, clean ? 0 : -3, 0);
        right.render(graphics, fieldSize - 2, 0);
        if (hasItem) {
            item.render(graphics, 3, 0);
            if (stack.getItem() != Items.STRUCTURE_VOID)
                graphics.renderItem(stack, 4, 0);
        }

        if (text != null)
            graphics.drawString(font, font.substrByWidth(text, 120).getString(), hasItem ? 28 : 8, 4, 0xff_f2f2ee, true);

        matrixStack.popMatrix();
        return fieldSize;
    }

    private Component clickToEdit = CreateLang.translateDirect("gui.schedule.lmb_edit").withStyle(ChatFormatting.DARK_GRAY, ChatFormatting.ITALIC);
    private Component rClickToDelete = CreateLang.translateDirect("gui.schedule.rmb_remove")
        .withStyle(ChatFormatting.DARK_GRAY, ChatFormatting.ITALIC);

    public boolean action(@Nullable GuiGraphics graphics, double mouseX, double mouseY, int click) {
        if (editingCondition != null || editingDestination != null)
            return false;

        Component empty = CommonComponents.EMPTY;

        int mx = (int) mouseX;
        int my = (int) mouseY;
        int x = mx - this.leftPos - 25;
        int y = my - this.topPos - 25;
        if (x < 0 || x >= 205)
            return false;
        if (y < 0 || y >= 173)
            return false;
        y += scroll.getValue(0);

        List<ScheduleEntry> entries = schedule.entries;
        for (int i = 0; i < entries.size(); i++) {
            ScheduleEntry entry = entries.get(i);
            int maxRows = 0;
            for (List<ScheduleWaitCondition> list : entry.conditions)
                maxRows = Math.max(maxRows, list.size());
            ScheduleInstruction instruction = entry.instruction;
            int cardHeight = CARD_HEADER + (instruction.supportsConditions() ? 24 + maxRows * 18 : 4);

            if (y >= cardHeight + 5) {
                y -= cardHeight + 10;
                if (y < 0)
                    return false;
                continue;
            }

            IScheduleInput<ScheduleInstruction> input = AllScheduleRenders.get(instruction);
            int fieldSize = getFieldSize(100, input.getSummary(instruction));
            if (x > 25 && x <= 25 + fieldSize && y > 4 && y <= 20) {
                List<Component> components = new ArrayList<>();
                components.addAll(input.getTitleAs(instruction, "instruction"));
                components.add(empty);
                components.add(clickToEdit);
                renderActionTooltip(graphics, components, mx, my);
                if (click == 0)
                    startEditing(
                        input, instruction, confirmed -> {
                            if (confirmed)
                                entry.instruction = editingDestination;
                        }, false
                    );
                return true;
            }

            if (x > 180 && x <= 192) {
                if (y > 0 && y <= 14) {
                    renderActionTooltip(graphics, ImmutableList.of(CreateLang.translateDirect("gui.schedule.remove_entry")), mx, my);
                    if (click == 0) {
                        entries.remove(entry);
                        renderedItem.getRenderElement().clear();
                        init();
                    }
                    return true;
                }
                if (y > cardHeight - 14) {
                    renderActionTooltip(graphics, ImmutableList.of(CreateLang.translateDirect("gui.schedule.duplicate")), mx, my);
                    if (click == 0) {
                        entries.add(entries.indexOf(entry), entry.clone(minecraft.level.registryAccess()));
                        renderedItem.getRenderElement().clear();
                        init();
                    }
                    return true;
                }
            }

            if (x > 194) {
                if (y > 7 && y <= 20 && i > 0) {
                    renderActionTooltip(graphics, ImmutableList.of(CreateLang.translateDirect("gui.schedule.move_up")), mx, my);
                    if (click == 0) {
                        entries.remove(entry);
                        entries.add(i - 1, entry);
                        renderedItem.getRenderElement().clear();
                        init();
                    }
                    return true;
                }
                if (y > 20 && y <= 33 && i < entries.size() - 1) {
                    renderActionTooltip(graphics, ImmutableList.of(CreateLang.translateDirect("gui.schedule.move_down")), mx, my);
                    if (click == 0) {
                        entries.remove(entry);
                        entries.add(i + 1, entry);
                        renderedItem.getRenderElement().clear();
                        init();
                    }
                    return true;
                }
            }

            int center = (cardHeight - 8 + CARD_HEADER) / 2;
            if (y > center - 1 && y <= center + 7 && isConditionAreaScrollable(entry)) {
                float chaseTarget = horizontalScrolls.get(i).getChaseTarget();
                if (x > 12 && x <= 19 && !Mth.equal(chaseTarget, 0)) {
                    if (click == 0)
                        horizontalScrolls.get(i).chase(chaseTarget - 1, 0.5f, Chaser.EXP);
                    return true;
                }
                if (x > 177 && x <= 184 && !Mth.equal(chaseTarget, entry.conditions.size() - 1)) {
                    if (click == 0)
                        horizontalScrolls.get(i).chase(chaseTarget + 1, 0.5f, Chaser.EXP);
                    return true;
                }
            }

            x -= 18;
            y -= 28;
            if (x < 0 || y < 0 || x > 160)
                return false;
            x += getConditionScroll(entry, 0, i) - 8;

            List<List<ScheduleWaitCondition>> columns = entry.conditions;
            for (int j = 0; j < columns.size(); j++) {
                List<ScheduleWaitCondition> conditions = columns.get(j);
                if (x < 0)
                    return false;
                int w = getConditionColumnWidth(conditions);
                if (x >= w) {
                    x -= w + 10;
                    continue;
                }

                int row = y / 18;
                if (row < conditions.size() && row >= 0) {
                    boolean canRemove = conditions.size() > 1 || columns.size() > 1;
                    List<Component> components = new ArrayList<>();
                    components.add(CreateLang.translateDirect("schedule.condition_type").withStyle(ChatFormatting.GRAY));
                    ScheduleWaitCondition condition = conditions.get(row);
                    IScheduleInput<ScheduleWaitCondition> scheduleInput = AllScheduleRenders.get(condition);
                    components.addAll(scheduleInput.getTitleAs(condition, "condition"));
                    components.add(empty);
                    components.add(clickToEdit);
                    if (canRemove)
                        components.add(rClickToDelete);
                    renderActionTooltip(graphics, components, mx, my);
                    if (canRemove && click == 1) {
                        conditions.remove(row);
                        if (conditions.isEmpty())
                            columns.remove(conditions);
                    }
                    if (click == 0)
                        startEditing(
                            scheduleInput, condition, confirmed -> {
                                conditions.remove(row);
                                if (confirmed) {
                                    conditions.add(row, editingCondition);
                                    return;
                                }
                                if (conditions.isEmpty())
                                    columns.remove(conditions);
                            }, canRemove
                        );
                    return true;
                }

                if (y > 18 * conditions.size() && y <= 18 * conditions.size() + 10 && x >= w / 2 - 5 && x < w / 2 + 5) {
                    renderActionTooltip(graphics, ImmutableList.of(CreateLang.translateDirect("gui.schedule.add_condition")), mx, my);
                    if (click == 0) {
                        ScheduleWaitCondition condition = AllSchedules.createScheduleWaitCondition(AllSchedules.DELAY);
                        IScheduleInput<ScheduleWaitCondition> scheduleInput = AllScheduleRenders.get(condition);
                        startEditing(
                            scheduleInput, condition, confirmed -> {
                                if (confirmed)
                                    conditions.add(editingCondition);
                            }, true
                        );
                    }
                    return true;
                }

                return false;
            }

            if (x < 0 || x > 15 || y > 20)
                return false;

            renderActionTooltip(graphics, ImmutableList.of(CreateLang.translateDirect("gui.schedule.alternative_condition")), mx, my);
            if (click == 0) {
                ScheduleWaitCondition condition = AllSchedules.createScheduleWaitCondition(AllSchedules.DELAY);
                IScheduleInput<ScheduleWaitCondition> scheduleInput = AllScheduleRenders.get(condition);
                startEditing(
                    scheduleInput, condition, confirmed -> {
                        if (!confirmed)
                            return;
                        ArrayList<ScheduleWaitCondition> conditions = new ArrayList<>();
                        conditions.add(editingCondition);
                        columns.add(conditions);
                    }, true
                );
            }
            return true;
        }

        if (x < 18 || x > 33 || y > 14)
            return false;

        renderActionTooltip(graphics, ImmutableList.of(CreateLang.translateDirect("gui.schedule.add_entry")), mx, my);
        if (click == 0) {
            ScheduleInstruction instruction = AllSchedules.createScheduleInstruction(AllSchedules.DESTINATION);
            IScheduleInput<ScheduleInstruction> scheduleInput = AllScheduleRenders.get(instruction);
            startEditing(
                scheduleInput, instruction, confirmed -> {
                    if (!confirmed)
                        return;

                    ScheduleEntry entry = new ScheduleEntry();
                    ScheduleWaitCondition condition = AllSchedules.createScheduleWaitCondition(AllSchedules.DELAY);
                    ArrayList<ScheduleWaitCondition> initialConditions = new ArrayList<>();
                    initialConditions.add(condition);
                    entry.instruction = editingDestination;
                    entry.conditions.add(initialConditions);
                    schedule.entries.add(entry);
                }, true
            );
        }
        return true;
    }

    private void renderActionTooltip(@Nullable GuiGraphics graphics, List<Component> tooltip, int mx, int my) {
        if (graphics != null)
            graphics.setTooltipForNextFrame(font, tooltip, Optional.empty(), mx, my);
    }

    private int getFieldSize(int minSize, Pair<ItemStack, Component> pair) {
        ItemStack stack = pair.getFirst();
        Component text = pair.getSecond();
        boolean hasItem = !stack.isEmpty();
        return Math.max((text == null ? 0 : font.width(text)) + (hasItem ? 20 : 0) + 16, minSize);
    }

    @Override
    public boolean mouseClicked(MouseButtonEvent click, boolean doubled) {
        if (destinationSuggestions != null && destinationSuggestions.mouseClicked(click))
            return true;
        double pMouseX = click.x();
        double pMouseY = click.y();
        if (editorConfirm != null && editorConfirm.isMouseOver(pMouseX, pMouseY) && onEditorClose != null) {
            onEditorClose.accept(true);
            stopEditing();
            return true;
        }
        if (editorDelete != null && editorDelete.isMouseOver(pMouseX, pMouseY) && onEditorClose != null) {
            onEditorClose.accept(false);
            stopEditing();
            return true;
        }
        if (action(null, pMouseX, pMouseY, click.button()))
            return true;

        return super.mouseClicked(click, doubled);
    }

    @Override
    public boolean keyPressed(KeyEvent input) {
        if (destinationSuggestions != null && destinationSuggestions.keyPressed(input))
            return true;
        if (editingCondition == null && editingDestination == null)
            return super.keyPressed(input);
        int pKeyCode = input.key();
        boolean hitEnter = getFocused() instanceof EditBox && (pKeyCode == 257 || pKeyCode == 335);
        boolean hitE = getFocused() == null && minecraft.options.keyInventory.matches(input);
        if (hitE || hitEnter) {
            onEditorClose.accept(true);
            stopEditing();
            return true;
        }
        return super.keyPressed(input);
    }

    @Override
    public boolean mouseScrolled(double pMouseX, double pMouseY, double pScrollX, double pScrollY) {
        if (destinationSuggestions != null && destinationSuggestions.mouseScrolled(Mth.clamp(pScrollY, -1.0D, 1.0D)))
            return true;
        if (editingCondition != null || editingDestination != null)
            return getChildAt(pMouseX, pMouseY).filter(element -> element.mouseScrolled(pMouseX, pMouseY, pScrollX, pScrollY)).isPresent();

        if (AllKeys.hasShiftDown()) {
            List<ScheduleEntry> entries = schedule.entries;
            int y = (int) (pMouseY - this.topPos - 25 + scroll.getValue());
            for (int i = 0; i < entries.size(); i++) {
                ScheduleEntry entry = entries.get(i);
                int maxRows = 0;
                for (List<ScheduleWaitCondition> list : entry.conditions)
                    maxRows = Math.max(maxRows, list.size());
                int cardHeight = CARD_HEADER + 24 + maxRows * 18;

                if (y >= cardHeight) {
                    y -= cardHeight + 9;
                    if (y < 0)
                        break;
                    continue;
                }

                if (!isConditionAreaScrollable(entry))
                    break;
                if (y < 24)
                    break;
                if (pMouseX < leftPos + 25)
                    break;
                if (pMouseX > leftPos + 205)
                    break;
                float chaseTarget = horizontalScrolls.get(i).getChaseTarget();
                if (pScrollY > 0 && !Mth.equal(chaseTarget, 0)) {
                    horizontalScrolls.get(i).chase(chaseTarget - 1, 0.5f, Chaser.EXP);
                    return true;
                }
                if (pScrollY < 0 && !Mth.equal(chaseTarget, entry.conditions.size() - 1)) {
                    horizontalScrolls.get(i).chase(chaseTarget + 1, 0.5f, Chaser.EXP);
                    return true;
                }
                return false;
            }
        }

        float chaseTarget = scroll.getChaseTarget();
        float max = 40 - 173;
        for (ScheduleEntry scheduleEntry : schedule.entries) {
            int maxRows = 0;
            for (List<ScheduleWaitCondition> list : scheduleEntry.conditions)
                maxRows = Math.max(maxRows, list.size());
            max += CARD_HEADER + 24 + maxRows * 18 + 10;
        }
        if (max > 0) {
            chaseTarget -= pScrollY * 12;
            chaseTarget = Mth.clamp(chaseTarget, 0, max);
            scroll.chase((int) chaseTarget, 0.7f, Chaser.EXP);
        } else
            scroll.chase(0, 0.7f, Chaser.EXP);

        return super.mouseScrolled(pMouseX, pMouseY, pScrollX, pScrollY);
    }

    @Override
    protected void renderForeground(GuiGraphics graphics, int mouseX, int mouseY, float partialTicks) {
        Matrix3x2fStack matrixStack = graphics.pose();
        if (destinationSuggestions != null) {
            matrixStack.pushMatrix();
            destinationSuggestions.render(graphics, mouseX, mouseY);
            matrixStack.popMatrix();
        }

        super.renderForeground(graphics, mouseX, mouseY, partialTicks);

        action(graphics, mouseX, mouseY, -1);

        if (editingCondition == null && editingDestination == null)
            return;

        int x = this.leftPos + 53;
        int y = this.topPos + 87;
        if (mouseX < x || mouseY < y || mouseX >= x + 120 || mouseY >= y + 18)
            return;

        ScheduleDataEntry entry = editingCondition == null ? editingDestination : editingCondition;
        IScheduleInput<ScheduleDataEntry> rendered = AllScheduleRenders.get(entry);

        for (int i = 0; i < Math.max(1, rendered.slotsTargeted()); i++) {
            List<Component> secondLineTooltip = rendered.getSecondLineTooltip(i);
            if (secondLineTooltip == null)
                continue;
            Slot slot = menu.getSlot(36 + i);
            if (slot == null || !slot.getItem().isEmpty())
                continue;
            if (mouseX < this.leftPos + slot.x || mouseX > this.leftPos + slot.x + 18)
                continue;
            if (mouseY < this.topPos + slot.y || mouseY > this.topPos + slot.y + 18)
                continue;
            renderActionTooltip(graphics, secondLineTooltip, mouseX, mouseY);
        }
    }

    @Override
    protected void renderBg(GuiGraphics graphics, float pPartialTick, int pMouseX, int pMouseY) {
        pPartialTick = AnimationTickHolder.getPartialTicksUI(minecraft.getDeltaTracker());
        AllGuiTextures.SCHEDULE.render(graphics, leftPos, topPos);
        FormattedCharSequence formattedcharsequence = title.getVisualOrderText();
        int center = leftPos + (AllGuiTextures.SCHEDULE.getWidth() - 8) / 2;
        graphics.drawString(font, formattedcharsequence, center - font.width(formattedcharsequence) / 2, topPos + 4, 0xFF505050, false);
        renderSchedule(graphics, pMouseX, pMouseY, pPartialTick);

        if (editingCondition == null && editingDestination == null)
            return;

        graphics.fillGradient(0, 0, this.width, this.height, -1072689136, -804253680);
        AllGuiTextures.SCHEDULE_EDITOR.render(graphics, leftPos - 2, topPos + 40);
        AllGuiTextures.PLAYER_INVENTORY.render(graphics, leftPos + 38, topPos + 122);
        graphics.drawString(font, playerInventoryTitle, leftPos + 46, topPos + 128, 0xFF505050, false);

        formattedcharsequence = editingCondition == null ? CreateLang.translateDirect("schedule.instruction.editor")
            .getVisualOrderText() : CreateLang.translateDirect("schedule.condition.editor").getVisualOrderText();
        graphics.drawString(font, formattedcharsequence, (center - font.width(formattedcharsequence) / 2), topPos + 44, 0xFF505050, false);

        ScheduleDataEntry entry = editingCondition == null ? editingDestination : editingCondition;
        IScheduleInput<ScheduleDataEntry> rendered = AllScheduleRenders.get(entry);

        for (int i = 0; i < rendered.slotsTargeted(); i++)
            AllGuiTextures.SCHEDULE_EDITOR_ADDITIONAL_SLOT.render(graphics, leftPos + 53 + 20 * i, topPos + 87);

        if (rendered.slotsTargeted() == 0 && !rendered.renderSpecialIcon(entry, graphics, leftPos + 54, topPos + 88)) {
            Pair<ItemStack, Component> summary = rendered.getSummary(entry);
            ItemStack icon = summary.getFirst();
            if (icon.isEmpty())
                icon = rendered.getSecondLineIcon();
            if (icon.isEmpty())
                AllGuiTextures.SCHEDULE_EDITOR_INACTIVE_SLOT.render(graphics, leftPos + 53, topPos + 87);
            else
                graphics.renderItem(icon, leftPos + 54, topPos + 88);
        }

        Matrix3x2fStack pPoseStack = graphics.pose();
        pPoseStack.pushMatrix();
        pPoseStack.translate(0, topPos + 87);
        editorSubWidgets.renderWidgetBG(leftPos + 77, graphics);
        pPoseStack.popMatrix();
    }

    @Override
    public void removed() {
        super.removed();
        minecraft.player.connection.send(new ScheduleEditPacket(schedule));
    }

    @Override
    public List<Rect2i> getExtraAreas() {
        return extraAreas;
    }

}
