package com.zurrtum.create.content.contraptions.actors.roller;

import com.zurrtum.create.AllBlockTags;
import com.zurrtum.create.AllDamageSources;
import com.zurrtum.create.catnip.data.Couple;
import com.zurrtum.create.catnip.data.Iterate;
import com.zurrtum.create.catnip.data.Pair;
import com.zurrtum.create.catnip.math.VecHelper;
import com.zurrtum.create.content.contraptions.actors.roller.RollerBlockEntity.RollingMode;
import com.zurrtum.create.content.contraptions.behaviour.MovementContext;
import com.zurrtum.create.content.contraptions.pulley.PulleyContraption;
import com.zurrtum.create.content.kinetics.base.BlockBreakingMovementBehaviour;
import com.zurrtum.create.content.logistics.filter.FilterItemStack;
import com.zurrtum.create.content.trains.bogey.StandardBogeyBlock;
import com.zurrtum.create.content.trains.entity.*;
import com.zurrtum.create.content.trains.entity.TravellingPoint.ITrackSelector;
import com.zurrtum.create.content.trains.entity.TravellingPoint.SteerDirection;
import com.zurrtum.create.content.trains.graph.TrackEdge;
import com.zurrtum.create.content.trains.graph.TrackGraph;
import com.zurrtum.create.foundation.utility.BlockHelper;
import com.zurrtum.create.infrastructure.config.AllConfigs;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.core.Direction.Axis;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.nbt.NbtOps;
import net.minecraft.nbt.Tag;
import net.minecraft.resources.RegistryOps;
import net.minecraft.resources.Identifier;
import net.minecraft.tags.BlockTags;
import net.minecraft.util.Mth;
import net.minecraft.world.Container;
import net.minecraft.world.damagesource.DamageSource;
import net.minecraft.world.item.BlockItem;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.block.Blocks;
import net.minecraft.world.level.block.FallingBlock;
import net.minecraft.world.level.block.SlabBlock;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.block.state.properties.SlabType;
import net.minecraft.world.level.levelgen.structure.templatesystem.StructureTemplate.StructureBlockInfo;
import net.minecraft.world.phys.Vec3;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.BiConsumer;

public class RollerMovementBehaviour extends BlockBreakingMovementBehaviour {

    @Override
    public boolean isActive(MovementContext context) {
        return super.isActive(context) && !(context.contraption instanceof PulleyContraption) && VecHelper.isVecPointingTowards(
            context.relativeMotion,
            context.state.getValue(RollerBlock.FACING)
        );
    }

    @Override
    public boolean disableBlockEntityRendering() {
        return true;
    }

    @Override
    public Vec3 getActiveAreaOffset(MovementContext context) {
        return Vec3.atLowerCornerOf(context.state.getValue(RollerBlock.FACING).getUnitVec3i()).scale(.45).subtract(0, 2, 0);
    }

    @Override
    protected float getBlockBreakingSpeed(MovementContext context) {
        return Mth.clamp(super.getBlockBreakingSpeed(context) * 1.5f, 1 / 128f, 16f);
    }

    @Override
    public boolean canBreak(Level world, BlockPos breakingPos, BlockState state) {
        for (Direction side : Iterate.directions)
            if (world.getBlockState(breakingPos.relative(side)).is(BlockTags.PORTALS))
                return false;

        return super.canBreak(world, breakingPos, state) && !state.getCollisionShape(world, breakingPos).isEmpty() && !state.is(AllBlockTags.TRACKS);
    }

    @Override
    protected DamageSource getDamageSource(Level level) {
        return AllDamageSources.get(level).roller;
    }

    @Override
    public void visitNewPosition(MovementContext context, BlockPos pos) {
        Level world = context.world;
        BlockState stateVisited = world.getBlockState(pos);
        if (!stateVisited.isRedstoneConductor(world, pos))
            damageEntities(context, pos, world);
        if (world.isClientSide())
            return;

        List<BlockPos> positionsToBreak = getPositionsToBreak(context, pos);
        if (positionsToBreak.isEmpty()) {
            triggerPaver(context, pos);
            return;
        }

        BlockPos argMax = null;
        double max = -1;
        for (BlockPos toBreak : positionsToBreak) {
            float hardness = context.world.getBlockState(toBreak).getDestroySpeed(world, toBreak);
            if (hardness < max)
                continue;
            max = hardness;
            argMax = toBreak;
        }

        if (argMax == null) {
            triggerPaver(context, pos);
            return;
        }

        context.data.store("ReferencePos", BlockPos.CODEC, pos);
        context.data.store("BreakingPos", BlockPos.CODEC, argMax);
        context.stall = true;
    }

    @Override
    protected void onBlockBroken(MovementContext context, BlockPos pos, BlockState brokenState) {
        super.onBlockBroken(context, pos, brokenState);
        if (!context.data.contains("ReferencePos"))
            return;

        BlockPos referencePos = context.data.read("ReferencePos", BlockPos.CODEC).orElse(BlockPos.ZERO);
        for (BlockPos otherPos : getPositionsToBreak(context, referencePos))
            if (!otherPos.equals(pos))
                destroyBlock(context, otherPos);

        triggerPaver(context, referencePos);
        context.data.remove("ReferencePos");
    }

    @Override
    protected void destroyBlock(MovementContext context, BlockPos breakingPos) {
        BlockState blockState = context.world.getBlockState(breakingPos);
        boolean noHarvest = blockState.is(BlockTags.NEEDS_IRON_TOOL) || blockState.is(BlockTags.NEEDS_STONE_TOOL) || blockState.is(BlockTags.NEEDS_DIAMOND_TOOL);

        BlockHelper.destroyBlock(
            context.world, breakingPos, 1f, stack -> {
                if (noHarvest || context.world.random.nextBoolean())
                    return;
                this.dropItem(context, stack);
            }
        );

        super.destroyBlock(context, breakingPos);
    }

    RollerTravellingPoint rollerScout = new RollerTravellingPoint();

    protected List<BlockPos> getPositionsToBreak(MovementContext context, BlockPos visitedPos) {
        ArrayList<BlockPos> positions = new ArrayList<>();

        RollingMode mode = getMode(context);
        if (mode != RollingMode.TUNNEL_PAVE)
            return positions;

        int startingY = 1;
        if (!getStateToPaveWith(context).isAir()) {
            FilterItemStack filter = context.getFilterFromBE();
            Container inventory = context.contraption.getStorage().getAllItems();
            ItemStack count = inventory.count(stack -> filter.test(context.world, stack), 1);
            if (!count.isEmpty()) {
                startingY = 0;
            }
        }

        // Train
        PaveTask profileForTracks = createHeightProfileForTracks(context);
        if (profileForTracks != null) {
            for (Couple<Integer> coords : profileForTracks.keys()) {
                float height = profileForTracks.get(coords);
                BlockPos targetPosition = BlockPos.containing(coords.getFirst(), height, coords.getSecond());
                boolean shouldPlaceSlab = height > Math.floor(height) + .45;
                if (startingY == 1 && shouldPlaceSlab && context.world.getBlockState(targetPosition.above())
                    .getValueOrElse(SlabBlock.TYPE, SlabType.DOUBLE) == SlabType.BOTTOM)
                    startingY = 2;
                for (int i = startingY; i <= (shouldPlaceSlab ? 3 : 2); i++)
                    if (testBreakerTarget(context, targetPosition.above(i), i))
                        positions.add(targetPosition.above(i));
            }
            return positions;
        }

        // Otherwise
        for (int i = startingY; i <= 2; i++)
            if (testBreakerTarget(context, visitedPos.above(i), i))
                positions.add(visitedPos.above(i));

        return positions;
    }

    protected boolean testBreakerTarget(MovementContext context, BlockPos target, int columnY) {
        BlockState stateToPaveWith = getStateToPaveWith(context);
        BlockState stateToPaveWithAsSlab = getStateToPaveWithAsSlab(context);
        BlockState stateAbove = context.world.getBlockState(target);
        if (columnY == 0 && stateAbove.is(stateToPaveWith.getBlock()))
            return false;
        if (stateToPaveWithAsSlab != null && columnY == 1 && stateAbove.is(stateToPaveWithAsSlab.getBlock()))
            return false;
        return canBreak(context.world, target, stateAbove);
    }

    @Nullable
    protected PaveTask createHeightProfileForTracks(MovementContext context) {
        if (context.contraption == null)
            return null;
        if (!(context.contraption.entity instanceof CarriageContraptionEntity cce))
            return null;
        Carriage carriage = cce.getCarriage();
        if (carriage == null)
            return null;
        Train train = carriage.train;
        if (train == null || train.graph == null)
            return null;

        CarriageBogey mainBogey = carriage.bogeys.getFirst();
        TravellingPoint point = mainBogey.trailing();

        rollerScout.node1 = point.node1;
        rollerScout.node2 = point.node2;
        rollerScout.edge = point.edge;
        rollerScout.position = point.position;

        Axis axis = Axis.X;
        StructureBlockInfo info = context.contraption.getBlocks().get(BlockPos.ZERO);
        if (info != null && info.state().hasProperty(StandardBogeyBlock.AXIS))
            axis = info.state().getValue(StandardBogeyBlock.AXIS);

        Direction orientation = cce.getInitialOrientation();
        Direction rollerFacing = context.state.getValue(RollerBlock.FACING);

        int step = orientation.getAxisDirection().getStep();
        double widthWiseOffset = axis.choose(-context.localPos.getZ(), 0, -context.localPos.getX()) * step;
        double lengthWiseOffset = axis.choose(-context.localPos.getX(), 0, context.localPos.getZ()) * step - 1;

        if (rollerFacing == orientation.getClockWise())
            lengthWiseOffset += 1;

        double distanceToTravel = 2;
        PaveTask heightProfile = new PaveTask(widthWiseOffset, widthWiseOffset);
        ITrackSelector steering = rollerScout.steer(SteerDirection.NONE, new Vec3(0, 1, 0));

        rollerScout.traversalCallback = (edge, coords) -> {
        };
        rollerScout.travel(train.graph, lengthWiseOffset + 1, steering);

        rollerScout.traversalCallback = (edge, coords) -> {
            if (edge == null)
                return;
            if (edge.isInterDimensional())
                return;
            if (edge.node1.getLocation().dimension != context.world.dimension())
                return;
            TrackPaverV2.pave(heightProfile, train.graph, edge, coords.getFirst(), coords.getSecond());
        };
        rollerScout.travel(train.graph, distanceToTravel, steering);

        for (Couple<Integer> entry : heightProfile.keys())
            heightProfile.put(entry.getFirst(), entry.getSecond(), context.localPos.getY() + heightProfile.get(entry));

        return heightProfile;
    }

    protected void triggerPaver(MovementContext context, BlockPos pos) {
        BlockState stateToPaveWith = getStateToPaveWith(context);
        BlockState stateToPaveWithAsSlab = getStateToPaveWithAsSlab(context);
        RollingMode mode = getMode(context);

        if (mode != RollingMode.TUNNEL_PAVE && stateToPaveWith.isAir())
            return;

        Vec3 directionVec = Vec3.atLowerCornerOf(context.state.getValue(RollerBlock.FACING).getClockWise().getUnitVec3i());
        directionVec = context.rotation.apply(directionVec);
        PaveResult paveResult = PaveResult.PASS;
        int yOffset = 0;

        List<Pair<BlockPos, Boolean>> paveSet = new ArrayList<>();
        PaveTask profileForTracks = createHeightProfileForTracks(context);
        if (profileForTracks == null)
            paveSet.add(Pair.of(pos, false));
        else
            for (Couple<Integer> coords : profileForTracks.keys()) {
                float height = profileForTracks.get(coords);
                boolean shouldPlaceSlab = height > Math.floor(height) + .45;
                BlockPos targetPosition = BlockPos.containing(coords.getFirst(), height, coords.getSecond());
                paveSet.add(Pair.of(targetPosition, shouldPlaceSlab));
            }

        if (paveSet.isEmpty())
            return;

        while (paveResult == PaveResult.PASS) {
            if (yOffset > AllConfigs.server().kinetics.rollerFillDepth.get()) {
                paveResult = PaveResult.FAIL;
                break;
            }

            Set<Pair<BlockPos, Boolean>> currentLayer = new HashSet<>();
            if (mode == RollingMode.WIDE_FILL) {
                for (Pair<BlockPos, Boolean> anchor : paveSet) {
                    int radius = (yOffset + 1) / 2;
                    for (int i = -radius; i <= radius; i++)
                        for (int j = -radius; j <= radius; j++)
                            if (BlockPos.ZERO.distManhattan(new BlockPos(i, 0, j)) <= radius)
                                currentLayer.add(Pair.of(anchor.getFirst().offset(i, -yOffset, j), anchor.getSecond()));
                }
            } else
                for (Pair<BlockPos, Boolean> anchor : paveSet)
                    currentLayer.add(Pair.of(anchor.getFirst().below(yOffset), anchor.getSecond()));

            boolean completelyBlocked = true;
            boolean anyBlockPlaced = false;

            for (Pair<BlockPos, Boolean> currentPos : currentLayer) {
                if (stateToPaveWithAsSlab != null && yOffset == 0 && currentPos.getSecond())
                    tryFill(context, currentPos.getFirst().above(), stateToPaveWithAsSlab);
                paveResult = tryFill(context, currentPos.getFirst(), stateToPaveWith);
                if (paveResult != PaveResult.FAIL)
                    completelyBlocked = false;
                if (paveResult == PaveResult.SUCCESS)
                    anyBlockPlaced = true;
            }

            if (anyBlockPlaced)
                paveResult = PaveResult.SUCCESS;
            else if (!completelyBlocked || yOffset == 0)
                paveResult = PaveResult.PASS;

            if (paveResult == PaveResult.SUCCESS && stateToPaveWith.getBlock() instanceof FallingBlock)
                paveResult = PaveResult.PASS;
            if (paveResult != PaveResult.PASS)
                break;
            if (mode == RollingMode.TUNNEL_PAVE)
                break;

            yOffset++;
        }

        if (paveResult == PaveResult.SUCCESS) {
            context.data.putInt("WaitingTicks", 2);
            context.data.store("LastPos", BlockPos.CODEC, pos);
            context.stall = true;
        }
    }

    public static BlockState getStateToPaveWith(ItemStack itemStack) {
        if (itemStack.getItem() instanceof BlockItem bi) {
            BlockState defaultBlockState = bi.getBlock().defaultBlockState();
            if (defaultBlockState.hasProperty(SlabBlock.TYPE))
                defaultBlockState = defaultBlockState.setValue(SlabBlock.TYPE, SlabType.DOUBLE);
            return defaultBlockState;
        }
        return Blocks.AIR.defaultBlockState();
    }

    protected BlockState getStateToPaveWith(MovementContext context) {
        RegistryOps<Tag> ops = context.world.registryAccess().createSerializationContext(NbtOps.INSTANCE);
        ItemStack filter = context.blockEntityData.read("Filter", ItemStack.OPTIONAL_CODEC, ops).orElse(ItemStack.EMPTY);
        return getStateToPaveWith(filter);
    }

    protected BlockState getStateToPaveWithAsSlab(MovementContext context) {
        BlockState stateToPaveWith = getStateToPaveWith(context);
        if (stateToPaveWith.hasProperty(SlabBlock.TYPE))
            return stateToPaveWith.setValue(SlabBlock.TYPE, SlabType.BOTTOM);

        Block block = stateToPaveWith.getBlock();
        if (block == null)
            return null;

        Identifier rl = BuiltInRegistries.BLOCK.getKey(block);
        String namespace = rl.getNamespace();
        String blockName = rl.getPath();
        int nameLength = blockName.length();

        List<String> possibleSlabLocations = new ArrayList<>();
        possibleSlabLocations.add(blockName + "_slab");

        if (blockName.endsWith("s") && nameLength > 1)
            possibleSlabLocations.add(blockName.substring(0, nameLength - 1) + "_slab");
        if (blockName.endsWith("planks") && nameLength > 7)
            possibleSlabLocations.add(blockName.substring(0, nameLength - 7) + "_slab");

        for (String locationAttempt : possibleSlabLocations) {
            Optional<Block> result = BuiltInRegistries.BLOCK.getOptional(Identifier.fromNamespaceAndPath(namespace, locationAttempt));
            if (result.isEmpty())
                continue;
            return result.get().defaultBlockState();
        }

        return null;
    }

    protected RollingMode getMode(MovementContext context) {
        return RollingMode.values()[context.blockEntityData.getIntOr("ScrollValue", 0)];
    }

    private static final class RollerTravellingPoint extends TravellingPoint {

        public BiConsumer<TrackEdge, Couple<Double>> traversalCallback;

        @Override
        protected Double edgeTraversedFrom(
            TrackGraph graph,
            boolean forward,
            IEdgePointListener edgePointListener,
            ITurnListener turnListener,
            double prevPos,
            double totalDistance
        ) {
            double from = forward ? prevPos : position;
            double to = forward ? position : prevPos;
            traversalCallback.accept(edge, Couple.create(from, to));
            return super.edgeTraversedFrom(graph, forward, edgePointListener, turnListener, prevPos, totalDistance);
        }

    }

    protected enum PaveResult {
        FAIL,
        PASS,
        SUCCESS;
    }

    protected PaveResult tryFill(MovementContext context, BlockPos targetPos, BlockState toPlace) {
        Level level = context.world;
        if (!level.isLoaded(targetPos))
            return PaveResult.FAIL;
        BlockState existing = level.getBlockState(targetPos);
        if (existing.is(toPlace.getBlock()))
            return PaveResult.PASS;
        if (!existing.is(BlockTags.LEAVES) && !existing.canBeReplaced() && (!existing.getCollisionShape(level, targetPos).isEmpty() || existing.is(
            BlockTags.PORTALS)))
            return PaveResult.FAIL;

        FilterItemStack filter = context.getFilterFromBE();
        Container inventory = context.contraption.getStorage().getAllItems();
        ItemStack held = inventory.extract(stack -> filter.test(context.world, stack), 1);
        if (held.isEmpty())
            return PaveResult.FAIL;

        level.setBlockAndUpdate(targetPos, toPlace);
        return PaveResult.SUCCESS;
    }

}
