package com.zurrtum.create.content.contraptions.chassis;

import com.zurrtum.create.AllBlockEntityTypes;
import com.zurrtum.create.AllItemTags;
import com.zurrtum.create.AllItems;
import com.zurrtum.create.AllSoundEvents;
import com.zurrtum.create.api.contraption.transformable.TransformableBlock;
import com.zurrtum.create.catnip.data.Iterate;
import com.zurrtum.create.content.contraptions.StructureTransform;
import com.zurrtum.create.content.equipment.wrench.IWrenchable;
import com.zurrtum.create.foundation.block.IBE;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.core.particles.ParticleTypes;
import net.minecraft.world.InteractionHand;
import net.minecraft.world.InteractionResult;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.BlockGetter;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.block.Mirror;
import net.minecraft.world.level.block.RotatedPillarBlock;
import net.minecraft.world.level.block.Rotation;
import net.minecraft.world.level.block.entity.BlockEntityType;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.block.state.properties.BooleanProperty;
import net.minecraft.world.phys.BlockHitResult;
import net.minecraft.world.phys.Vec3;

public abstract class AbstractChassisBlock extends RotatedPillarBlock implements IWrenchable, IBE<ChassisBlockEntity>, TransformableBlock {

    public AbstractChassisBlock(Properties properties) {
        super(properties);
    }

    @Override
    protected InteractionResult useItemOn(
        ItemStack stack,
        BlockState state,
        Level level,
        BlockPos pos,
        Player player,
        InteractionHand hand,
        BlockHitResult hitResult
    ) {
        if (!player.mayBuild())
            return InteractionResult.TRY_WITH_EMPTY_HAND;

        boolean isSlimeBall = stack.is(AllItemTags.SLIME_BALLS) || stack.is(AllItems.SUPER_GLUE);

        BooleanProperty affectedSide = getGlueableSide(state, hitResult.getDirection());
        if (affectedSide == null)
            return InteractionResult.TRY_WITH_EMPTY_HAND;

        if (isSlimeBall && state.getValue(affectedSide)) {
            for (Direction face : Iterate.directions) {
                BooleanProperty glueableSide = getGlueableSide(state, face);
                if (glueableSide != null && !state.getValue(glueableSide) && glueAllowedOnSide(level, pos, state, face)) {
                    if (level.isClientSide()) {
                        Vec3 vec = hitResult.getLocation();
                        level.addParticle(ParticleTypes.ITEM_SLIME, vec.x, vec.y, vec.z, 0, 0, 0);
                        return InteractionResult.SUCCESS;
                    }
                    AllSoundEvents.SLIME_ADDED.playOnServer(level, pos, .5f, 1);
                    state = state.setValue(glueableSide, true);
                }
            }
            if (!level.isClientSide())
                level.setBlockAndUpdate(pos, state);
            return InteractionResult.SUCCESS;
        }

        if ((!stack.isEmpty() || !player.isShiftKeyDown()) && !isSlimeBall)
            return InteractionResult.TRY_WITH_EMPTY_HAND;
        if (state.getValue(affectedSide) == isSlimeBall)
            return InteractionResult.TRY_WITH_EMPTY_HAND;
        if (!glueAllowedOnSide(level, pos, state, hitResult.getDirection()))
            return InteractionResult.TRY_WITH_EMPTY_HAND;
        if (level.isClientSide()) {
            Vec3 vec = hitResult.getLocation();
            level.addParticle(ParticleTypes.ITEM_SLIME, vec.x, vec.y, vec.z, 0, 0, 0);
            return InteractionResult.SUCCESS;
        }

        AllSoundEvents.SLIME_ADDED.playOnServer(level, pos, .5f, 1);
        level.setBlockAndUpdate(pos, state.setValue(affectedSide, isSlimeBall));
        return InteractionResult.SUCCESS;
    }

    @Override
    public BlockState rotate(BlockState state, Rotation rotation) {
        if (rotation == Rotation.NONE)
            return state;

        BlockState rotated = super.rotate(state, rotation);
        for (Direction face : Iterate.directions) {
            BooleanProperty glueableSide = getGlueableSide(rotated, face);
            if (glueableSide != null)
                rotated = rotated.setValue(glueableSide, false);
        }

        for (Direction face : Iterate.directions) {
            BooleanProperty glueableSide = getGlueableSide(state, face);
            if (glueableSide == null || !state.getValue(glueableSide))
                continue;
            Direction rotatedFacing = rotation.rotate(face);
            BooleanProperty rotatedGlueableSide = getGlueableSide(rotated, rotatedFacing);
            if (rotatedGlueableSide != null)
                rotated = rotated.setValue(rotatedGlueableSide, true);
        }

        return rotated;
    }

    @Override
    public BlockState mirror(BlockState state, Mirror mirrorIn) {
        if (mirrorIn == Mirror.NONE)
            return state;

        BlockState mirrored = state;
        for (Direction face : Iterate.directions) {
            BooleanProperty glueableSide = getGlueableSide(mirrored, face);
            if (glueableSide != null)
                mirrored = mirrored.setValue(glueableSide, false);
        }

        for (Direction face : Iterate.directions) {
            BooleanProperty glueableSide = getGlueableSide(state, face);
            if (glueableSide == null || !state.getValue(glueableSide))
                continue;
            Direction mirroredFacing = mirrorIn.mirror(face);
            BooleanProperty mirroredGlueableSide = getGlueableSide(mirrored, mirroredFacing);
            if (mirroredGlueableSide != null)
                mirrored = mirrored.setValue(mirroredGlueableSide, true);
        }

        return mirrored;
    }

    @Override
    public BlockState transform(BlockState state, StructureTransform transform) {
        if (transform.mirror != null) {
            state = mirror(state, transform.mirror);
        }

        if (transform.rotationAxis == Direction.Axis.Y) {
            return rotate(state, transform.rotation);
        }
        return transformInner(state, transform);
    }

    protected BlockState transformInner(BlockState state, StructureTransform transform) {
        if (transform.rotation == Rotation.NONE)
            return state;

        BlockState rotated = state.setValue(AXIS, transform.rotateAxis(state.getValue(AXIS)));
        AbstractChassisBlock block = (AbstractChassisBlock) state.getBlock();

        for (Direction face : Iterate.directions) {
            BooleanProperty glueableSide = block.getGlueableSide(rotated, face);
            if (glueableSide != null)
                rotated = rotated.setValue(glueableSide, false);
        }

        for (Direction face : Iterate.directions) {
            BooleanProperty glueableSide = block.getGlueableSide(state, face);
            if (glueableSide == null || !state.getValue(glueableSide))
                continue;
            Direction rotatedFacing = transform.rotateFacing(face);
            BooleanProperty rotatedGlueableSide = block.getGlueableSide(rotated, rotatedFacing);
            if (rotatedGlueableSide != null)
                rotated = rotated.setValue(rotatedGlueableSide, true);
        }

        return rotated;
    }

    public abstract BooleanProperty getGlueableSide(BlockState state, Direction face);

    protected boolean glueAllowedOnSide(BlockGetter world, BlockPos pos, BlockState state, Direction side) {
        return true;
    }

    @Override
    public Class<ChassisBlockEntity> getBlockEntityClass() {
        return ChassisBlockEntity.class;
    }

    @Override
    public BlockEntityType<? extends ChassisBlockEntity> getBlockEntityType() {
        return AllBlockEntityTypes.CHASSIS;
    }

}
