package com.zurrtum.create.content.contraptions.glue;

import com.zurrtum.create.AllBlocks;
import com.zurrtum.create.AllEntityTypes;
import com.zurrtum.create.AllItems;
import com.zurrtum.create.AllSoundEvents;
import com.zurrtum.create.api.contraption.BlockMovementChecks;
import com.zurrtum.create.api.schematic.requirement.SpecialEntityItemRequirement;
import com.zurrtum.create.catnip.data.Iterate;
import com.zurrtum.create.catnip.math.VecHelper;
import com.zurrtum.create.content.contraptions.bearing.BearingBlock;
import com.zurrtum.create.content.contraptions.chassis.AbstractChassisBlock;
import com.zurrtum.create.content.kinetics.base.DirectionalKineticBlock;
import com.zurrtum.create.content.schematics.requirement.ItemRequirement;
import com.zurrtum.create.content.schematics.requirement.ItemRequirement.ItemUseType;
import com.zurrtum.create.foundation.codec.CreateCodecs;
import com.zurrtum.create.infrastructure.packet.s2c.SuperGlueSpawnPacket;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.core.Direction.Axis;
import net.minecraft.core.Direction.AxisDirection;
import net.minecraft.core.particles.ParticleTypes;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.network.protocol.Packet;
import net.minecraft.network.protocol.game.ClientGamePacketListener;
import net.minecraft.network.protocol.game.ClientboundAddEntityPacket;
import net.minecraft.network.syncher.SynchedEntityData;
import net.minecraft.server.level.ServerEntity;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.util.Mth;
import net.minecraft.world.InteractionHand;
import net.minecraft.world.InteractionResult;
import net.minecraft.world.damagesource.DamageSource;
import net.minecraft.world.entity.*;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.LevelAccessor;
import net.minecraft.world.level.block.Blocks;
import net.minecraft.world.level.block.DirectionalBlock;
import net.minecraft.world.level.block.Rotation;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.block.state.properties.BooleanProperty;
import net.minecraft.world.level.material.PushReaction;
import net.minecraft.world.level.storage.ValueInput;
import net.minecraft.world.level.storage.ValueOutput;
import net.minecraft.world.phys.AABB;
import net.minecraft.world.phys.Vec3;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

public class SuperGlueEntity extends Entity implements SpecialEntityItemRequirement {
    public static AABB span(BlockPos startPos, BlockPos endPos) {
        return new AABB(Vec3.atLowerCornerOf(startPos), Vec3.atLowerCornerOf(endPos)).expandTowards(1, 1, 1);
    }

    public static boolean isGlued(LevelAccessor level, BlockPos blockPos, Direction direction, Set<SuperGlueEntity> cached) {
        BlockPos targetPos = blockPos.relative(direction);
        if (cached != null)
            for (SuperGlueEntity glueEntity : cached)
                if (glueEntity.contains(blockPos) && glueEntity.contains(targetPos))
                    return true;
        for (SuperGlueEntity glueEntity : level.getEntitiesOfClass(SuperGlueEntity.class, span(blockPos, targetPos).inflate(16))) {
            if (!glueEntity.contains(blockPos) || !glueEntity.contains(targetPos))
                continue;
            if (cached != null)
                cached.add(glueEntity);
            return true;
        }
        return false;
    }

    public static List<SuperGlueEntity> collectCropped(Level level, AABB bb) {
        List<SuperGlueEntity> glue = new ArrayList<>();
        for (SuperGlueEntity glueEntity : level.getEntitiesOfClass(SuperGlueEntity.class, bb)) {
            AABB glueBox = glueEntity.getBoundingBox();
            AABB intersect = bb.intersect(glueBox);
            if (intersect.getXsize() * intersect.getYsize() * intersect.getZsize() == 0)
                continue;
            if (Mth.equal(intersect.getSize(), 1))
                continue;
            glue.add(new SuperGlueEntity(level, intersect));
        }
        return glue;
    }

    public SuperGlueEntity(EntityType<? extends SuperGlueEntity> type, Level world) {
        super(type, world);
    }

    public SuperGlueEntity(Level world, AABB boundingBox) {
        this(AllEntityTypes.SUPER_GLUE, world);
        setBoundingBox(boundingBox);
        resetPositionToBB();
    }

    public void resetPositionToBB() {
        AABB bb = getBoundingBox();
        setPosRaw(bb.getCenter().x, bb.minY, bb.getCenter().z);
    }

    @Override
    protected void defineSynchedData(SynchedEntityData.Builder builder) {
    }

    public static boolean isValidFace(Level world, BlockPos pos, Direction direction) {
        BlockState state = world.getBlockState(pos);
        if (BlockMovementChecks.isBlockAttachedTowards(state, world, pos, direction))
            return true;
        if (!BlockMovementChecks.isMovementNecessary(state, world, pos))
            return false;
        return !BlockMovementChecks.isNotSupportive(state, direction);
    }

    public static boolean isSideSticky(Level world, BlockPos pos, Direction direction) {
        BlockState state = world.getBlockState(pos);
        if (state.is(AllBlocks.STICKY_MECHANICAL_PISTON))
            return state.getValue(DirectionalKineticBlock.FACING) == direction;

        if (state.is(AllBlocks.STICKER))
            return state.getValue(DirectionalBlock.FACING) == direction;

        if (state.getBlock() == Blocks.SLIME_BLOCK)
            return true;
        if (state.getBlock() == Blocks.HONEY_BLOCK)
            return true;

        if (state.is(AllBlocks.CART_ASSEMBLER))
            return Direction.UP == direction;

        if (state.is(AllBlocks.GANTRY_CARRIAGE))
            return state.getValue(DirectionalKineticBlock.FACING) == direction;

        if (state.getBlock() instanceof BearingBlock) {
            return state.getValue(DirectionalKineticBlock.FACING) == direction;
        }

        if (state.getBlock() instanceof AbstractChassisBlock) {
            BooleanProperty glueableSide = ((AbstractChassisBlock) state.getBlock()).getGlueableSide(state, direction);
            if (glueableSide == null)
                return false;
            return state.getValue(glueableSide);
        }

        return false;
    }

    @Override
    public boolean hurtServer(ServerLevel world, DamageSource source, float amount) {
        return false;
    }

    @Override
    public void tick() {
        xRotO = getXRot();
        yRotO = getYRot();
        //        walkDistO = walkDist;
        xo = getX();
        yo = getY();
        zo = getZ();

        if (getBoundingBox().getXsize() == 0)
            discard();
    }

    @Override
    public void setPos(double x, double y, double z) {
        AABB bb = getBoundingBox();
        setPosRaw(x, y, z);
        Vec3 center = bb.getCenter();
        setBoundingBox(bb.move(-center.x, -bb.minY, -center.z).move(x, y, z));
    }

    @Override
    public void move(MoverType typeIn, Vec3 pos) {
        if (!level().isClientSide() && isAlive() && pos.lengthSqr() > 0.0D)
            discard();
    }

    @Override
    public void push(double x, double y, double z) {
        if (!level().isClientSide() && isAlive() && x * x + y * y + z * z > 0.0D)
            discard();
    }

    @Override
    public EntityDimensions getDimensions(Pose pose) {
        return super.getDimensions(pose).withEyeHeight(0.0F);
    }

    public void playPlaceSound() {
        AllSoundEvents.SLIME_ADDED.playFrom(this, 0.5F, 0.75F);
    }

    @Override
    public InteractionResult interact(Player player, InteractionHand hand) {
        return InteractionResult.PASS;
    }

    @Override
    public void addAdditionalSaveData(ValueOutput view) {
        AABB box = getBoundingBox().move(position().scale(-1));
        view.store("Box", CreateCodecs.BOX_CODEC, box);
    }

    @Override
    public void readAdditionalSaveData(ValueInput view) {
        AABB box = view.read("Box", CreateCodecs.BOX_CODEC).orElseThrow().move(position());
        setBoundingBox(box);
    }

    public static void writeBoundingBox(CompoundTag compound, AABB bb) {
        compound.put("From", VecHelper.writeNBT(new Vec3(bb.minX, bb.minY, bb.minZ)));
        compound.put("To", VecHelper.writeNBT(new Vec3(bb.maxX, bb.maxY, bb.maxZ)));
    }

    public static AABB readBoundingBox(CompoundTag compound) {
        Vec3 from = VecHelper.readNBT(compound.getListOrEmpty("From"));
        Vec3 to = VecHelper.readNBT(compound.getListOrEmpty("To"));
        return new AABB(from, to);
    }

    @Override
    public Packet<ClientGamePacketListener> getAddEntityPacket(ServerEntity entityTrackerEntry) {
        return new SuperGlueSpawnPacket(this, entityTrackerEntry);
    }

    @Override
    public void recreateFromPacket(ClientboundAddEntityPacket packet) {
        super.recreateFromPacket(packet);
        setBoundingBox(((SuperGlueSpawnPacket) packet).getBox());
    }

    @Override
    protected boolean repositionEntityAfterLoad() {
        return false;
    }

    @Override
    public float rotate(Rotation transformRotation) {
        AABB bb = getBoundingBox().move(position().scale(-1));
        if (transformRotation == Rotation.CLOCKWISE_90 || transformRotation == Rotation.COUNTERCLOCKWISE_90)
            setBoundingBox(new AABB(bb.minZ, bb.minY, bb.minX, bb.maxZ, bb.maxY, bb.maxX).move(position()));
        return super.rotate(transformRotation);
    }

    @Override
    public void thunderHit(ServerLevel world, LightningBolt lightningBolt) {
    }

    @Override
    public void refreshDimensions() {
    }

    @Override
    public ItemRequirement getRequiredItems() {
        return new ItemRequirement(ItemUseType.DAMAGE, AllItems.SUPER_GLUE);
    }

    @Override
    public boolean isIgnoringBlockTriggers() {
        return true;
    }

    public boolean contains(BlockPos pos) {
        return getBoundingBox().contains(Vec3.atCenterOf(pos));
    }

    @Override
    public PushReaction getPistonPushReaction() {
        return PushReaction.IGNORE;
    }

    public void spawnParticles() {
        AABB bb = getBoundingBox();
        Vec3 origin = new Vec3(bb.minX, bb.minY, bb.minZ);
        Vec3 extents = new Vec3(bb.getXsize(), bb.getYsize(), bb.getZsize());

        if (!(level() instanceof ServerLevel slevel))
            return;

        for (Axis axis : Iterate.axes) {
            AxisDirection positive = AxisDirection.POSITIVE;
            double max = axis.choose(extents.x, extents.y, extents.z);
            Vec3 normal = Vec3.atLowerCornerOf(Direction.fromAxisAndDirection(axis, positive).getUnitVec3i());
            for (Axis axis2 : Iterate.axes) {
                if (axis2 == axis)
                    continue;
                double max2 = axis2.choose(extents.x, extents.y, extents.z);
                Vec3 normal2 = Vec3.atLowerCornerOf(Direction.fromAxisAndDirection(axis2, positive).getUnitVec3i());
                for (Axis axis3 : Iterate.axes) {
                    if (axis3 == axis2 || axis3 == axis)
                        continue;
                    double max3 = axis3.choose(extents.x, extents.y, extents.z);
                    Vec3 normal3 = Vec3.atLowerCornerOf(Direction.fromAxisAndDirection(axis3, positive).getUnitVec3i());

                    for (int i = 0; i <= max * 2; i++) {
                        for (int o1 : Iterate.zeroAndOne) {
                            for (int o2 : Iterate.zeroAndOne) {
                                Vec3 v = origin.add(normal.scale(i / 2f)).add(normal2.scale(max2 * o1)).add(normal3.scale(max3 * o2));

                                slevel.sendParticles(ParticleTypes.ITEM_SLIME, v.x, v.y, v.z, 1, 0, 0, 0, 0);

                            }
                        }
                    }
                    break;
                }
                break;
            }
        }
    }
}
