package com.zurrtum.create.content.contraptions.pulley;

import com.zurrtum.create.AllAdvancements;
import com.zurrtum.create.AllBlockEntityTypes;
import com.zurrtum.create.AllBlocks;
import com.zurrtum.create.api.contraption.BlockMovementChecks;
import com.zurrtum.create.catnip.data.Iterate;
import com.zurrtum.create.content.contraptions.AbstractContraptionEntity;
import com.zurrtum.create.content.contraptions.AssemblyException;
import com.zurrtum.create.content.contraptions.ContraptionCollider;
import com.zurrtum.create.content.contraptions.ControlledContraptionEntity;
import com.zurrtum.create.content.contraptions.piston.LinearActuatorBlockEntity;
import com.zurrtum.create.content.redstone.thresholdSwitch.ThresholdSwitchObservable;
import com.zurrtum.create.foundation.advancement.CreateTrigger;
import com.zurrtum.create.foundation.codec.CreateCodecs;
import com.zurrtum.create.infrastructure.config.AllConfigs;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.MutableComponent;
import net.minecraft.util.Mth;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.block.entity.BlockEntityType;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.block.state.properties.BlockStateProperties;
import net.minecraft.world.level.material.FluidState;
import net.minecraft.world.level.material.Fluids;
import net.minecraft.world.level.storage.ValueInput;
import net.minecraft.world.level.storage.ValueOutput;
import net.minecraft.world.phys.AABB;
import net.minecraft.world.phys.Vec3;
import org.jetbrains.annotations.Nullable;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.List;

public class PulleyBlockEntity extends LinearActuatorBlockEntity implements ThresholdSwitchObservable {

    protected int initialOffset;
    private float prevAnimatedOffset;

    protected BlockPos mirrorParent;
    protected List<BlockPos> mirrorChildren;
    public WeakReference<AbstractContraptionEntity> sharedMirrorContraption;

    public PulleyBlockEntity(BlockEntityType<?> type, BlockPos pos, BlockState state) {
        super(type, pos, state);
    }

    public PulleyBlockEntity(BlockPos pos, BlockState state) {
        super(AllBlockEntityTypes.ROPE_PULLEY, pos, state);
    }

    @Override
    protected AABB createRenderBoundingBox() {
        double expandY = -offset;
        if (sharedMirrorContraption != null) {
            AbstractContraptionEntity ace = sharedMirrorContraption.get();
            if (ace != null)
                expandY = ace.getY() - worldPosition.getY();
        }
        return super.createRenderBoundingBox().inflate(0, expandY, 0);
    }

    @Override
    public List<CreateTrigger> getAwardables() {
        return List.of(AllAdvancements.PULLEY_MAXED);
    }

    @Override
    public void tick() {
        float prevOffset = offset;
        super.tick();

        if (level.isClientSide() && mirrorParent != null)
            if (sharedMirrorContraption == null || sharedMirrorContraption.get() == null || !sharedMirrorContraption.get().isAlive()) {
                sharedMirrorContraption = null;
                if (level.getBlockEntity(mirrorParent) instanceof PulleyBlockEntity pte && pte.movedContraption != null)
                    sharedMirrorContraption = new WeakReference<>(pte.movedContraption);
            }

        if (isVirtual())
            prevAnimatedOffset = offset;
        invalidateRenderBoundingBox();

        if (prevOffset < 200 && offset >= 200)
            award(AllAdvancements.PULLEY_MAXED);
    }

    @Override
    protected boolean isPassive() {
        return mirrorParent != null;
    }

    @Nullable
    public AbstractContraptionEntity getAttachedContraption() {
        return mirrorParent != null && sharedMirrorContraption != null ? sharedMirrorContraption.get() : movedContraption;
    }

    @Override
    protected void assemble() throws AssemblyException {
        if (!(level.getBlockState(worldPosition).getBlock() instanceof PulleyBlock))
            return;
        if (speed == 0 && mirrorParent == null)
            return;
        int maxLength = AllConfigs.server().kinetics.maxRopeLength.get();
        int i = 1;
        while (i <= maxLength) {
            BlockPos ropePos = worldPosition.below(i);
            BlockState ropeState = level.getBlockState(ropePos);
            if (!ropeState.is(AllBlocks.ROPE) && !ropeState.is(AllBlocks.PULLEY_MAGNET)) {
                break;
            }
            ++i;
        }
        offset = i - 1;
        if (offset >= getExtensionRange() && getSpeed() > 0)
            return;
        if (offset <= 0 && getSpeed() < 0)
            return;

        // Collect Construct
        if (!level.isClientSide() && mirrorParent == null) {
            needsContraption = false;
            BlockPos anchor = worldPosition.below(Mth.floor(offset + 1));
            initialOffset = Mth.floor(offset);
            PulleyContraption contraption = new PulleyContraption(initialOffset);
            boolean canAssembleStructure = contraption.assemble(level, anchor);

            if (canAssembleStructure) {
                Direction movementDirection = getSpeed() > 0 ? Direction.DOWN : Direction.UP;
                if (ContraptionCollider.isCollidingWithWorld(level, contraption, anchor.relative(movementDirection), movementDirection))
                    canAssembleStructure = false;
            }

            if (!canAssembleStructure && getSpeed() > 0)
                return;

            removeRopes();

            if (!contraption.getBlocks().isEmpty()) {
                contraption.removeBlocksFromWorld(level, BlockPos.ZERO);
                movedContraption = ControlledContraptionEntity.create(level, this, contraption);
                movedContraption.setPos(anchor.getX(), anchor.getY(), anchor.getZ());
                level.addFreshEntity(movedContraption);
                forceMove = true;
                needsContraption = true;

                if (contraption.containsBlockBreakers())
                    award(AllAdvancements.CONTRAPTION_ACTORS);

                for (BlockPos pos : contraption.createColliders(level, Direction.UP)) {
                    if (pos.getY() != 0)
                        continue;
                    pos = pos.offset(anchor);
                    if (level.getBlockEntity(new BlockPos(pos.getX(), this.worldPosition.getY(), pos.getZ())) instanceof PulleyBlockEntity pbe)
                        pbe.startMirroringOther(this.worldPosition);
                }
            }
        }

        if (mirrorParent != null)
            removeRopes();

        clientOffsetDiff = 0;
        running = true;
        sendData();
    }

    private void removeRopes() {
        for (int i = ((int) offset); i > 0; i--) {
            BlockPos offset = worldPosition.below(i);
            BlockState oldState = level.getBlockState(offset);
            level.setBlock(offset, oldState.getFluidState().createLegacyBlock(), Block.UPDATE_CLIENTS | Block.UPDATE_MOVE_BY_PISTON);
        }
    }

    @Override
    public void disassemble() {
        if (!running && movedContraption == null && mirrorParent == null)
            return;
        offset = getGridOffset(offset);
        if (movedContraption != null)
            resetContraptionToOffset();

        if (!level.isClientSide()) {
            if (shouldCreateRopes()) {
                if (offset > 0) {
                    BlockPos magnetPos = worldPosition.below((int) offset);
                    FluidState ifluidstate = level.getFluidState(magnetPos);
                    if (level.getBlockState(magnetPos).getDestroySpeed(level, magnetPos) != -1) {

                        level.destroyBlock(magnetPos, level.getBlockState(magnetPos).getCollisionShape(level, magnetPos).isEmpty());
                        level.setBlock(
                            magnetPos,
                            AllBlocks.PULLEY_MAGNET.defaultBlockState()
                                .setValue(BlockStateProperties.WATERLOGGED, Boolean.valueOf(ifluidstate.getType() == Fluids.WATER)),
                            Block.UPDATE_CLIENTS | Block.UPDATE_MOVE_BY_PISTON
                        );
                    }
                }

                boolean[] waterlog = new boolean[(int) offset];

                for (boolean destroyPass : Iterate.trueAndFalse) {
                    for (int i = 1; i <= ((int) offset) - 1; i++) {
                        BlockPos ropePos = worldPosition.below(i);
                        if (level.getBlockState(ropePos).getDestroySpeed(level, ropePos) == -1)
                            continue;

                        if (destroyPass) {
                            FluidState ifluidstate = level.getFluidState(ropePos);
                            waterlog[i] = ifluidstate.getType() == Fluids.WATER;
                            level.destroyBlock(ropePos, level.getBlockState(ropePos).getCollisionShape(level, ropePos).isEmpty());
                            continue;
                        }

                        level.setBlock(
                            worldPosition.below(i),
                            AllBlocks.ROPE.defaultBlockState().setValue(BlockStateProperties.WATERLOGGED, waterlog[i]),
                            Block.UPDATE_CLIENTS | Block.UPDATE_MOVE_BY_PISTON
                        );
                    }
                }

            }

            if (movedContraption != null && mirrorParent == null)
                movedContraption.disassemble();
            notifyMirrorsOfDisassembly();
        }

        if (movedContraption != null)
            movedContraption.discard();

        movedContraption = null;
        initialOffset = 0;
        running = false;
        sendData();
    }

    protected boolean shouldCreateRopes() {
        return !remove;
    }

    @Override
    protected Vec3 toPosition(float offset) {
        if (movedContraption.getContraption() instanceof PulleyContraption contraption) {
            return Vec3.atLowerCornerOf(contraption.anchor).add(0, contraption.getInitialOffset() - offset, 0);

        }
        return Vec3.ZERO;
    }

    @Override
    protected void visitNewPosition() {
        super.visitNewPosition();
        if (level.isClientSide())
            return;
        if (movedContraption != null)
            return;
        if (getSpeed() <= 0)
            return;

        BlockPos posBelow = worldPosition.below((int) (offset + getMovementSpeed()) + 1);
        BlockState state = level.getBlockState(posBelow);
        if (!BlockMovementChecks.isMovementNecessary(state, level, posBelow))
            return;
        if (BlockMovementChecks.isBrittle(state))
            return;

        disassemble();
        assembleNextTick = true;
    }

    @Override
    protected void read(ValueInput view, boolean clientPacket) {
        initialOffset = view.getIntOr("InitialOffset", 0);
        needsContraption = view.getBooleanOr("NeedsContraption", false);
        super.read(view, clientPacket);

        BlockPos prevMirrorParent = mirrorParent;
        mirrorParent = view.read("MirrorParent", BlockPos.CODEC).orElse(null);
        mirrorChildren = view.read("MirrorChildren", CreateCodecs.BLOCK_POS_LIST_CODEC).map(ArrayList::new).orElse(null);

        if (mirrorParent != null) {
            offset = 0;
            if (prevMirrorParent == null || !prevMirrorParent.equals(mirrorParent))
                sharedMirrorContraption = null;
        }

        if (mirrorParent == null)
            sharedMirrorContraption = null;
    }

    @Override
    public void write(ValueOutput view, boolean clientPacket) {
        view.putInt("InitialOffset", initialOffset);
        super.write(view, clientPacket);

        if (mirrorParent != null)
            view.store("MirrorParent", BlockPos.CODEC, mirrorParent);
        if (mirrorChildren != null)
            view.store("MirrorChildren", CreateCodecs.BLOCK_POS_LIST_CODEC, mirrorChildren);
    }

    public void startMirroringOther(BlockPos parent) {
        if (parent.equals(worldPosition))
            return;
        if (!(level.getBlockEntity(parent) instanceof PulleyBlockEntity pbe))
            return;
        if (pbe.getType() != getType())
            return;
        if (pbe.mirrorChildren == null)
            pbe.mirrorChildren = new ArrayList<>();
        pbe.mirrorChildren.add(worldPosition);
        pbe.notifyUpdate();

        mirrorParent = parent;
        try {
            assemble();
        } catch (AssemblyException e) {
        }
        notifyUpdate();
    }

    public void notifyMirrorsOfDisassembly() {
        if (mirrorChildren == null)
            return;
        for (BlockPos blockPos : mirrorChildren) {
            if (!(level.getBlockEntity(blockPos) instanceof PulleyBlockEntity pbe))
                continue;
            pbe.offset = offset;
            pbe.disassemble();
            pbe.mirrorParent = null;
            pbe.notifyUpdate();
        }
        mirrorChildren.clear();
        notifyUpdate();
    }

    @Override
    protected int getExtensionRange() {
        return Math.max(0, Math.min(AllConfigs.server().kinetics.maxRopeLength.get(), (worldPosition.getY() - 1) - level.getMinY()));
    }

    @Override
    protected int getInitialOffset() {
        return initialOffset;
    }

    @Override
    protected Vec3 toMotionVector(float speed) {
        return new Vec3(0, -speed, 0);
    }

    @Override
    public float getInterpolatedOffset(float partialTicks) {
        if (isVirtual())
            return Mth.lerp(partialTicks, prevAnimatedOffset, offset);
        boolean moving = running && (movedContraption == null || !movedContraption.isStalled());
        return super.getInterpolatedOffset(moving ? partialTicks : 0.5f);
    }

    public void animateOffset(float forcedOffset) {
        offset = forcedOffset;
    }

    public BlockPos getMirrorParent() {
        return mirrorParent;
    }

    // Threshold switch

    @Override
    public int getCurrentValue() {
        return worldPosition.getY() - (int) getInterpolatedOffset(.5f);
    }

    @Override
    public int getMinValue() {
        return level.getMinY();
    }

    @Override
    public int getMaxValue() {
        return worldPosition.getY();
    }

    @Override
    public MutableComponent format(int value) {
        return Component.translatable("create.gui.threshold_switch.pulley_y_level", value);
    }

}
