package com.zurrtum.create.content.kinetics.belt;

import com.zurrtum.create.AllBlockEntityTypes;
import com.zurrtum.create.AllBlocks;
import com.zurrtum.create.AllItemTags;
import com.zurrtum.create.AllItems;
import com.zurrtum.create.api.contraption.transformable.TransformableBlock;
import com.zurrtum.create.api.schematic.requirement.SpecialBlockItemRequirement;
import com.zurrtum.create.catnip.data.Iterate;
import com.zurrtum.create.catnip.math.VecHelper;
import com.zurrtum.create.content.contraptions.StructureTransform;
import com.zurrtum.create.content.equipment.armor.DivingBootsItem;
import com.zurrtum.create.content.fluids.transfer.GenericItemEmptying;
import com.zurrtum.create.content.kinetics.base.HorizontalKineticBlock;
import com.zurrtum.create.content.kinetics.base.KineticBlockEntity;
import com.zurrtum.create.content.kinetics.belt.BeltBlockEntity.CasingType;
import com.zurrtum.create.content.kinetics.belt.BeltSlicer.Feedback;
import com.zurrtum.create.content.kinetics.belt.behaviour.TransportedItemStackHandlerBehaviour.TransportedResult;
import com.zurrtum.create.content.kinetics.belt.transport.BeltMovementHandler.TransportedEntityInfo;
import com.zurrtum.create.content.kinetics.belt.transport.BeltTunnelInteractionHandler;
import com.zurrtum.create.content.logistics.box.PackageEntity;
import com.zurrtum.create.content.logistics.box.PackageItem;
import com.zurrtum.create.content.logistics.funnel.FunnelBlock;
import com.zurrtum.create.content.logistics.tunnel.BeltTunnelBlock;
import com.zurrtum.create.content.schematics.requirement.ItemRequirement;
import com.zurrtum.create.content.schematics.requirement.ItemRequirement.ItemUseType;
import com.zurrtum.create.foundation.block.IBE;
import com.zurrtum.create.foundation.block.ProperWaterloggedBlock;
import com.zurrtum.create.foundation.item.ItemHelper;
import com.zurrtum.create.infrastructure.items.ItemInventoryProvider;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.core.Direction.Axis;
import net.minecraft.core.Direction.AxisDirection;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.sounds.SoundEvents;
import net.minecraft.sounds.SoundSource;
import net.minecraft.util.RandomSource;
import net.minecraft.world.Container;
import net.minecraft.world.InteractionHand;
import net.minecraft.world.InteractionResult;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.EquipmentSlot;
import net.minecraft.world.entity.InsideBlockEffectApplier;
import net.minecraft.world.entity.item.ItemEntity;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.context.UseOnContext;
import net.minecraft.world.level.*;
import net.minecraft.world.level.block.*;
import net.minecraft.world.level.block.entity.BlockEntity;
import net.minecraft.world.level.block.entity.BlockEntityType;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.block.state.StateDefinition.Builder;
import net.minecraft.world.level.block.state.properties.BlockStateProperties;
import net.minecraft.world.level.block.state.properties.BooleanProperty;
import net.minecraft.world.level.block.state.properties.EnumProperty;
import net.minecraft.world.level.levelgen.DebugLevelSource;
import net.minecraft.world.level.material.FluidState;
import net.minecraft.world.level.material.Fluids;
import net.minecraft.world.level.pathfinder.PathComputationType;
import net.minecraft.world.level.storage.loot.LootParams;
import net.minecraft.world.level.storage.loot.parameters.LootContextParams;
import net.minecraft.world.phys.AABB;
import net.minecraft.world.phys.BlockHitResult;
import net.minecraft.world.phys.Vec3;
import net.minecraft.world.phys.shapes.CollisionContext;
import net.minecraft.world.phys.shapes.EntityCollisionContext;
import net.minecraft.world.phys.shapes.Shapes;
import net.minecraft.world.phys.shapes.VoxelShape;
import org.apache.commons.lang3.mutable.MutableBoolean;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

public class BeltBlock extends HorizontalKineticBlock implements IBE<BeltBlockEntity>, SpecialBlockItemRequirement, TransformableBlock, ProperWaterloggedBlock, ItemInventoryProvider<BeltBlockEntity> {

    public static final EnumProperty<BeltSlope> SLOPE = EnumProperty.create("slope", BeltSlope.class);
    public static final EnumProperty<BeltPart> PART = EnumProperty.create("part", BeltPart.class);
    public static final BooleanProperty CASING = BooleanProperty.create("casing");

    public BeltBlock(Properties properties) {
        super(properties);
        registerDefaultState(defaultBlockState().setValue(SLOPE, BeltSlope.HORIZONTAL).setValue(PART, BeltPart.START).setValue(CASING, false)
            .setValue(WATERLOGGED, false));
    }

    @Override
    public Container getInventory(LevelAccessor world, BlockPos pos, BlockState state, BeltBlockEntity blockEntity, Direction context) {
        if (!BeltBlock.canTransportObjects(blockEntity.getBlockState()))
            return null;
        if (!blockEntity.isRemoved() && blockEntity.itemHandler == null)
            blockEntity.initializeItemHandler();
        return blockEntity.itemHandler;
    }

    @Override
    protected boolean areStatesKineticallyEquivalent(BlockState oldState, BlockState newState) {
        return super.areStatesKineticallyEquivalent(oldState, newState) && oldState.getValue(PART) == newState.getValue(PART);
    }

    @Override
    public boolean hasShaftTowards(LevelReader world, BlockPos pos, BlockState state, Direction face) {
        if (face.getAxis() != getRotationAxis(state))
            return false;
        return getBlockEntityOptional(world, pos).map(BeltBlockEntity::hasPulley).orElse(false);
    }

    @Override
    public Axis getRotationAxis(BlockState state) {
        if (state.getValue(SLOPE) == BeltSlope.SIDEWAYS)
            return Axis.Y;
        return state.getValue(HORIZONTAL_FACING).getClockWise().getAxis();
    }

    public ItemStack getCloneItemStack(LevelReader world, BlockPos pos, BlockState state, boolean includeData) {
        return AllItems.BELT_CONNECTOR.getDefaultInstance();
    }

    @Override
    public List<ItemStack> getDrops(BlockState state, LootParams.Builder builder) {
        List<ItemStack> drops = super.getDrops(state, builder);
        BlockEntity blockEntity = builder.getOptionalParameter(LootContextParams.BLOCK_ENTITY);
        if (blockEntity instanceof BeltBlockEntity && ((BeltBlockEntity) blockEntity).hasPulley())
            drops.addAll(AllBlocks.SHAFT.defaultBlockState().getDrops(builder));
        return drops;
    }

    @Override
    public void spawnAfterBreak(BlockState state, ServerLevel worldIn, BlockPos pos, ItemStack p_220062_4_, boolean b) {
        BeltBlockEntity controllerBE = BeltHelper.getControllerBE(worldIn, pos);
        if (controllerBE != null)
            controllerBE.getInventory().ejectAll();
    }

    @Override
    public void updateEntityMovementAfterFallOn(BlockGetter worldIn, Entity entityIn) {
        super.updateEntityMovementAfterFallOn(worldIn, entityIn);
        BlockPos entityPosition = entityIn.blockPosition();
        BlockPos beltPos = null;

        if (worldIn.getBlockState(entityPosition).is(AllBlocks.BELT))
            beltPos = entityPosition;
        else if (worldIn.getBlockState(entityPosition.below()).is(AllBlocks.BELT))
            beltPos = entityPosition.below();
        if (beltPos == null)
            return;
        if (!(worldIn instanceof Level world))
            return;

        entityInside(worldIn.getBlockState(beltPos), world, beltPos, entityIn, InsideBlockEffectApplier.NOOP, false);
    }

    @Override
    public void entityInside(BlockState state, Level worldIn, BlockPos pos, Entity entityIn, InsideBlockEffectApplier handler, boolean bl) {
        if (!canTransportObjects(state))
            return;
        if (entityIn instanceof Player player) {
            if (player.isShiftKeyDown() && !player.getItemBySlot(EquipmentSlot.FEET).is(AllItems.CARDBOARD_BOOTS))
                return;
            if (player.getAbilities().flying)
                return;
        }

        if (DivingBootsItem.isWornBy(entityIn))
            return;

        BeltBlockEntity belt = BeltHelper.getSegmentBE(worldIn, pos);
        if (belt == null)
            return;
        ItemStack asItem = ItemHelper.fromItemEntity(entityIn);
        if (!asItem.isEmpty()) {
            if (worldIn.isClientSide())
                return;
            if (entityIn.getDeltaMovement().y > 0)
                return;
            Vec3 targetLocation = VecHelper.getCenterOf(pos).add(0, 5 / 16f, 0);
            if (!PackageEntity.centerPackage(entityIn, targetLocation))
                return;
            if (BeltTunnelInteractionHandler.getTunnelOnPosition(worldIn, pos) != null)
                return;
            withBlockEntityDo(
                worldIn, pos, be -> {
                    Container inventory = ItemHelper.getInventory(worldIn, pos, state, be, null);
                    if (inventory == null)
                        return;
                    int insert = inventory.insert(asItem);
                    if (asItem.getCount() == insert) {
                        entityIn.discard();
                    } else if (entityIn instanceof ItemEntity itemEntity && insert != 0) {
                        asItem.shrink(insert);
                        itemEntity.setItem(asItem);
                    }
                }
            );
            return;
        }

        BeltBlockEntity controller = BeltHelper.getControllerBE(worldIn, pos);
        if (controller == null || controller.passengers == null)
            return;
        if (controller.passengers.containsKey(entityIn)) {
            TransportedEntityInfo info = controller.passengers.get(entityIn);
            if (info.getTicksSinceLastCollision() != 0 || pos.equals(entityIn.blockPosition()))
                info.refresh(pos, state);
        } else {
            controller.passengers.put(entityIn, new TransportedEntityInfo(pos, state));
            entityIn.setOnGround(true);
        }
    }

    public static boolean canTransportObjects(BlockState state) {
        if (!state.is(AllBlocks.BELT))
            return false;
        BeltSlope slope = state.getValue(SLOPE);
        return slope != BeltSlope.VERTICAL && slope != BeltSlope.SIDEWAYS;
    }

    @Override
    protected InteractionResult useItemOn(
        ItemStack stack,
        BlockState state,
        Level level,
        BlockPos pos,
        Player player,
        InteractionHand hand,
        BlockHitResult hitResult
    ) {
        if (player.isShiftKeyDown() || !player.mayBuild())
            return InteractionResult.TRY_WITH_EMPTY_HAND;
        boolean isWrench = stack.is(AllItems.WRENCH);
        boolean isConnector = stack.is(AllItems.BELT_CONNECTOR);
        boolean isShaft = stack.is(AllItems.SHAFT);
        boolean isDye = stack.is(AllItemTags.DYES);
        boolean hasWater = !stack.isEmpty() && GenericItemEmptying.emptyItem(level, stack, true).getFirst().getFluid().isSame(Fluids.WATER);
        boolean isHand = stack.isEmpty() && hand == InteractionHand.MAIN_HAND;

        if (isDye || hasWater)
            return onBlockEntityUseItemOn(
                level,
                pos,
                be -> be.applyColor(AllItemTags.getDyeColor(stack)) ? InteractionResult.SUCCESS : InteractionResult.TRY_WITH_EMPTY_HAND
            );

        if (isConnector)
            return BeltSlicer.useConnector(state, level, pos, player, hand, hitResult, new Feedback());
        if (isWrench)
            return BeltSlicer.useWrench(state, level, pos, player, hand, hitResult, new Feedback());

        BeltBlockEntity belt = BeltHelper.getSegmentBE(level, pos);
        if (belt == null)
            return InteractionResult.TRY_WITH_EMPTY_HAND;

        if (PackageItem.isPackage(stack)) {
            ItemStack toInsert = stack.copy();
            Container handler = ItemHelper.getInventory(level, belt.getBlockPos(), null);
            if (handler == null)
                return InteractionResult.TRY_WITH_EMPTY_HAND;
            int insert = handler.insert(toInsert);
            if (insert != 0) {
                stack.shrink(insert);
                return InteractionResult.SUCCESS;
            }
        }

        if (isHand) {
            BeltBlockEntity controllerBelt = belt.getControllerBE();
            if (controllerBelt == null)
                return InteractionResult.TRY_WITH_EMPTY_HAND;
            if (level.isClientSide())
                return InteractionResult.SUCCESS;
            MutableBoolean success = new MutableBoolean(false);
            controllerBelt.getInventory().applyToEachWithin(
                belt.index + .5f, .55f, (transportedItemStack) -> {
                    player.getInventory().placeItemBackInInventory(transportedItemStack.stack);
                    success.setTrue();
                    return TransportedResult.removeItem();
                }
            );
            if (success.isTrue())
                level.playSound(null, pos, SoundEvents.ITEM_PICKUP, SoundSource.PLAYERS, .2f, 1f + level.random.nextFloat());
        }

        if (isShaft) {
            if (state.getValue(PART) != BeltPart.MIDDLE)
                return InteractionResult.TRY_WITH_EMPTY_HAND;
            if (level.isClientSide())
                return InteractionResult.SUCCESS;
            if (!player.isCreative())
                stack.shrink(1);
            KineticBlockEntity.switchToBlockState(level, pos, state.setValue(PART, BeltPart.PULLEY));
            return InteractionResult.SUCCESS;
        }

        if (stack.is(AllItems.BRASS_CASING)) {
            withBlockEntityDo(level, pos, be -> be.setCasingType(CasingType.BRASS));
            updateCoverProperty(level, pos, level.getBlockState(pos));

            SoundType soundType = AllBlocks.BRASS_CASING.defaultBlockState().getSoundType();
            level.playSound(
                null,
                pos,
                soundType.getPlaceSound(),
                SoundSource.BLOCKS,
                (soundType.getVolume() + 1.0F) / 2.0F,
                soundType.getPitch() * 0.8F
            );

            return InteractionResult.SUCCESS;
        }

        if (stack.is(AllItems.ANDESITE_CASING)) {
            withBlockEntityDo(level, pos, be -> be.setCasingType(CasingType.ANDESITE));
            updateCoverProperty(level, pos, level.getBlockState(pos));

            SoundType soundType = AllBlocks.ANDESITE_CASING.defaultBlockState().getSoundType();
            level.playSound(
                null,
                pos,
                soundType.getPlaceSound(),
                SoundSource.BLOCKS,
                (soundType.getVolume() + 1.0F) / 2.0F,
                soundType.getPitch() * 0.8F
            );

            return InteractionResult.SUCCESS;
        }

        return InteractionResult.TRY_WITH_EMPTY_HAND;
    }

    @Override
    public InteractionResult onWrenched(BlockState state, UseOnContext context) {
        Level world = context.getLevel();
        Player player = context.getPlayer();
        BlockPos pos = context.getClickedPos();

        if (state.getValue(CASING)) {
            if (world.isClientSide())
                return InteractionResult.SUCCESS;
            withBlockEntityDo(world, pos, be -> be.setCasingType(CasingType.NONE));
            return InteractionResult.SUCCESS;
        }

        if (state.getValue(PART) == BeltPart.PULLEY) {
            if (world.isClientSide())
                return InteractionResult.SUCCESS;
            KineticBlockEntity.switchToBlockState(world, pos, state.setValue(PART, BeltPart.MIDDLE));
            if (player != null && !player.isCreative())
                player.getInventory().placeItemBackInInventory(AllItems.SHAFT.getDefaultInstance());
            return InteractionResult.SUCCESS;
        }

        return InteractionResult.PASS;
    }

    @Override
    protected void createBlockStateDefinition(Builder<Block, BlockState> builder) {
        builder.add(SLOPE, PART, CASING, WATERLOGGED);
        super.createBlockStateDefinition(builder);
    }

    //TODO
    //    @Override
    //    public PathType getBlockPathType(BlockState state, BlockView world, BlockPos pos, Mob entity) {
    //        return PathType.RAIL;
    //    }

    @Override
    public VoxelShape getShape(BlockState state, BlockGetter worldIn, BlockPos pos, CollisionContext context) {
        return BeltShapes.getShape(state);
    }

    @Override
    public VoxelShape getCollisionShape(BlockState state, BlockGetter worldIn, BlockPos pos, CollisionContext context) {
        if (state.getBlock() != this)
            return Shapes.empty();

        VoxelShape shape = getShape(state, worldIn, pos, context);
        if (!(context instanceof EntityCollisionContext))
            return shape;

        return getBlockEntityOptional(worldIn, pos).map(be -> {
            Entity entity = ((EntityCollisionContext) context).getEntity();
            if (entity == null)
                return shape;

            BeltBlockEntity controller = be.getControllerBE();
            if (controller == null)
                return shape;
            if (controller.passengers == null || !controller.passengers.containsKey(entity))
                return BeltShapes.getCollisionShape(state);
            return shape;

        }).orElse(shape);
    }

    public static void initBelt(Level world, BlockPos pos) {
        if (world.isClientSide())
            return;
        if (world instanceof ServerLevel serverWorld && serverWorld.getChunkSource().getGenerator() instanceof DebugLevelSource)
            return;

        BlockState state = world.getBlockState(pos);
        if (!state.is(AllBlocks.BELT))
            return;
        // Find controller
        int limit = 1000;
        BlockPos currentPos = pos;
        while (limit-- > 0) {
            BlockState currentState = world.getBlockState(currentPos);
            if (!currentState.is(AllBlocks.BELT)) {
                world.destroyBlock(pos, true);
                return;
            }
            BlockPos nextSegmentPosition = nextSegmentPosition(currentState, currentPos, false);
            if (nextSegmentPosition == null)
                break;
            if (!world.isLoaded(nextSegmentPosition))
                return;
            currentPos = nextSegmentPosition;
        }

        // Init belts
        int index = 0;
        List<BlockPos> beltChain = getBeltChain(world, currentPos);
        if (beltChain.size() < 2) {
            world.destroyBlock(currentPos, true);
            return;
        }

        for (BlockPos beltPos : beltChain) {
            BlockEntity blockEntity = world.getBlockEntity(beltPos);
            BlockState currentState = world.getBlockState(beltPos);

            if (blockEntity instanceof BeltBlockEntity be && currentState.is(AllBlocks.BELT)) {
                be.setController(currentPos);
                be.beltLength = beltChain.size();
                be.index = index;
                be.attachKinetics();
                be.setChanged();
                be.sendData();

                if (be.isController() && !canTransportObjects(currentState))
                    be.getInventory().ejectAll();
            } else {
                world.destroyBlock(currentPos, true);
                return;
            }
            index++;
        }

    }

    @Override
    public void affectNeighborsAfterRemoval(BlockState state, ServerLevel world, BlockPos pos, boolean isMoving) {
        super.affectNeighborsAfterRemoval(state, world, pos, isMoving);

        if (world.isClientSide())
            return;
        if (isMoving)
            return;

        // Destroy chain
        for (boolean forward : Iterate.trueAndFalse) {
            BlockPos currentPos = nextSegmentPosition(state, pos, forward);
            if (currentPos == null)
                continue;
            BlockState currentState = world.getBlockState(currentPos);
            if (!currentState.is(AllBlocks.BELT))
                continue;

            boolean hasPulley = false;
            BlockEntity blockEntity = world.getBlockEntity(currentPos);
            if (blockEntity instanceof BeltBlockEntity belt) {
                if (belt.isController())
                    belt.getInventory().ejectAll();

                hasPulley = belt.hasPulley();
            }

            world.removeBlockEntity(currentPos);
            BlockState shaftState = AllBlocks.SHAFT.defaultBlockState().setValue(BlockStateProperties.AXIS, getRotationAxis(currentState));
            world.setBlock(
                currentPos,
                ProperWaterloggedBlock.withWater(world, hasPulley ? shaftState : Blocks.AIR.defaultBlockState(), currentPos),
                Block.UPDATE_ALL
            );
            world.levelEvent(LevelEvent.PARTICLES_DESTROY_BLOCK, currentPos, Block.getId(currentState));
        }
    }

    @Override
    public BlockState updateShape(
        BlockState state,
        LevelReader world,
        ScheduledTickAccess tickView,
        BlockPos pos,
        Direction side,
        BlockPos p_196271_6_,
        BlockState p_196271_3_,
        RandomSource random
    ) {
        updateWater(world, tickView, state, pos);
        if (side.getAxis().isHorizontal())
            updateTunnelConnections((LevelAccessor) world, pos.above());
        if (side == Direction.UP)
            updateCoverProperty(world, pos, state);
        return state;
    }

    public void updateCoverProperty(LevelReader world, BlockPos pos, BlockState state) {
        if (world.isClientSide())
            return;
        if (state.getValue(CASING) && state.getValue(SLOPE) == BeltSlope.HORIZONTAL)
            withBlockEntityDo(world, pos, bbe -> bbe.setCovered(isBlockCoveringBelt(world, pos.above())));
    }

    public static boolean isBlockCoveringBelt(LevelReader world, BlockPos pos) {
        BlockState blockState = world.getBlockState(pos);
        VoxelShape collisionShape = blockState.getCollisionShape(world, pos);
        if (collisionShape.isEmpty())
            return false;
        AABB bounds = collisionShape.bounds();
        if (bounds.getXsize() < .5f || bounds.getZsize() < .5f)
            return false;
        if (bounds.minY > 0)
            return false;
        if (blockState.is(AllBlocks.CRUSHING_WHEEL_CONTROLLER))
            return false;
        if (FunnelBlock.isFunnel(blockState) && FunnelBlock.getFunnelFacing(blockState) != Direction.UP)
            return false;
        return !(blockState.getBlock() instanceof BeltTunnelBlock);
    }

    private void updateTunnelConnections(LevelAccessor world, BlockPos pos) {
        Block tunnelBlock = world.getBlockState(pos).getBlock();
        if (tunnelBlock instanceof BeltTunnelBlock)
            ((BeltTunnelBlock) tunnelBlock).updateTunnel(world, pos);
    }

    public static List<BlockPos> getBeltChain(LevelReader world, BlockPos controllerPos) {
        List<BlockPos> positions = new LinkedList<>();

        BlockState blockState = world.getBlockState(controllerPos);
        if (!blockState.is(AllBlocks.BELT))
            return positions;

        int limit = 1000;
        BlockPos current = controllerPos;
        while (limit-- > 0 && current != null) {
            BlockState state = world.getBlockState(current);
            if (!state.is(AllBlocks.BELT))
                break;
            positions.add(current);
            current = nextSegmentPosition(state, current, true);
        }

        return positions;
    }

    public static BlockPos nextSegmentPosition(BlockState state, BlockPos pos, boolean forward) {
        Direction direction = state.getValue(HORIZONTAL_FACING);
        BeltSlope slope = state.getValue(SLOPE);
        BeltPart part = state.getValue(PART);

        int offset = forward ? 1 : -1;

        if (part == BeltPart.END && forward || part == BeltPart.START && !forward)
            return null;
        if (slope == BeltSlope.VERTICAL)
            return pos.above(direction.getAxisDirection() == AxisDirection.POSITIVE ? offset : -offset);
        pos = pos.relative(direction, offset);
        if (slope != BeltSlope.HORIZONTAL && slope != BeltSlope.SIDEWAYS)
            return pos.above(slope == BeltSlope.UPWARD ? offset : -offset);
        return pos;
    }

    @Override
    public Class<BeltBlockEntity> getBlockEntityClass() {
        return BeltBlockEntity.class;
    }

    @Override
    public BlockEntityType<? extends BeltBlockEntity> getBlockEntityType() {
        return AllBlockEntityTypes.BELT;
    }

    @Override
    public ItemRequirement getRequiredItems(BlockState state, BlockEntity be) {
        List<ItemStack> required = new ArrayList<>();
        if (state.getValue(PART) != BeltPart.MIDDLE)
            required.add(AllItems.SHAFT.getDefaultInstance());
        if (state.getValue(PART) == BeltPart.START)
            required.add(AllItems.BELT_CONNECTOR.getDefaultInstance());
        if (required.isEmpty())
            return ItemRequirement.NONE;
        return new ItemRequirement(ItemUseType.CONSUME, required);
    }

    @Override
    public BlockState rotate(BlockState state, Rotation rot) {
        BlockState rotate = super.rotate(state, rot);

        if (state.getValue(SLOPE) != BeltSlope.VERTICAL)
            return rotate;
        if (state.getValue(HORIZONTAL_FACING).getAxisDirection() != rotate.getValue(HORIZONTAL_FACING).getAxisDirection()) {
            if (state.getValue(PART) == BeltPart.START)
                return rotate.setValue(PART, BeltPart.END);
            if (state.getValue(PART) == BeltPart.END)
                return rotate.setValue(PART, BeltPart.START);
        }

        return rotate;
    }

    public BlockState transform(BlockState state, StructureTransform transform) {
        if (transform.mirror != null) {
            state = mirror(state, transform.mirror);
        }

        if (transform.rotationAxis == Direction.Axis.Y) {
            return rotate(state, transform.rotation);
        }
        return transformInner(state, transform);
    }

    protected BlockState transformInner(BlockState state, StructureTransform transform) {
        boolean halfTurn = transform.rotation == Rotation.CLOCKWISE_180;

        Direction initialDirection = state.getValue(HORIZONTAL_FACING);
        boolean diagonal = state.getValue(SLOPE) == BeltSlope.DOWNWARD || state.getValue(SLOPE) == BeltSlope.UPWARD;

        if (!diagonal) {
            for (int i = 0; i < transform.rotation.ordinal(); i++) {
                Direction direction = state.getValue(HORIZONTAL_FACING);
                BeltSlope slope = state.getValue(SLOPE);
                boolean vertical = slope == BeltSlope.VERTICAL;
                boolean horizontal = slope == BeltSlope.HORIZONTAL;
                boolean sideways = slope == BeltSlope.SIDEWAYS;

                Direction newDirection = direction.getOpposite();
                BeltSlope newSlope = BeltSlope.VERTICAL;

                if (vertical) {
                    if (direction.getAxis() == transform.rotationAxis) {
                        newDirection = direction.getCounterClockWise();
                        newSlope = BeltSlope.SIDEWAYS;
                    } else {
                        newSlope = BeltSlope.HORIZONTAL;
                        newDirection = direction;
                        if (direction.getAxis() == Axis.Z)
                            newDirection = direction.getOpposite();
                    }
                }

                if (sideways) {
                    newDirection = direction;
                    if (direction.getAxis() == transform.rotationAxis)
                        newSlope = BeltSlope.HORIZONTAL;
                    else
                        newDirection = direction.getCounterClockWise();
                }

                if (horizontal) {
                    newDirection = direction;
                    if (direction.getAxis() == transform.rotationAxis)
                        newSlope = BeltSlope.SIDEWAYS;
                    else if (direction.getAxis() != Axis.Z)
                        newDirection = direction.getOpposite();
                }

                state = state.setValue(HORIZONTAL_FACING, newDirection);
                state = state.setValue(SLOPE, newSlope);
            }

        } else if (initialDirection.getAxis() != transform.rotationAxis) {
            for (int i = 0; i < transform.rotation.ordinal(); i++) {
                Direction direction = state.getValue(HORIZONTAL_FACING);
                Direction newDirection = direction.getOpposite();
                BeltSlope slope = state.getValue(SLOPE);
                boolean upward = slope == BeltSlope.UPWARD;
                boolean downward = slope == BeltSlope.DOWNWARD;

                // Rotate diagonal
                if (direction.getAxisDirection() == AxisDirection.POSITIVE ^ downward ^ direction.getAxis() == Axis.Z) {
                    state = state.setValue(SLOPE, upward ? BeltSlope.DOWNWARD : BeltSlope.UPWARD);
                } else {
                    state = state.setValue(HORIZONTAL_FACING, newDirection);
                }
            }

        } else if (halfTurn) {
            Direction direction = state.getValue(HORIZONTAL_FACING);
            Direction newDirection = direction.getOpposite();
            BeltSlope slope = state.getValue(SLOPE);
            boolean vertical = slope == BeltSlope.VERTICAL;

            if (diagonal) {
                state = state.setValue(
                    SLOPE,
                    slope == BeltSlope.UPWARD ? BeltSlope.DOWNWARD : slope == BeltSlope.DOWNWARD ? BeltSlope.UPWARD : slope
                );
            } else if (vertical) {
                state = state.setValue(HORIZONTAL_FACING, newDirection);
            }
        }

        return state;
    }

    @Override
    protected boolean isPathfindable(BlockState state, PathComputationType pathComputationType) {
        return false;
    }

    @Override
    public FluidState getFluidState(BlockState pState) {
        return fluidState(pState);
    }
}
