package com.zurrtum.create.content.processing.burner;

import com.mojang.serialization.Codec;
import com.mojang.serialization.MapCodec;
import com.zurrtum.create.AllBlockEntityTypes;
import com.zurrtum.create.AllBlocks;
import com.zurrtum.create.AllItems;
import com.zurrtum.create.AllShapes;
import com.zurrtum.create.api.entity.FakePlayerHandler;
import com.zurrtum.create.api.schematic.requirement.SpecialBlockItemRequirement;
import com.zurrtum.create.content.equipment.wrench.IWrenchable;
import com.zurrtum.create.content.logistics.stockTicker.StockTickerBlockEntity;
import com.zurrtum.create.content.logistics.stockTicker.StockTickerInteractionHandler;
import com.zurrtum.create.content.processing.basin.BasinBlockEntity;
import com.zurrtum.create.content.schematics.requirement.ItemRequirement;
import com.zurrtum.create.content.schematics.requirement.ItemRequirement.ItemUseType;
import com.zurrtum.create.foundation.block.IBE;
import org.jetbrains.annotations.Nullable;

import java.util.Locale;

import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.sounds.SoundEvents;
import net.minecraft.sounds.SoundSource;
import net.minecraft.util.RandomSource;
import net.minecraft.util.StringRepresentable;
import net.minecraft.world.InteractionHand;
import net.minecraft.world.InteractionResult;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.item.FlintAndSteelItem;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.context.BlockPlaceContext;
import net.minecraft.world.level.BlockGetter;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.LevelReader;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.block.HorizontalDirectionalBlock;
import net.minecraft.world.level.block.entity.BlockEntity;
import net.minecraft.world.level.block.entity.BlockEntityType;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.block.state.StateDefinition.Builder;
import net.minecraft.world.level.block.state.properties.EnumProperty;
import net.minecraft.world.level.pathfinder.PathComputationType;
import net.minecraft.world.phys.BlockHitResult;
import net.minecraft.world.phys.shapes.CollisionContext;
import net.minecraft.world.phys.shapes.VoxelShape;

public class BlazeBurnerBlock extends HorizontalDirectionalBlock implements IBE<BlazeBurnerBlockEntity>, IWrenchable, SpecialBlockItemRequirement {
    public static final MapCodec<BlazeBurnerBlock> CODEC = simpleCodec(BlazeBurnerBlock::new);
    public static final EnumProperty<HeatLevel> HEAT_LEVEL = EnumProperty.create("blaze", HeatLevel.class);

    public BlazeBurnerBlock(Properties settings) {
        super(settings);
        registerDefaultState(defaultBlockState().setValue(HEAT_LEVEL, HeatLevel.NONE));
    }

    @Override
    protected void createBlockStateDefinition(Builder<Block, BlockState> builder) {
        super.createBlockStateDefinition(builder);
        builder.add(HEAT_LEVEL, FACING);
    }

    @Override
    public void onPlace(BlockState state, Level world, BlockPos pos, BlockState p_220082_4_, boolean p_220082_5_) {
        if (world.isClientSide())
            return;
        BlockEntity blockEntity = world.getBlockEntity(pos.above());
        if (!(blockEntity instanceof BasinBlockEntity basin))
            return;
        basin.notifyChangeOfContents();
    }

    @Nullable
    @Override
    public BlockEntity newBlockEntity(BlockPos pos, BlockState state) {
        if (state.getValue(HEAT_LEVEL) == HeatLevel.NONE)
            return null;
        return IBE.super.newBlockEntity(pos, state);
    }

    @Override
    protected InteractionResult useItemOn(
        ItemStack stack,
        BlockState state,
        Level level,
        BlockPos pos,
        Player player,
        InteractionHand hand,
        BlockHitResult hitResult
    ) {
        HeatLevel heat = state.getValue(HEAT_LEVEL);

        if (stack.is(AllItems.GOGGLES) && heat != HeatLevel.NONE)
            return onBlockEntityUseItemOn(
                level, pos, bbte -> {
                    if (bbte.goggles)
                        return InteractionResult.TRY_WITH_EMPTY_HAND;
                    bbte.goggles = true;
                    bbte.notifyUpdate();
                    return InteractionResult.SUCCESS;
                }
            );

        BlazeBurnerBlockEntity be = getBlockEntity(level, pos);
        if (be != null && be.stockKeeper) {
            StockTickerBlockEntity stockTicker = BlazeBurnerBlockEntity.getStockTicker(level, pos);
            if (stockTicker != null)
                StockTickerInteractionHandler.interactWithLogisticsManagerAt(player, level, stockTicker.getBlockPos());
            return InteractionResult.SUCCESS;
        }

        if (stack.isEmpty() && heat != HeatLevel.NONE)
            return onBlockEntityUseItemOn(
                level, pos, bbte -> {
                    if (!bbte.goggles)
                        return InteractionResult.TRY_WITH_EMPTY_HAND;
                    bbte.goggles = false;
                    bbte.notifyUpdate();
                    return InteractionResult.SUCCESS;
                }
            );

        if (heat == HeatLevel.NONE) {
            if (stack.getItem() instanceof FlintAndSteelItem) {
                level.playSound(player, pos, SoundEvents.FLINTANDSTEEL_USE, SoundSource.BLOCKS, 1.0F, level.random.nextFloat() * 0.4F + 0.8F);
                if (level.isClientSide())
                    return InteractionResult.SUCCESS;
                stack.hurtAndBreak(1, player, hand.asEquipmentSlot());
                level.setBlockAndUpdate(pos, AllBlocks.LIT_BLAZE_BURNER.defaultBlockState());
                return InteractionResult.SUCCESS;
            }
            return InteractionResult.TRY_WITH_EMPTY_HAND;
        }

        boolean doNotConsume = player.isCreative();
        boolean forceOverflow = !FakePlayerHandler.has(player);

        InteractionResult res = tryInsert(state, level, pos, stack, doNotConsume, forceOverflow, false);
        if (res instanceof InteractionResult.Success success) {
            ItemStack leftover = success.heldItemTransformedTo();
            if (!level.isClientSide() && !doNotConsume && leftover != null && !leftover.isEmpty()) {
                if (stack.isEmpty()) {
                    player.setItemInHand(hand, leftover);
                } else if (!player.getInventory().add(leftover)) {
                    player.drop(leftover, false);
                }
            }
        }

        if (res.consumesAction())
            return res;
        return InteractionResult.TRY_WITH_EMPTY_HAND;
    }

    public static InteractionResult tryInsert(
        BlockState state,
        Level world,
        BlockPos pos,
        ItemStack stack,
        boolean doNotConsume,
        boolean forceOverflow,
        boolean simulate
    ) {
        if (!state.hasBlockEntity())
            return InteractionResult.FAIL;

        BlockEntity be = world.getBlockEntity(pos);
        if (!(be instanceof BlazeBurnerBlockEntity burnerBE))
            return InteractionResult.FAIL;

        if (burnerBE.isCreativeFuel(stack)) {
            if (!simulate)
                burnerBE.applyCreativeFuel();
            return InteractionResult.SUCCESS.heldItemTransformedTo(ItemStack.EMPTY);
        }
        if (!burnerBE.tryUpdateFuel(stack, forceOverflow, simulate))
            return InteractionResult.FAIL;

        if (!doNotConsume) {
            ItemStack container = stack.getItem().getCraftingRemainder();
            if (!world.isClientSide()) {
                stack.shrink(1);
            }
            return InteractionResult.SUCCESS.heldItemTransformedTo(container);
        }
        return InteractionResult.SUCCESS.heldItemTransformedTo(ItemStack.EMPTY);
    }

    @Override
    public BlockState getStateForPlacement(BlockPlaceContext context) {
        ItemStack stack = context.getItemInHand();
        Item item = stack.getItem();
        BlockState defaultState = defaultBlockState();
        if (!(item instanceof BlazeBurnerBlockItem))
            return defaultState;
        HeatLevel initialHeat = ((BlazeBurnerBlockItem) item).hasCapturedBlaze() ? HeatLevel.SMOULDERING : HeatLevel.NONE;
        return defaultState.setValue(HEAT_LEVEL, initialHeat).setValue(FACING, context.getHorizontalDirection().getOpposite());
    }

    @Override
    public VoxelShape getShape(BlockState state, BlockGetter reader, BlockPos pos, CollisionContext context) {
        return AllShapes.HEATER_BLOCK_SHAPE;
    }

    @Override
    public VoxelShape getCollisionShape(BlockState p_220071_1_, BlockGetter p_220071_2_, BlockPos p_220071_3_, CollisionContext p_220071_4_) {
        if (p_220071_4_ == CollisionContext.empty())
            return AllShapes.HEATER_BLOCK_SPECIAL_COLLISION_SHAPE;
        return getShape(p_220071_1_, p_220071_2_, p_220071_3_, p_220071_4_);
    }

    @Override
    public boolean hasAnalogOutputSignal(BlockState p_149740_1_) {
        return true;
    }

    @Override
    public int getAnalogOutputSignal(BlockState state, Level p_180641_2_, BlockPos p_180641_3_, Direction direction) {
        return Math.max(0, state.getValue(HEAT_LEVEL).ordinal() - 1);
    }

    @Override
    protected boolean isPathfindable(BlockState state, PathComputationType pathComputationType) {
        return false;
    }

    @Override
    public void animateTick(BlockState state, Level world, BlockPos pos, RandomSource random) {
        if (random.nextInt(10) != 0)
            return;
        if (!state.getValue(HEAT_LEVEL).isAtLeast(HeatLevel.SMOULDERING))
            return;
        world.playLocalSound(
            pos.getX() + 0.5F,
            pos.getY() + 0.5F,
            pos.getZ() + 0.5F,
            SoundEvents.CAMPFIRE_CRACKLE,
            SoundSource.BLOCKS,
            0.5F + random.nextFloat(),
            random.nextFloat() * 0.7F + 0.6F,
            false
        );
    }

    @Override
    protected MapCodec<? extends HorizontalDirectionalBlock> codec() {
        return CODEC;
    }

    public static HeatLevel getHeatLevelOf(BlockState blockState) {
        return blockState.hasProperty(BlazeBurnerBlock.HEAT_LEVEL) ? blockState.getValue(BlazeBurnerBlock.HEAT_LEVEL) : HeatLevel.NONE;
    }

    public static int getLight(BlockState state) {
        HeatLevel level = state.getValue(HEAT_LEVEL);
        return switch (level) {
            case NONE -> 0;
            case SMOULDERING -> 8;
            default -> 15;
        };
    }

    @Override
    protected ItemStack getCloneItemStack(LevelReader world, BlockPos pos, BlockState state, boolean includeData) {
        return getLitOrUnlitStack(state);
    }

    @Override
    public Class<BlazeBurnerBlockEntity> getBlockEntityClass() {
        return BlazeBurnerBlockEntity.class;
    }

    @Override
    public BlockEntityType<? extends BlazeBurnerBlockEntity> getBlockEntityType() {
        return AllBlockEntityTypes.HEATER;
    }

    @Override
    public ItemRequirement getRequiredItems(BlockState state, @Nullable BlockEntity blockEntity) {
        return new ItemRequirement(ItemUseType.CONSUME, getLitOrUnlitStack(state));
    }

    private static ItemStack getLitOrUnlitStack(BlockState state) {
        boolean isLit = state.getValue(HEAT_LEVEL) != HeatLevel.NONE;
        return (isLit ? AllItems.BLAZE_BURNER : AllItems.EMPTY_BLAZE_BURNER).getDefaultInstance();
    }

    public enum HeatLevel implements StringRepresentable {
        NONE,
        SMOULDERING,
        FADING,
        KINDLED,
        SEETHING;

        public static final Codec<HeatLevel> CODEC = StringRepresentable.fromEnum(HeatLevel::values);

        public static HeatLevel byIndex(int index) {
            return values()[index];
        }

        @Override
        public String getSerializedName() {
            return name().toLowerCase(Locale.ROOT);
        }

        public boolean isAtLeast(HeatLevel heatLevel) {
            return this.ordinal() >= heatLevel.ordinal();
        }

        public HeatLevel nextActiveLevel() {
            return byIndex(ordinal() % (values().length - 1) + 1);
        }
    }

}
