package com.zurrtum.create.content.redstone.displayLink;

import com.mojang.serialization.MapCodec;
import com.zurrtum.create.AllBlockEntityTypes;
import com.zurrtum.create.AllBlocks;
import com.zurrtum.create.AllClientHandle;
import com.zurrtum.create.AllShapes;
import com.zurrtum.create.api.behaviour.display.DisplaySource;
import com.zurrtum.create.catnip.data.Iterate;
import com.zurrtum.create.content.redstone.displayLink.source.RedstonePowerDisplaySource;
import com.zurrtum.create.foundation.advancement.AdvancementBehaviour;
import com.zurrtum.create.foundation.block.IBE;
import com.zurrtum.create.foundation.block.NeighborUpdateListeningBlock;
import com.zurrtum.create.foundation.block.WrenchableDirectionalBlock;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.function.BiConsumer;
import java.util.function.Consumer;

import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.world.InteractionResult;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.context.BlockPlaceContext;
import net.minecraft.world.level.BlockGetter;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.LevelAccessor;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.block.DirectionalBlock;
import net.minecraft.world.level.block.entity.BlockEntity;
import net.minecraft.world.level.block.entity.BlockEntityType;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.block.state.StateDefinition.Builder;
import net.minecraft.world.level.block.state.properties.BlockStateProperties;
import net.minecraft.world.level.block.state.properties.BooleanProperty;
import net.minecraft.world.level.pathfinder.PathComputationType;
import net.minecraft.world.level.redstone.Orientation;
import net.minecraft.world.phys.BlockHitResult;
import net.minecraft.world.phys.shapes.CollisionContext;
import net.minecraft.world.phys.shapes.VoxelShape;

public class DisplayLinkBlock extends WrenchableDirectionalBlock implements IBE<DisplayLinkBlockEntity>, NeighborUpdateListeningBlock {

    public static final BooleanProperty POWERED = BlockStateProperties.POWERED;

    public static final MapCodec<DisplayLinkBlock> CODEC = simpleCodec(DisplayLinkBlock::new);

    public DisplayLinkBlock(Properties p_i48415_1_) {
        super(p_i48415_1_);
        registerDefaultState(defaultBlockState().setValue(POWERED, false));
    }

    @Override
    public BlockState getStateForPlacement(BlockPlaceContext context) {
        BlockState placed = super.getStateForPlacement(context);
        placed = placed.setValue(FACING, context.getClickedFace());
        return placed.setValue(POWERED, shouldBePowered(placed, context.getLevel(), context.getClickedPos()));
    }

    @Override
    public void setPlacedBy(Level pLevel, BlockPos pPos, BlockState pState, LivingEntity pPlacer, ItemStack pStack) {
        super.setPlacedBy(pLevel, pPos, pState, pPlacer, pStack);
        AdvancementBehaviour.setPlacedBy(pLevel, pPos, pPlacer);
    }

    public static void notifyGatherers(LevelAccessor level, BlockPos pos) {
        forEachAttachedGatherer(level, pos, DisplayLinkBlockEntity::tickSource);
    }

    @SuppressWarnings("unchecked")
    public static <T extends DisplaySource> void sendToGatherers(
        LevelAccessor level,
        BlockPos pos,
        BiConsumer<DisplayLinkBlockEntity, T> callback,
        Class<T> type
    ) {
        forEachAttachedGatherer(
            level, pos, dgte -> {
                if (type.isInstance(dgte.activeSource))
                    callback.accept(dgte, (T) dgte.activeSource);
            }
        );
    }

    private static void forEachAttachedGatherer(LevelAccessor level, BlockPos pos, Consumer<DisplayLinkBlockEntity> callback) {
        for (Direction d : Iterate.directions) {
            BlockPos offsetPos = pos.relative(d);
            BlockState blockState = level.getBlockState(offsetPos);
            if (!blockState.is(AllBlocks.DISPLAY_LINK))
                continue;

            BlockEntity blockEntity = level.getBlockEntity(offsetPos);
            if (!(blockEntity instanceof DisplayLinkBlockEntity dlbe))
                continue;
            if (dlbe.activeSource == null)
                continue;
            if (dlbe.getDirection() != d.getOpposite())
                continue;

            callback.accept(dlbe);
        }
    }

    @Override
    public void neighborUpdate(BlockState state, Level worldIn, BlockPos pos, Block blockIn, BlockPos fromPos, boolean isMoving) {
        if (worldIn.isClientSide())
            return;
        if (fromPos.equals(pos.relative(state.getValue(FACING).getOpposite())))
            sendToGatherers(worldIn, fromPos, (dlte, p) -> dlte.tickSource(), RedstonePowerDisplaySource.class);
    }

    @Override
    public void neighborChanged(
        BlockState state,
        Level worldIn,
        BlockPos pos,
        Block blockIn,
        @Nullable Orientation wireOrientation,
        boolean isMoving
    ) {
        if (worldIn.isClientSide())
            return;

        boolean powered = shouldBePowered(state, worldIn, pos);
        boolean previouslyPowered = state.getValue(POWERED);
        if (previouslyPowered != powered) {
            worldIn.setBlock(pos, state.cycle(POWERED), Block.UPDATE_CLIENTS);
            if (!powered)
                withBlockEntityDo(worldIn, pos, DisplayLinkBlockEntity::onNoLongerPowered);
        }
    }

    private boolean shouldBePowered(BlockState state, Level worldIn, BlockPos pos) {
        boolean powered = false;
        for (Direction d : Iterate.directions) {
            if (d.getOpposite() == state.getValue(FACING))
                continue;
            if (worldIn.getSignal(pos.relative(d), d) == 0)
                continue;
            powered = true;
            break;
        }
        return powered;
    }

    @Override
    protected void createBlockStateDefinition(Builder<Block, BlockState> builder) {
        super.createBlockStateDefinition(builder.add(POWERED));
    }

    @Override
    protected InteractionResult useWithoutItem(BlockState state, Level level, BlockPos pos, Player player, BlockHitResult hitResult) {
        if (player == null)
            return InteractionResult.PASS;
        if (player.isShiftKeyDown())
            return InteractionResult.PASS;
        if (level.isClientSide()) {
            withBlockEntityDo(level, pos, be -> AllClientHandle.INSTANCE.openDisplayLinkScreen(be, player));
        }
        return InteractionResult.SUCCESS;
    }

    @Override
    protected boolean isPathfindable(BlockState state, PathComputationType pathComputationType) {
        return false;
    }

    @Override
    public VoxelShape getShape(BlockState pState, BlockGetter pLevel, BlockPos pPos, CollisionContext pContext) {
        return AllShapes.DATA_GATHERER.get(pState.getValue(FACING));
    }

    @Override
    public Class<DisplayLinkBlockEntity> getBlockEntityClass() {
        return DisplayLinkBlockEntity.class;
    }

    @Override
    public BlockEntityType<? extends DisplayLinkBlockEntity> getBlockEntityType() {
        return AllBlockEntityTypes.DISPLAY_LINK;
    }

    @Override
    protected @NotNull MapCodec<? extends DirectionalBlock> codec() {
        return CODEC;
    }
}
