package com.zurrtum.create.content.redstone.thresholdSwitch;

import com.zurrtum.create.AllBlockEntityTypes;
import com.zurrtum.create.catnip.math.BlockFace;
import com.zurrtum.create.content.logistics.stockTicker.StockTickerBlockEntity;
import com.zurrtum.create.content.processing.recipe.ProcessingInventory;
import com.zurrtum.create.content.redstone.DirectedDirectionalBlock;
import com.zurrtum.create.content.redstone.displayLink.DisplayLinkBlock;
import com.zurrtum.create.foundation.blockEntity.SmartBlockEntity;
import com.zurrtum.create.foundation.blockEntity.behaviour.BlockEntityBehaviour;
import com.zurrtum.create.foundation.blockEntity.behaviour.filtering.ServerFilteringBehaviour;
import com.zurrtum.create.foundation.blockEntity.behaviour.inventory.CapManipulationBehaviourBase.InterfaceProvider;
import com.zurrtum.create.foundation.blockEntity.behaviour.inventory.InvManipulationBehaviour;
import com.zurrtum.create.foundation.blockEntity.behaviour.inventory.TankManipulationBehaviour;
import com.zurrtum.create.foundation.blockEntity.behaviour.inventory.VersionedInventoryTrackerBehaviour;
import com.zurrtum.create.foundation.item.ItemHelper;
import com.zurrtum.create.infrastructure.fluids.FluidInventory;
import com.zurrtum.create.infrastructure.fluids.FluidStack;

import java.util.List;

import net.minecraft.core.BlockPos;
import net.minecraft.util.Mth;
import net.minecraft.world.Container;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.block.entity.BlockEntity;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.storage.ValueInput;
import net.minecraft.world.level.storage.ValueOutput;
import net.minecraft.world.ticks.TickPriority;

public class ThresholdSwitchBlockEntity extends SmartBlockEntity {

    public int onWhenAbove;
    public int offWhenBelow;

    public int currentMinLevel;
    public int currentLevel;
    public int currentMaxLevel;
    public boolean inStacks;

    private boolean redstoneState;
    private boolean inverted;
    private boolean poweredAfterDelay;

    private ServerFilteringBehaviour filtering;
    private InvManipulationBehaviour observedInventory;
    private TankManipulationBehaviour observedTank;
    private VersionedInventoryTrackerBehaviour invVersionTracker;

    //TODO
    //    private static final List<ThresholdSwitchCompat> COMPAT = List.of(new FunctionalStorage(), new SophisticatedStorage(), new StorageDrawers());

    public ThresholdSwitchBlockEntity(BlockPos pos, BlockState state) {
        super(AllBlockEntityTypes.THRESHOLD_SWITCH, pos, state);
        onWhenAbove = 128;
        offWhenBelow = 64;
        currentLevel = -1;
        redstoneState = false;
        inverted = false;
        poweredAfterDelay = false;
        setLazyTickRate(10);
    }

    @Override
    protected void read(ValueInput view, boolean clientPacket) {
        onWhenAbove = view.getIntOr("OnAboveAmount", 0);
        offWhenBelow = view.getIntOr("OffBelowAmount", 0);
        currentLevel = view.getIntOr("CurrentAmount", 0);
        currentMinLevel = view.getIntOr("CurrentMinAmount", 0);
        currentMaxLevel = view.getIntOr("CurrentMaxAmount", 0);
        inStacks = view.getBooleanOr("InStacks", false);
        redstoneState = view.getBooleanOr("Powered", false);
        inverted = view.getBooleanOr("Inverted", false);
        poweredAfterDelay = view.getBooleanOr("PoweredAfterDelay", false);
        super.read(view, clientPacket);
    }

    protected void writeCommon(ValueOutput view) {
        view.putFloat("OnAboveAmount", onWhenAbove);
        view.putFloat("OffBelowAmount", offWhenBelow);
        view.putBoolean("Inverted", inverted);
    }

    @Override
    public void write(ValueOutput view, boolean clientPacket) {
        writeCommon(view);
        view.putInt("CurrentAmount", currentLevel);
        view.putInt("CurrentMinAmount", currentMinLevel);
        view.putInt("CurrentMaxAmount", currentMaxLevel);
        view.putBoolean("InStacks", inStacks);
        view.putBoolean("Powered", redstoneState);
        view.putBoolean("PoweredAfterDelay", poweredAfterDelay);
        super.write(view, clientPacket);
    }

    @Override
    public void writeSafe(ValueOutput view) {
        writeCommon(view);
        super.writeSafe(view);
    }

    public int getMinLevel() {
        return currentMinLevel;
    }

    public int getStockLevel() {
        return currentLevel;
    }

    public int getMaxLevel() {
        return currentMaxLevel;
    }

    public void updateCurrentLevel() {
        boolean changed = false;
        int prevLevel = currentLevel;
        int prevMaxLevel = currentMaxLevel;

        BlockPos target = getTargetPos();
        BlockEntity targetBlockEntity = level.getBlockEntity(target);

        observedInventory.findNewCapability();
        observedTank.findNewCapability();

        if (targetBlockEntity instanceof ThresholdSwitchObservable observable) {
            currentMinLevel = observable.getMinValue();
            currentLevel = observable.getCurrentValue();
            currentMaxLevel = observable.getMaxValue();

		/*} else if (StorageDrawers.isDrawer(targetBlockEntity) && observedInventory.hasInventory()) {
			currentMinLevel = 0;
			currentLevel = StorageDrawers.getItemCount(observedInventory.getInventory(), filtering);
			currentMaxLevel = StorageDrawers.getTotalStorageSpace(observedInventory.getInventory());
		*/

        } else if (observedInventory.hasInventory() || observedTank.hasInventory()) {
            currentMinLevel = 0;
            currentLevel = 0;
            currentMaxLevel = 0;

            if (observedInventory.hasInventory()) {

                // Item inventory
                Container inv = observedInventory.getInventory();
                if (invVersionTracker.stillWaiting(inv)) {
                    currentLevel = prevLevel;
                    currentMaxLevel = prevMaxLevel;

                } else {
                    invVersionTracker.awaitNewVersion(inv);
                    for (int slot = 0, size = inv.getContainerSize(); slot < size; slot++) {
                        ItemStack stackInSlot = inv.getItem(slot);
                        int space = stackInSlot.isEmpty() ? 64 : inv.getMaxStackSize(stackInSlot);
                        if (space == 0)
                            continue;

                        currentMaxLevel += space;
                        if (filtering.test(stackInSlot))
                            currentLevel += stackInSlot.getCount();
                    }
                }
            }

            if (observedTank.hasInventory()) {
                // Fluid inventory
                FluidInventory tank = observedTank.getInventory();
                for (int slot = 0, size = tank.size(); slot < size; slot++) {
                    FluidStack stackInSlot = tank.getStack(slot);
                    int space = tank.getMaxAmount(stackInSlot);
                    if (space == 0)
                        continue;

                    currentMaxLevel += space;
                    if (filtering.test(stackInSlot))
                        currentLevel += stackInSlot.getAmount();
                }
            }

        } else {
            // No compatible inventories found
            currentMinLevel = -1;
            currentMaxLevel = -1;
            if (currentLevel == -1)
                return;

            level.setBlock(worldPosition, getBlockState().setValue(ThresholdSwitchBlock.LEVEL, 0), Block.UPDATE_ALL);
            currentLevel = -1;
            redstoneState = false;
            sendData();
            scheduleBlockTick();
            return;
        }

        currentLevel = Mth.clamp(currentLevel, currentMinLevel, currentMaxLevel);
        changed = currentLevel != prevLevel;

        boolean previouslyPowered = redstoneState;
        if (redstoneState && currentLevel <= offWhenBelow)
            redstoneState = false;
        else if (!redstoneState && currentLevel >= onWhenAbove)
            redstoneState = true;
        boolean update = previouslyPowered != redstoneState;

        int displayLevel = 0;
        float normedLevel = (float) (currentLevel - currentMinLevel) / (currentMaxLevel - currentMinLevel);
        if (currentLevel > 0)
            displayLevel = (int) (1 + normedLevel * 4);
        level.setBlock(worldPosition, getBlockState().setValue(ThresholdSwitchBlock.LEVEL, displayLevel), update ? 3 : 2);

        if (update)
            scheduleBlockTick();

        if (changed || update) {
            DisplayLinkBlock.notifyGatherers(level, worldPosition);
            notifyUpdate();
        }
    }

    private boolean isSuitableInventory(BlockEntity be) {
        return be != null && !(be instanceof StockTickerBlockEntity || ItemHelper.getInventory(
            level,
            be.getBlockPos(),
            null,
            be,
            null
        ) instanceof ProcessingInventory);
    }

    public BlockPos getTargetPos() {
        return worldPosition.relative(ThresholdSwitchBlock.getTargetDirection(getBlockState()));
    }

    public ItemStack getDisplayItemForScreen() {
        BlockPos target = getTargetPos();
        return new ItemStack(level.getBlockState(target).getBlock());
    }

    public static enum ThresholdType {
        UNSUPPORTED,
        ITEM,
        FLUID,
        CUSTOM;
    }

    public ThresholdType getTypeOfCurrentTarget() {
        if (observedInventory.hasInventory())
            return ThresholdType.ITEM;
        if (observedTank.hasInventory())
            return ThresholdType.FLUID;
        if (level.getBlockEntity(getTargetPos()) instanceof ThresholdSwitchObservable)
            return ThresholdType.CUSTOM;
        return ThresholdType.UNSUPPORTED;
    }

    protected void scheduleBlockTick() {
        Block block = getBlockState().getBlock();
        if (!level.getBlockTicks().willTickThisTick(worldPosition, block))
            level.scheduleTick(worldPosition, block, 2, TickPriority.NORMAL);
    }

    @Override
    public void lazyTick() {
        super.lazyTick();
        if (level.isClientSide())
            return;
        updateCurrentLevel();
    }

    @Override
    public void addBehaviours(List<BlockEntityBehaviour<?>> behaviours) {
        behaviours.add(filtering = new ServerFilteringBehaviour(this).withCallback($ -> {
            this.updateCurrentLevel();
            invVersionTracker.reset();
        }));

        behaviours.add(invVersionTracker = new VersionedInventoryTrackerBehaviour(this));

        InterfaceProvider towardBlockFacing = (w, p, s) -> new BlockFace(p, DirectedDirectionalBlock.getTargetDirection(s));

        behaviours.add(observedInventory = new InvManipulationBehaviour(this, towardBlockFacing).bypassSidedness()
            .withFilter(this::isSuitableInventory));
        behaviours.add(observedTank = new TankManipulationBehaviour(this, towardBlockFacing).bypassSidedness());
    }

    public float getLevelForDisplay() {
        return currentLevel == -1 ? 0 : currentLevel;
    }

    public boolean getState() {
        return redstoneState;
    }

    public boolean shouldBePowered() {
        return inverted != redstoneState;
    }

    public void updatePowerAfterDelay() {
        poweredAfterDelay = shouldBePowered();
        level.updateNeighborsAt(worldPosition, getBlockState().getBlock());
        sendData();
    }

    public boolean isPowered() {
        return poweredAfterDelay;
    }

    public boolean isInverted() {
        return inverted;
    }

    public void setInverted(boolean inverted) {
        if (inverted == this.inverted)
            return;
        this.inverted = inverted;
        updatePowerAfterDelay();
    }
}
