package com.zurrtum.create.client.flywheel.lib.model;

import com.zurrtum.create.client.flywheel.api.material.Material;
import com.zurrtum.create.client.flywheel.api.model.Mesh;
import com.zurrtum.create.client.flywheel.api.model.Model;
import com.zurrtum.create.client.flywheel.api.vertex.VertexList;
import com.zurrtum.create.client.flywheel.lib.material.Materials;
import com.zurrtum.create.client.flywheel.lib.memory.MemoryBlock;
import com.zurrtum.create.client.flywheel.lib.vertex.PosVertexView;
import org.jetbrains.annotations.Nullable;
import org.joml.Vector3f;
import org.joml.Vector4f;

import java.util.Collection;
import net.minecraft.class_11515;
import net.minecraft.class_1921;
import net.minecraft.class_4722;

public final class ModelUtil {
    private static final float BOUNDING_SPHERE_EPSILON = 1e-4f;

    private ModelUtil() {
    }

    @Nullable
    public static Material getMaterial(class_11515 chunkRenderType, boolean shaded) {
        if (chunkRenderType == class_11515.field_60923) {
            return shaded ? Materials.SOLID_BLOCK : Materials.SOLID_UNSHADED_BLOCK;
        }
        if (chunkRenderType == class_11515.field_60924) {
            return shaded ? Materials.CUTOUT_MIPPED_BLOCK : Materials.CUTOUT_MIPPED_UNSHADED_BLOCK;
        }
        if (chunkRenderType == class_11515.field_60925) {
            return shaded ? Materials.CUTOUT_BLOCK : Materials.CUTOUT_UNSHADED_BLOCK;
        }
        if (chunkRenderType == class_11515.field_60926) {
            return shaded ? Materials.TRANSLUCENT_BLOCK : Materials.TRANSLUCENT_UNSHADED_BLOCK;
        }
        if (chunkRenderType == class_11515.field_60927) {
            return shaded ? Materials.TRIPWIRE_BLOCK : Materials.TRIPWIRE_UNSHADED_BLOCK;
        }
        return null;
    }

    @Nullable
    public static Material getItemMaterial(class_1921 renderType) {
        if (renderType == class_1921.method_23577()) {
            return Materials.SOLID_BLOCK;
        }
        if (renderType == class_1921.method_23579()) {
            return Materials.CUTOUT_MIPPED_BLOCK;
        }
        if (renderType == class_1921.method_23581()) {
            return Materials.CUTOUT_BLOCK;
        }
        if (renderType == class_1921.method_29997()) {
            return Materials.TRIPWIRE_BLOCK;
        }

        if (renderType == class_4722.method_24074()) {
            return Materials.CUTOUT_BLOCK;
        }

        if (renderType == class_4722.method_24073()) {
            return Materials.SOLID_BLOCK;
        }

        if (renderType == class_4722.method_29382()) {
            return Materials.TRANSLUCENT_ENTITY;
        }

        if (renderType == class_1921.method_23590()) {
            return Materials.GLINT;
        }
        if (renderType == class_1921.method_30676()) {
            return Materials.TRANSLUCENT_GLINT;
        }
        if (renderType == class_1921.method_23591()) {
            return Materials.GLINT_ENTITY;
        }
        return null;
    }

    public static int computeTotalVertexCount(Iterable<Mesh> meshes) {
        int vertexCount = 0;
        for (Mesh mesh : meshes) {
            vertexCount += mesh.vertexCount();
        }
        return vertexCount;
    }

    public static Vector4f computeBoundingSphere(Collection<Model.ConfiguredMesh> meshes) {
        return computeBoundingSphere(meshes.stream().map(Model.ConfiguredMesh::mesh).toList());
    }

    public static Vector4f computeBoundingSphere(Iterable<Mesh> meshes) {
        int vertexCount = computeTotalVertexCount(meshes);
        var block = MemoryBlock.malloc((long) vertexCount * PosVertexView.STRIDE);
        var vertexList = new PosVertexView();

        int baseVertex = 0;
        for (Mesh mesh : meshes) {
            vertexList.ptr(block.ptr() + (long) baseVertex * PosVertexView.STRIDE);
            vertexList.vertexCount(mesh.vertexCount());
            mesh.write(vertexList);
            baseVertex += mesh.vertexCount();
        }

        vertexList.ptr(block.ptr());
        vertexList.vertexCount(vertexCount);
        var sphere = computeBoundingSphere(vertexList);

        block.free();

        return sphere;
    }

    public static Vector4f computeBoundingSphere(VertexList vertexList) {
        var center = computeCenterOfAABBContaining(vertexList);

        var radius = computeMaxDistanceTo(vertexList, center) + BOUNDING_SPHERE_EPSILON;

        return new Vector4f(center, radius);
    }

    private static float computeMaxDistanceTo(VertexList vertexList, Vector3f pos) {
        float farthestDistanceSquared = -1;

        for (int i = 0; i < vertexList.vertexCount(); i++) {
            var distanceSquared = pos.distanceSquared(vertexList.x(i), vertexList.y(i), vertexList.z(i));

            if (distanceSquared > farthestDistanceSquared) {
                farthestDistanceSquared = distanceSquared;
            }
        }

        return (float) Math.sqrt(farthestDistanceSquared);
    }

    private static Vector3f computeCenterOfAABBContaining(VertexList vertexList) {
        var min = new Vector3f(Float.MAX_VALUE);
        var max = new Vector3f(Float.MIN_VALUE);

        for (int i = 0; i < vertexList.vertexCount(); i++) {
            float x = vertexList.x(i);
            float y = vertexList.y(i);
            float z = vertexList.z(i);

            // JOML's min/max methods don't accept floats :whywheel:
            min.x = Math.min(min.x, x);
            min.y = Math.min(min.y, y);
            min.z = Math.min(min.z, z);

            max.x = Math.max(max.x, x);
            max.y = Math.max(max.y, y);
            max.z = Math.max(max.z, z);
        }

        return min.add(max).mul(0.5f);
    }
}
