package com.zurrtum.create.client.flywheel.lib.visual;

import com.zurrtum.create.client.flywheel.api.visual.*;
import com.zurrtum.create.client.flywheel.api.visualization.VisualManager;
import com.zurrtum.create.client.flywheel.api.visualization.VisualizationContext;
import com.zurrtum.create.client.flywheel.lib.instance.FlatLit;
import com.zurrtum.create.client.flywheel.lib.math.MoreMath;
import it.unimi.dsi.fastutil.longs.LongSet;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.UnknownNullability;
import org.joml.FrustumIntersection;

import java.util.Iterator;
import net.minecraft.class_2338;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_4076;
import net.minecraft.class_761;

/**
 * The layer between a {@link class_2586} and the Flywheel backend.
 * <br>
 * <br> There are a few additional features that overriding classes can opt in to:
 * <ul>
 *     <li>{@link DynamicVisual}</li>
 *     <li>{@link TickableVisual}</li>
 *     <li>{@link LightUpdatedVisual}</li>
 *     <li>{@link ShaderLightVisual}</li>
 * </ul>
 * See the interfaces' documentation for more information about each one.
 *
 * <br> Implementing one or more of these will give an {@link AbstractBlockEntityVisual} access
 * to more interesting and regular points within a tick or a frame.
 *
 * @param <T> The type of {@link class_2586}.
 */
public abstract class AbstractBlockEntityVisual<T extends class_2586> extends AbstractVisual implements BlockEntityVisual<T>, LightUpdatedVisual {
    protected final T blockEntity;
    protected final class_2338 pos;
    protected final class_2338 visualPos;
    protected final class_2680 blockState;
    @UnknownNullability
    protected SectionCollector lightSections;

    public AbstractBlockEntityVisual(VisualizationContext ctx, T blockEntity, float partialTick) {
        super(ctx, blockEntity.method_10997(), partialTick);
        this.blockEntity = blockEntity;
        this.pos = blockEntity.method_11016();
        this.blockState = blockEntity.method_11010();
        this.visualPos = pos.method_10059(ctx.renderOrigin());
    }

    @Override
    public void setSectionCollector(SectionCollector sectionCollector) {
        this.lightSections = sectionCollector;
        lightSections.sections(LongSet.of(class_4076.method_33706(pos)));
    }

    /**
     * In order to accommodate for floating point precision errors at high coordinates,
     * {@link VisualManager}s are allowed to arbitrarily adjust the origin, and
     * shift the level matrix provided as a shader uniform accordingly.
     *
     * @return The {@link class_2338 position} of the {@link class_2586} this visual
     * represents should be rendered at to appear in the correct location.
     */
    public class_2338 getVisualPosition() {
        return visualPos;
    }

    /**
     * Check if this visual is within the given frustum.
     *
     * @param frustum The current frustum.
     * @return {@code true} if this visual is possibly visible.
     */
    public boolean isVisible(FrustumIntersection frustum) {
        float x = visualPos.method_10263() + 0.5f;
        float y = visualPos.method_10264() + 0.5f;
        float z = visualPos.method_10260() + 0.5f;
        // Default to checking a sphere exactly encompassing the block.
        return frustum.testSphere(x, y, z, MoreMath.SQRT_3_OVER_2);
    }

    /**
     * Limits which frames this visual is updated on based on its distance from the camera.
     * <p>
     * You may optionally do this check to avoid updating your visual every frame when it is far away.
     *
     * @param context The current frame context.
     * @return {@code true} if this visual shouldn't be updated this frame based on its distance from the camera.
     */
    public boolean doDistanceLimitThisFrame(DynamicVisual.Context context) {
        return !context.limiter().shouldUpdate(pos.method_19770(context.camera().method_19326()));
    }

    protected int computePackedLight() {
        return class_761.method_23794(level, pos);
    }

    protected void relight(class_2338 pos, @Nullable FlatLit... instances) {
        FlatLit.relight(class_761.method_23794(level, pos), instances);
    }

    protected void relight(@Nullable FlatLit... instances) {
        relight(pos, instances);
    }

    protected void relight(class_2338 pos, Iterator<@Nullable FlatLit> instances) {
        FlatLit.relight(class_761.method_23794(level, pos), instances);
    }

    protected void relight(Iterator<@Nullable FlatLit> instances) {
        relight(pos, instances);
    }

    protected void relight(class_2338 pos, Iterable<@Nullable FlatLit> instances) {
        FlatLit.relight(class_761.method_23794(level, pos), instances);
    }

    protected void relight(Iterable<@Nullable FlatLit> instances) {
        relight(pos, instances);
    }
}
