package com.zurrtum.create.client.flywheel.lib.visual;

import com.zurrtum.create.client.flywheel.lib.math.MoreMath;
import net.minecraft.class_1297;
import net.minecraft.class_238;
import net.minecraft.class_2382;
import net.minecraft.class_3532;
import org.jetbrains.annotations.Nullable;
import org.joml.FrustumIntersection;

/**
 * A helper class for testing whether an Entity is visible.
 * <p>
 * The last visible AABB is also checked to prevent the Entity from freezing when it goes offscreen.
 */
public class EntityVisibilityTester {
    private final class_1297 entity;
    private final class_2382 renderOrigin;
    private final float scale;
    @Nullable
    private class_238 lastVisibleAABB;

    /**
     * Create a new EntityVisibilityTester.
     *
     * @param entity       The Entity to test.
     * @param renderOrigin The render origin according to the VisualizationContext.
     * @param scale        Multiplier for the Entity's size, can be used to adjust for when
     *                     an entity's model is larger than its hitbox.
     */
    public EntityVisibilityTester(class_1297 entity, class_2382 renderOrigin, float scale) {
        this.entity = entity;
        this.renderOrigin = renderOrigin;
        this.scale = scale;
    }

    /**
     * Check whether the Entity is visible.
     *
     * @param frustum The frustum to test against.
     * @return {@code true} if the Entity is visible, {@code false} otherwise.
     */
    public boolean check(FrustumIntersection frustum) {
        class_238 aabb = entity.method_5829();

        // If we've never seen the entity before assume its visible.
        // Fixes entities freezing when they first spawn.
        // There might be a more sound solution to that, but this works.
        boolean visible = lastVisibleAABB == null;

        if (!visible) {
            visible = adjustAndTestAABB(frustum, aabb);
        }

        if (!visible && lastVisibleAABB != aabb) {
            // If the entity isn't visible, check the last visible AABB as well.
            // This is to avoid Entities freezing when the go offscreen.
            visible = adjustAndTestAABB(frustum, lastVisibleAABB);
        }

        if (visible) {
            lastVisibleAABB = aabb;
        }
        return visible;
    }

    private boolean adjustAndTestAABB(FrustumIntersection frustum, class_238 aabb) {
        float x = (float) class_3532.method_16436(0.5D, aabb.field_1323, aabb.field_1320) - renderOrigin.method_10263();
        float y = (float) class_3532.method_16436(0.5D, aabb.field_1322, aabb.field_1325) - renderOrigin.method_10264();
        float z = (float) class_3532.method_16436(0.5D, aabb.field_1321, aabb.field_1324) - renderOrigin.method_10260();
        float maxSize = (float) Math.max(aabb.method_17939(), Math.max(aabb.method_17940(), aabb.method_17941()));
        return frustum.testSphere(x, y, z, maxSize * MoreMath.SQRT_3_OVER_2 * scale);
    }
}
