package com.zurrtum.create.client.foundation.sound;

import com.zurrtum.create.AllSoundEvents;
import com.zurrtum.create.catnip.data.Pair;
import com.zurrtum.create.client.catnip.animation.AnimationTickHolder;
import com.zurrtum.create.client.infrastructure.config.AllConfigs;
import java.util.*;
import java.util.function.BiFunction;
import net.minecraft.class_1297;
import net.minecraft.class_2338;
import net.minecraft.class_310;
import net.minecraft.class_3417;

public class SoundScapes {

    private static final Map<AmbienceGroup, Map<PitchGroup, Set<class_2338>>> counter = new IdentityHashMap<>();
    private static final Map<Pair<AmbienceGroup, PitchGroup>, SoundScape> activeSounds = new HashMap<>();
    static final int MAX_AMBIENT_SOURCE_DISTANCE = 16;
    static final int UPDATE_INTERVAL = 5;
    static final int SOUND_VOLUME_ARG_MAX = 15;

    private static SoundScape kinetic(float pitch, AmbienceGroup group) {
        return new SoundScape(pitch, group).continuous(class_3417.field_14832, .25f, 1);
    }

    private static SoundScape cogwheel(float pitch, AmbienceGroup group) {
        return new SoundScape(pitch, group).continuous(AllSoundEvents.COGS.getMainEvent(), 1.5f, 1);
    }

    private static SoundScape crushing(float pitch, AmbienceGroup group) {
        return new SoundScape(pitch, group).repeating(AllSoundEvents.CRUSHING_1.getMainEvent(), 1.545f, .75f, 1)
            .repeating(AllSoundEvents.CRUSHING_2.getMainEvent(), 0.425f, .75f, 2).repeating(AllSoundEvents.CRUSHING_3.getMainEvent(), 2f, 1.75f, 2);
    }

    private static SoundScape milling(float pitch, AmbienceGroup group) {
        return new SoundScape(pitch, group).repeating(AllSoundEvents.CRUSHING_1.getMainEvent(), 1.545f, .75f, 1)
            .repeating(AllSoundEvents.CRUSHING_2.getMainEvent(), 0.425f, .75f, 2);
    }

    public static void play(AmbienceGroup group, class_2338 pos, float pitch) {
        if (!AllConfigs.client().enableAmbientSounds.get())
            return;
        if (!outOfRange(pos))
            addSound(group, pos, pitch);
    }

    public static void tick() {
        activeSounds.values().forEach(SoundScape::tick);

        if (AnimationTickHolder.getTicks() % UPDATE_INTERVAL != 0)
            return;

        boolean disable = !AllConfigs.client().enableAmbientSounds.get();
        for (Iterator<Map.Entry<Pair<AmbienceGroup, PitchGroup>, SoundScape>> iterator = activeSounds.entrySet().iterator(); iterator.hasNext(); ) {

            Map.Entry<Pair<AmbienceGroup, PitchGroup>, SoundScape> entry = iterator.next();
            Pair<AmbienceGroup, PitchGroup> key = entry.getKey();
            SoundScape value = entry.getValue();

            if (disable || getSoundCount(key.getFirst(), key.getSecond()) == 0) {
                value.remove();
                iterator.remove();
            }
        }

        counter.values().forEach(m -> m.values().forEach(Set::clear));
    }

    private static void addSound(AmbienceGroup group, class_2338 pos, float pitch) {
        PitchGroup groupFromPitch = getGroupFromPitch(pitch);
        Set<class_2338> set = counter.computeIfAbsent(group, ag -> new IdentityHashMap<>()).computeIfAbsent(groupFromPitch, pg -> new HashSet<>());
        set.add(pos);

        Pair<AmbienceGroup, PitchGroup> pair = Pair.of(group, groupFromPitch);
        activeSounds.computeIfAbsent(
            pair, $ -> {
                SoundScape soundScape = group.instantiate(pitch);
                soundScape.play();
                return soundScape;
            }
        );
    }

    public static void invalidateAll() {
        counter.clear();
        activeSounds.forEach(($, sound) -> sound.remove());
        activeSounds.clear();
    }

    protected static boolean outOfRange(class_2338 pos) {
        return !getCameraPos().method_19771(pos, MAX_AMBIENT_SOURCE_DISTANCE);
    }

    protected static class_2338 getCameraPos() {
        class_1297 renderViewEntity = class_310.method_1551().field_1719;
        if (renderViewEntity == null)
            return class_2338.field_10980;
        return renderViewEntity.method_24515();
    }

    public static int getSoundCount(AmbienceGroup group, PitchGroup pitchGroup) {
        return getAllLocations(group, pitchGroup).size();
    }

    public static Set<class_2338> getAllLocations(AmbienceGroup group, PitchGroup pitchGroup) {
        return counter.getOrDefault(group, Collections.emptyMap()).getOrDefault(pitchGroup, Collections.emptySet());
    }

    public static PitchGroup getGroupFromPitch(float pitch) {
        if (pitch < .70)
            return PitchGroup.VERY_LOW;
        if (pitch < .90)
            return PitchGroup.LOW;
        if (pitch < 1.10)
            return PitchGroup.NORMAL;
        if (pitch < 1.30)
            return PitchGroup.HIGH;
        return PitchGroup.VERY_HIGH;
    }

    public enum AmbienceGroup {

        KINETIC(SoundScapes::kinetic),
        COG(SoundScapes::cogwheel),
        CRUSHING(SoundScapes::crushing),
        MILLING(SoundScapes::milling),

        ;

        private final BiFunction<Float, AmbienceGroup, SoundScape> factory;

        AmbienceGroup(BiFunction<Float, AmbienceGroup, SoundScape> factory) {
            this.factory = factory;
        }

        public SoundScape instantiate(float pitch) {
            return factory.apply(pitch, this);
        }

    }

    public enum PitchGroup {
        VERY_LOW,
        LOW,
        NORMAL,
        HIGH,
        VERY_HIGH
    }

}