package com.zurrtum.create.content.contraptions.minecart.capability;

import com.zurrtum.create.AllSynchedDatas;
import com.zurrtum.create.Create;
import com.zurrtum.create.catnip.data.Couple;
import com.zurrtum.create.content.contraptions.AbstractContraptionEntity;
import com.zurrtum.create.content.contraptions.OrientedContraptionEntity;
import com.zurrtum.create.content.contraptions.minecart.CouplingHandler;
import org.apache.commons.lang3.mutable.MutableBoolean;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import net.minecraft.class_1297;
import net.minecraft.class_1688;
import net.minecraft.class_1695;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_243;
import net.minecraft.class_2680;
import net.minecraft.class_3481;
import net.minecraft.class_3532;
import net.minecraft.class_4844;
import net.minecraft.class_9129;
import net.minecraft.class_9135;
import net.minecraft.class_9139;

/**
 * Extended code for Minecarts, this allows for handling stalled carts and
 * coupled trains
 */
public class MinecartController {
    public static final class_9139<class_9129, MinecartController> PACKET_CODEC = class_9139.method_56435(
        Couple.streamCodec(StallData.PACKET_CODEC.method_56433(class_9135::method_56382)),
        i -> i.stallData,
        Couple.streamCodec(CouplingData.PACKET_CODEC.method_56433(class_9135::method_56382)),
        i -> i.couplings,
        MinecartController::new
    );
    private boolean needsEntryRefresh;
    private class_1688 cart;

    /*
     * Stall information, <Internal (waiting couplings), External (stalled
     * contraptions)>
     */
    private Couple<Optional<StallData>> stallData;

    /*
     * Coupling information, <Main (helmed by this cart), Connected (handled by
     * other cart)>
     */
    private Couple<Optional<CouplingData>> couplings;

    public MinecartController(class_1688 minecart) {
        cart = minecart;
        stallData = Couple.create(Optional::empty);
        couplings = Couple.create(Optional::empty);
        needsEntryRefresh = true;
    }

    private MinecartController(Couple<Optional<StallData>> stallData, Couple<Optional<CouplingData>> couplings) {
        this.stallData = stallData;
        this.couplings = couplings;
        needsEntryRefresh = true;
    }

    public class_1688 cart() {
        return cart;
    }

    public void coupleWith(boolean isLeading, UUID coupled, float length, boolean contraption) {
        UUID mainID = isLeading ? cart().method_5667() : coupled;
        UUID connectedID = isLeading ? coupled : cart().method_5667();
        couplings.set(isLeading, Optional.of(new CouplingData(mainID, connectedID, length, contraption)));
        needsEntryRefresh |= isLeading;
        sendData();
    }

    public void decouple() {
        couplings.forEachWithContext((opt, main) -> opt.ifPresent(cd -> {
            UUID idOfOther = cd.idOfCart(!main);
            MinecartController otherCart = CapabilityMinecartController.getIfPresent(cart.method_37908(), idOfOther);
            if (otherCart == null)
                return;

            removeConnection(main);
            otherCart.removeConnection(!main);
        }));
    }

    private void disassemble(class_1688 cart) {
        if (cart instanceof class_1695) {
            return;
        }
        List<class_1297> passengers = cart.method_5685();
        if (passengers.isEmpty() || !(passengers.getFirst() instanceof AbstractContraptionEntity)) {
            return;
        }
        class_1937 world = cart.method_37908();
        int i = class_3532.method_15357(cart.method_23317());
        int j = class_3532.method_15357(cart.method_23318());
        int k = class_3532.method_15357(cart.method_23321());
        if (world.method_8320(new class_2338(i, j - 1, k)).method_26164(class_3481.field_15463)) {
            --j;
        }
        class_2338 blockpos = new class_2338(i, j, k);
        class_2680 blockstate = world.method_8320(blockpos);
        if (blockstate.method_26164(class_3481.field_15463) && blockstate.method_26204() == class_2246.field_10546) {
            if (cart.method_5782()) {
                cart.method_5772();
            }

            if (cart.method_54295() == 0) {
                cart.method_54300(-cart.method_54296());
                cart.method_54299(10);
                cart.method_54297(50.0F);
                cart.field_6037 = true;
            }
        }
    }

    @Nullable
    public UUID getCoupledCart(boolean asMain) {
        Optional<CouplingData> optional = couplings.get(asMain);
        if (optional.isEmpty())
            return null;
        CouplingData couplingData = optional.get();
        return asMain ? couplingData.connectedCartID : couplingData.mainCartID;
    }

    public float getCouplingLength(boolean leading) {
        Optional<CouplingData> optional = couplings.get(leading);
        return optional.map(couplingData -> couplingData.length).orElse(0F);
    }

    public boolean hasContraptionCoupling(boolean current) {
        Optional<CouplingData> optional = couplings.get(current);
        return optional.isPresent() && optional.get().contraption;
    }

    public boolean isConnectedToCoupling() {
        return couplings.get(false).isPresent();
    }

    public boolean isCoupledThroughContraption() {
        return couplings.stream().anyMatch(i -> i.map(CouplingData::getContraption).orElse(false));
    }

    public boolean isFullyCoupled() {
        return isLeadingCoupling() && isConnectedToCoupling();
    }

    public boolean isLeadingCoupling() {
        return couplings.get(true).isPresent();
    }

    public boolean isPresent() {
        return cart.method_5805();
    }

    public boolean isStalled() {
        return isStalled(true) || isStalled(false);
    }

    private boolean isStalled(boolean internal) {
        return stallData.get(internal).isPresent();
    }

    public void prepareForCoupling(boolean isLeading) {
        // reverse existing chain if necessary
        if (isLeading && isLeadingCoupling() || !isLeading && isConnectedToCoupling()) {

            List<MinecartController> cartsToFlip = new ArrayList<>();
            cartsToFlip.add(this);
            MinecartController current = this;
            boolean forward = current.isLeadingCoupling();
            int safetyCount = 1000;

            class_1937 world = cart.method_37908();
            while (safetyCount-- > 0) {
                Optional<MinecartController> next = CouplingHandler.getNextInCouplingChain(world, current, forward);
                if (next.isEmpty()) {
                    break;
                }
                current = next.get();
                cartsToFlip.add(current);
            }
            if (safetyCount == 0) {
                Create.LOGGER.warn("Infinite loop in coupling iteration");
                return;
            }

            for (MinecartController minecartController : cartsToFlip) {
                minecartController.couplings.forEachWithContext((opt, leading) -> opt.ifPresent(cd -> {
                    cd.flip();
                    if (!cd.contraption)
                        return;
                    List<class_1297> passengers = minecartController.cart().method_5685();
                    if (passengers.isEmpty())
                        return;
                    class_1297 entity = passengers.getFirst();
                    if (!(entity instanceof OrientedContraptionEntity contraption))
                        return;
                    UUID couplingId = contraption.getCouplingId();
                    if (couplingId == cd.mainCartID) {
                        contraption.setCouplingId(cd.connectedCartID);
                        return;
                    }
                    if (couplingId == cd.connectedCartID) {
                        contraption.setCouplingId(cd.mainCartID);
                    }
                }));
                minecartController.couplings = minecartController.couplings.swap();
                minecartController.needsEntryRefresh = true;
                if (minecartController == this)
                    continue;
                minecartController.sendData();
            }
        }
    }

    public void removeConnection(boolean main) {
        if (hasContraptionCoupling(main)) {
            class_1937 world = cart.method_37908();
            if (world != null && !world.field_9236) {
                List<class_1297> passengers = cart().method_5685();
                if (!passengers.isEmpty()) {
                    class_1297 entity = passengers.getFirst();
                    if (entity instanceof AbstractContraptionEntity abstractContraptionEntity)
                        abstractContraptionEntity.disassemble();
                }
            }
        }

        couplings.set(main, Optional.empty());
        needsEntryRefresh |= main;
        sendData();
    }

    public void sendData() {
        sendData(null);
    }

    public void sendData(@Nullable class_1688 cart) {
        if (cart != null) {
            this.cart = cart;
            needsEntryRefresh = true;
        }
        if (this.cart.method_37908().field_9236) {
            return;
        }
        AllSynchedDatas.MINECART_CONTROLLER.set(this.cart, Optional.of(this), true);
    }

    public void setCart(class_1688 cart) {
        this.cart = cart;
    }

    private void setStalled(boolean stall, boolean internal) {
        if (isStalled(internal) == stall || cart == null)
            return;

        if (stall) {
            stallData.set(internal, Optional.of(new StallData(cart)));
            sendData();
            return;
        }

        if (!isStalled(!internal))
            stallData.get(internal).ifPresent(data -> data.release(cart));
        stallData.set(internal, Optional.empty());

        sendData();
    }

    public void setStalledExternally(boolean stall) {
        setStalled(stall, false);
    }

    public void tick() {
        if (cart == null) {
            return;
        }
        class_1937 world = cart.method_37908();
        if (world == null) {
            return;
        }

        if (needsEntryRefresh) {
            CapabilityMinecartController.queuedAdditions.get(world).add(cart);
            needsEntryRefresh = false;
        }

        stallData.forEach(opt -> opt.ifPresent(sd -> sd.tick(cart)));

        MutableBoolean internalStall = new MutableBoolean(false);
        couplings.forEachWithContext((opt, main) -> opt.ifPresent(cd -> {

            UUID idOfOther = cd.idOfCart(!main);
            MinecartController otherCart = CapabilityMinecartController.getIfPresent(world, idOfOther);
            internalStall.setValue(internalStall.booleanValue() || otherCart == null || !otherCart.isPresent() || otherCart.isStalled(false));

        }));
        if (!world.field_9236) {
            setStalled(internalStall.booleanValue(), true);
            disassemble(cart);
        }
    }

    private static class CouplingData {
        public static final class_9139<class_9129, CouplingData> PACKET_CODEC = class_9139.method_56905(
            class_4844.field_48453,
            i -> i.mainCartID,
            class_4844.field_48453,
            i -> i.connectedCartID,
            class_9135.field_48552,
            i -> i.length,
            class_9135.field_48547,
            i -> i.contraption,
            CouplingData::new
        );
        private final float length;
        private final boolean contraption;
        private UUID mainCartID;
        private UUID connectedCartID;

        public CouplingData(UUID mainCartID, UUID connectedCartID, float length, boolean contraption) {
            this.mainCartID = mainCartID;
            this.connectedCartID = connectedCartID;
            this.length = length;
            this.contraption = contraption;
        }

        public void flip() {
            UUID swap = mainCartID;
            mainCartID = connectedCartID;
            connectedCartID = swap;
        }

        public boolean getContraption() {
            return contraption;
        }

        public UUID idOfCart(boolean main) {
            return main ? mainCartID : connectedCartID;
        }
    }

    private record StallData(class_243 position, class_243 motion, float yaw, float pitch) {
        public static final class_9139<class_9129, StallData> PACKET_CODEC = class_9139.method_56905(
            class_243.field_52694,
            StallData::position,
            class_243.field_52694,
            StallData::motion,
            class_9135.field_48552,
            StallData::yaw,
            class_9135.field_48552,
            StallData::pitch,
            StallData::new
        );

        public StallData(class_1688 entity) {
            this(entity.method_19538(), entity.method_18798(), entity.method_36454(), entity.method_36455());
            tick(entity);
        }

        public void release(class_1688 entity) {
            entity.method_18799(motion);
        }

        public void tick(class_1688 entity) {
            entity.method_18799(class_243.field_1353);
            entity.method_36456(yaw);
            entity.method_36457(pitch);
        }
    }
}
