package com.zurrtum.create.content.kinetics.belt.transport;

import com.zurrtum.create.AllClientHandle;
import com.zurrtum.create.content.kinetics.belt.BeltBlock;
import com.zurrtum.create.content.kinetics.belt.BeltBlockEntity;
import com.zurrtum.create.content.kinetics.belt.BeltHelper;
import com.zurrtum.create.content.kinetics.belt.BeltSlope;
import com.zurrtum.create.content.kinetics.belt.behaviour.BeltProcessingBehaviour;
import com.zurrtum.create.content.kinetics.belt.behaviour.BeltProcessingBehaviour.ProcessingResult;
import com.zurrtum.create.content.kinetics.belt.behaviour.DirectBeltInputBehaviour;
import com.zurrtum.create.content.kinetics.belt.behaviour.TransportedItemStackHandlerBehaviour;
import com.zurrtum.create.content.kinetics.belt.behaviour.TransportedItemStackHandlerBehaviour.TransportedResult;
import com.zurrtum.create.foundation.blockEntity.behaviour.BlockEntityBehaviour;
import com.zurrtum.create.foundation.utility.BlockHelper;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.function.Function;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1542;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_243;
import net.minecraft.class_3532;

public class BeltInventory {

    final BeltBlockEntity belt;
    private final List<TransportedItemStack> items;
    final List<TransportedItemStack> toInsert;
    final List<TransportedItemStack> toRemove;
    boolean beltMovementPositive;

    TransportedItemStack lazyClientItem;

    public BeltInventory(BeltBlockEntity be) {
        this.belt = be;
        items = new LinkedList<>();
        toInsert = new LinkedList<>();
        toRemove = new LinkedList<>();
    }

    public void tick() {

        // Residual item for "smooth" transitions
        if (lazyClientItem != null) {
            if (lazyClientItem.locked)
                lazyClientItem = null;
            else
                lazyClientItem.locked = true;
        }

        // Added/Removed items from previous cycle
        if (!toInsert.isEmpty() || !toRemove.isEmpty()) {
            toInsert.forEach(this::insert);
            toInsert.clear();
            items.removeAll(toRemove);
            toRemove.clear();
            belt.notifyUpdate();
        }

        if (belt.getSpeed() == 0)
            return;

        // Reverse item collection if belt just reversed
        if (beltMovementPositive != belt.getDirectionAwareBeltMovementSpeed() > 0) {
            beltMovementPositive = !beltMovementPositive;
            Collections.reverse(items);
            belt.notifyUpdate();
        }

        // Assuming the first entry is furthest on the belt
        TransportedItemStack stackInFront = null;
        TransportedItemStack currentItem = null;
        Iterator<TransportedItemStack> iterator = items.iterator();

        // Useful stuff
        float beltSpeed = belt.getDirectionAwareBeltMovementSpeed();
        class_2350 movementFacing = belt.getMovementFacing();
        boolean horizontal = belt.method_11010().method_11654(BeltBlock.SLOPE) == BeltSlope.HORIZONTAL;
        float spacing = 1;
        class_1937 world = belt.method_10997();
        boolean onClient = world.field_9236 && !belt.isVirtual();

        // resolve ending only when items will reach it this tick
        Ending ending = Ending.UNRESOLVED;

        // Loop over items
        while (iterator.hasNext()) {
            stackInFront = currentItem;
            currentItem = iterator.next();
            currentItem.prevBeltPosition = currentItem.beltPosition;
            currentItem.prevSideOffset = currentItem.sideOffset;

            if (currentItem.stack.method_7960()) {
                iterator.remove();
                currentItem = null;
                continue;
            }

            float movement = beltSpeed;
            if (onClient)
                movement *= AllClientHandle.INSTANCE.getServerSpeed();

            // Don't move if held by processing (client)
            if (world.field_9236 && currentItem.locked)
                continue;

            // Don't move if held by external components
            if (currentItem.lockedExternally) {
                currentItem.lockedExternally = false;
                continue;
            }

            // Don't move if other items are waiting in front
            boolean noMovement = false;
            float currentPos = currentItem.beltPosition;
            if (stackInFront != null) {
                float diff = stackInFront.beltPosition - currentPos;
                if (Math.abs(diff) <= spacing)
                    noMovement = true;
                movement = beltMovementPositive ? Math.min(movement, diff - spacing) : Math.max(movement, diff + spacing);
            }

            // Don't move beyond the edge
            float diffToEnd = beltMovementPositive ? belt.beltLength - currentPos : -currentPos;
            if (Math.abs(diffToEnd) < Math.abs(movement) + 1) {
                if (ending == Ending.UNRESOLVED)
                    ending = resolveEnding();
                diffToEnd += beltMovementPositive ? -ending.margin : ending.margin;
            }
            float limitedMovement = beltMovementPositive ? Math.min(movement, diffToEnd) : Math.max(movement, diffToEnd);
            float nextOffset = currentItem.beltPosition + limitedMovement;

            // Belt item processing
            if (!onClient && horizontal) {
                class_1799 item = currentItem.stack;
                if (handleBeltProcessingAndCheckIfRemoved(currentItem, nextOffset, noMovement)) {
                    iterator.remove();
                    belt.notifyUpdate();
                    continue;
                }
                if (item != currentItem.stack)
                    belt.notifyUpdate();
                if (currentItem.locked)
                    continue;
            }

            // Belt Funnels
            if (BeltFunnelInteractionHandler.checkForFunnels(this, currentItem, nextOffset))
                continue;

            if (noMovement)
                continue;

            // Belt Tunnels
            if (BeltTunnelInteractionHandler.flapTunnelsAndCheckIfStuck(this, currentItem, nextOffset))
                continue;

            // Horizontal Crushing Wheels
            if (BeltCrusherInteractionHandler.checkForCrushers(this, world.field_9236, currentItem, nextOffset))
                continue;

            // Apply Movement
            currentItem.beltPosition += limitedMovement;
            float diffToMiddle = currentItem.getTargetSideOffset() - currentItem.sideOffset;
            currentItem.sideOffset += class_3532.method_15363(
                diffToMiddle * Math.abs(limitedMovement) * 6f,
                -Math.abs(diffToMiddle),
                Math.abs(diffToMiddle)
            );
            currentPos = currentItem.beltPosition;

            // Movement successful
            if (limitedMovement == movement || onClient)
                continue;

            // End reached
            int lastOffset = beltMovementPositive ? belt.beltLength - 1 : 0;
            class_2338 nextPosition = BeltHelper.getPositionForOffset(belt, beltMovementPositive ? belt.beltLength : -1);

            if (ending == Ending.FUNNEL)
                continue;

            if (ending == Ending.INSERT) {
                DirectBeltInputBehaviour inputBehaviour = BlockEntityBehaviour.get(world, nextPosition, DirectBeltInputBehaviour.TYPE);
                if (inputBehaviour == null)
                    continue;
                if (!inputBehaviour.canInsertFromSide(movementFacing))
                    continue;

                class_1799 remainder = inputBehaviour.handleInsertion(currentItem, movementFacing, false);
                if (class_1799.method_7973(remainder, currentItem.stack))
                    continue;

                currentItem.stack = remainder;
                if (remainder.method_7960()) {
                    lazyClientItem = currentItem;
                    lazyClientItem.locked = false;
                    iterator.remove();
                } else
                    currentItem.stack = remainder;

                BeltTunnelInteractionHandler.flapTunnel(this, lastOffset, movementFacing, false);
                belt.notifyUpdate();
                continue;
            }

            if (ending == Ending.BLOCKED)
                continue;

            if (ending == Ending.EJECT) {
                eject(currentItem);
                iterator.remove();
                BeltTunnelInteractionHandler.flapTunnel(this, lastOffset, movementFacing, false);
                belt.notifyUpdate();
            }
        }
    }

    protected boolean handleBeltProcessingAndCheckIfRemoved(TransportedItemStack currentItem, float nextOffset, boolean noMovement) {
        int currentSegment = (int) currentItem.beltPosition;

        // Continue processing if held
        if (currentItem.locked) {
            BeltProcessingBehaviour processingBehaviour = getBeltProcessingAtSegment(currentSegment);
            TransportedItemStackHandlerBehaviour stackHandlerBehaviour = getTransportedItemStackHandlerAtSegment(currentSegment);

            if (stackHandlerBehaviour == null)
                return false;
            if (processingBehaviour == null) {
                currentItem.locked = false;
                belt.notifyUpdate();
                return false;
            }

            ProcessingResult result = processingBehaviour.handleHeldItem(currentItem, stackHandlerBehaviour);
            if (result == ProcessingResult.REMOVE)
                return true;
            if (result == ProcessingResult.HOLD)
                return false;

            currentItem.locked = false;
            belt.notifyUpdate();
            return false;
        }

        if (noMovement)
            return false;

        // See if any new belt processing catches the item
        if (currentItem.beltPosition > .5f || beltMovementPositive) {
            int firstUpcomingSegment = (int) (currentItem.beltPosition + (beltMovementPositive ? .5f : -.5f));
            int step = beltMovementPositive ? 1 : -1;

            for (int segment = firstUpcomingSegment; beltMovementPositive ? segment + .5f <= nextOffset : segment + .5f >= nextOffset; segment += step) {

                BeltProcessingBehaviour processingBehaviour = getBeltProcessingAtSegment(segment);
                TransportedItemStackHandlerBehaviour stackHandlerBehaviour = getTransportedItemStackHandlerAtSegment(segment);

                if (processingBehaviour == null)
                    continue;
                if (stackHandlerBehaviour == null)
                    continue;
                if (BeltProcessingBehaviour.isBlocked(belt.method_10997(), BeltHelper.getPositionForOffset(belt, segment)))
                    continue;

                ProcessingResult result = processingBehaviour.handleReceivedItem(currentItem, stackHandlerBehaviour);
                if (result == ProcessingResult.REMOVE)
                    return true;

                if (result == ProcessingResult.HOLD) {
                    currentItem.beltPosition = segment + .5f + (beltMovementPositive ? 1 / 512f : -1 / 512f);
                    currentItem.locked = true;
                    belt.notifyUpdate();
                    return false;
                }
            }
        }

        return false;
    }

    protected BeltProcessingBehaviour getBeltProcessingAtSegment(int segment) {
        return BlockEntityBehaviour.get(belt.method_10997(), BeltHelper.getPositionForOffset(belt, segment).method_10086(2), BeltProcessingBehaviour.TYPE);
    }

    protected TransportedItemStackHandlerBehaviour getTransportedItemStackHandlerAtSegment(int segment) {
        return BlockEntityBehaviour.get(belt.method_10997(), BeltHelper.getPositionForOffset(belt, segment), TransportedItemStackHandlerBehaviour.TYPE);
    }

    private enum Ending {
        UNRESOLVED(0),
        EJECT(0),
        INSERT(.25f),
        FUNNEL(.5f),
        BLOCKED(.45f);

        private float margin;

        Ending(float f) {
            this.margin = f;
        }
    }

    private Ending resolveEnding() {
        class_1937 world = belt.method_10997();
        class_2338 nextPosition = BeltHelper.getPositionForOffset(belt, beltMovementPositive ? belt.beltLength : -1);

        //		if (AllBlocks.BRASS_BELT_FUNNEL.has(world.getBlockState(lastPosition.up())))
        //			return Ending.FUNNEL;

        DirectBeltInputBehaviour inputBehaviour = BlockEntityBehaviour.get(world, nextPosition, DirectBeltInputBehaviour.TYPE);
        if (inputBehaviour != null)
            return Ending.INSERT;

        if (BlockHelper.hasBlockSolidSide(world.method_8320(nextPosition), world, nextPosition, belt.getMovementFacing().method_10153()))
            return Ending.BLOCKED;

        return Ending.EJECT;
    }

    //

    public boolean canInsertAt(int segment) {
        return canInsertAtFromSide(segment, class_2350.field_11036);
    }

    public boolean canInsertAtFromSide(int segment, class_2350 side) {
        float segmentPos = segment;
        if (belt.getMovementFacing() == side.method_10153())
            return false;
        if (belt.getMovementFacing() != side)
            segmentPos += .5f;
        else if (!beltMovementPositive)
            segmentPos += 1f;

        for (TransportedItemStack stack : items)
            if (isBlocking(segment, side, segmentPos, stack))
                return false;
        for (TransportedItemStack stack : toInsert)
            if (isBlocking(segment, side, segmentPos, stack))
                return false;

        return true;
    }

    private boolean isBlocking(int segment, class_2350 side, float segmentPos, TransportedItemStack stack) {
        float currentPos = stack.beltPosition;
        if (stack.insertedAt == segment && stack.insertedFrom == side && (beltMovementPositive ? currentPos <= segmentPos + 1 : currentPos >= segmentPos - 1))
            return true;
        return false;
    }

    public void addItem(TransportedItemStack newStack) {
        toInsert.add(newStack);
    }

    private void insert(TransportedItemStack newStack) {
        if (items.isEmpty())
            items.add(newStack);
        else {
            int index = 0;
            for (TransportedItemStack stack : items) {
                if (stack.compareTo(newStack) > 0 == beltMovementPositive)
                    break;
                index++;
            }
            items.add(index, newStack);
        }
    }

    public TransportedItemStack getStackAtOffset(int offset) {
        float min = offset;
        float max = offset + 1;
        for (TransportedItemStack stack : items) {
            if (toRemove.contains(stack))
                continue;
            if (stack.beltPosition > max)
                continue;
            if (stack.beltPosition > min)
                return stack;
        }
        return null;
    }

    public void read(class_11368 view) {
        items.clear();
        class_11368.class_11369<TransportedItemStack> list = view.method_71437("Items", TransportedItemStack.CODEC);
        list.forEach(items::add);
        lazyClientItem = view.method_71426("LazyItem", TransportedItemStack.CODEC).orElse(null);
        beltMovementPositive = view.method_71433("PositiveOrder", false);
    }

    public void write(class_11372 view) {
        class_11372.class_11373<TransportedItemStack> list = view.method_71467("Items", TransportedItemStack.CODEC);
        items.forEach(list::method_71484);
        if (lazyClientItem != null)
            view.method_71468("LazyItem", TransportedItemStack.CODEC, lazyClientItem);
        view.method_71472("PositiveOrder", beltMovementPositive);
    }

    public void eject(TransportedItemStack stack) {
        class_1799 ejected = stack.stack;
        class_243 outPos = BeltHelper.getVectorForOffset(belt, stack.beltPosition);
        float movementSpeed = Math.max(Math.abs(belt.getBeltMovementSpeed()), 1 / 8f);
        class_243 outMotion = class_243.method_24954(belt.getBeltChainDirection()).method_1021(movementSpeed).method_1031(0, 1 / 8f, 0);
        outPos = outPos.method_1019(outMotion.method_1029().method_1021(0.001));
        class_1542 entity = new class_1542(belt.method_10997(), outPos.field_1352, outPos.field_1351 + 6 / 16f, outPos.field_1350, ejected);
        entity.method_18799(outMotion);
        entity.method_6988();
        entity.field_6037 = true;
        belt.method_10997().method_8649(entity);
    }

    public void ejectAll() {
        items.forEach(this::eject);
        items.clear();
    }

    public void applyToEachWithin(float position, float maxDistanceToPosition, Function<TransportedItemStack, TransportedResult> processFunction) {
        boolean dirty = false;
        for (TransportedItemStack transported : items) {
            if (toRemove.contains(transported))
                continue;
            class_1799 stackBefore = transported.stack.method_7972();
            if (Math.abs(position - transported.beltPosition) >= maxDistanceToPosition)
                continue;
            TransportedResult result = processFunction.apply(transported);
            if (result == null || result.didntChangeFrom(stackBefore))
                continue;

            dirty = true;
            if (result.hasHeldOutput()) {
                TransportedItemStack held = result.getHeldOutput();
                held.beltPosition = ((int) position) + .5f - (beltMovementPositive ? 1 / 512f : -1 / 512f);
                toInsert.add(held);
            }
            toInsert.addAll(result.getOutputs());
            toRemove.add(transported);
        }
        if (dirty) {
            belt.notifyUpdate();
        }
    }

    public List<TransportedItemStack> getTransportedItems() {
        return items;
    }

    @Nullable
    public TransportedItemStack getLazyClientItem() {
        return lazyClientItem;
    }

}
