package com.zurrtum.create.content.kinetics.drill;

import com.zurrtum.create.AllBlocks;
import com.zurrtum.create.AllFluids;
import java.util.*;
import java.util.function.Function;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2404;
import net.minecraft.class_2680;
import net.minecraft.class_3609;
import net.minecraft.class_3610;
import net.minecraft.class_3611;
import net.minecraft.class_3612;

/**
 * A registry which defines the interactions a source fluid can have with its
 * surroundings. Each possible flow direction is checked for all interactions with
 * the source.
 *
 * <p>Fluid interactions mimic the behavior of {@code FluidBlock#receiveNeighborFluids}.
 * As such, all directions, besides {@link class_2350#field_11033} is tested and then replaced.
 * Any fluids which cause a change in the down interaction must be handled in
 * {@code FlowingFluid#flow} and not by this interaction manager.
 */
public final class FluidInteractionRegistry {
    public static final Map<class_3611, List<InteractionInformation>> INTERACTIONS = new HashMap<>();

    /**
     * Adds an interaction between a source and its surroundings.
     *
     * @param source      the source of the interaction, this will be replaced if the interaction occurs
     * @param interaction the interaction data to check and perform
     */
    public static synchronized void addInteraction(class_3611 source, InteractionInformation interaction) {
        if (source instanceof class_3609 flowableFluid) {
            INTERACTIONS.computeIfAbsent(flowableFluid.method_15751(), s -> new ArrayList<>()).add(interaction);
            INTERACTIONS.computeIfAbsent(flowableFluid.method_15750(), s -> new ArrayList<>()).add(interaction);
        } else {
            INTERACTIONS.computeIfAbsent(source, s -> new ArrayList<>()).add(interaction);
        }
    }

    /**
     * Performs all potential fluid interactions at a given position.
     *
     * <p>Note: Only the first interaction check that succeeds will occur.
     *
     * @param level the level the interactions take place in
     * @param pos   the position of the source fluid
     * @return {@code true} if an interaction took place, {@code false} otherwise
     */
    public static boolean canInteract(class_1937 level, class_2338 pos) {
        class_3610 state = level.method_8316(pos);
        class_3611 fluid = state.method_15772();
        for (class_2350 direction : class_2404.field_34006) {
            class_2338 relativePos = pos.method_10093(direction.method_10153());
            List<InteractionInformation> interactions = INTERACTIONS.getOrDefault(fluid, Collections.emptyList());
            for (InteractionInformation interaction : interactions) {
                if (interaction.predicate().test(level, pos, relativePos, state)) {
                    interaction.interaction().interact(level, pos, relativePos, state);
                    return true;
                }
            }
        }

        return false;
    }

    static {
        // Lava + Water = Obsidian (Source Lava) / Cobblestone (Flowing Lava)
        addInteraction(
            class_3612.field_15908,
            new InteractionInformation(
                class_3612.field_15910,
                fluidState -> fluidState.method_15771() ? class_2246.field_10540.method_9564() : class_2246.field_10445.method_9564()
            )
        );

        // Lava + Soul Soil (Below) + Blue Ice = Basalt
        addInteraction(
            class_3612.field_15908,
            new InteractionInformation(
                (level, currentPos, relativePos, currentState) -> level.method_8320(currentPos.method_10074())
                    .method_27852(class_2246.field_22090) && level.method_8320(relativePos).method_27852(class_2246.field_10384), class_2246.field_22091.method_9564()
            )
        );
        addInteraction(
            class_3612.field_15908, new InteractionInformation(
                AllFluids.HONEY, fluidState -> {
                if (fluidState.method_15771()) {
                    return class_2246.field_10540.method_9564();
                } else {
                    return AllBlocks.LIMESTONE.method_9564();
                }
            }
            )
        );
        addInteraction(
            class_3612.field_15908, new InteractionInformation(
                AllFluids.CHOCOLATE, fluidState -> {
                if (fluidState.method_15771()) {
                    return class_2246.field_10540.method_9564();
                } else {
                    return AllBlocks.SCORIA.method_9564();
                }
            }
            )
        );
    }

    /**
     * Holds the interaction data for a given source type on when to succeed
     * and what to perform.
     *
     * @param predicate   a test to see whether an interaction can occur
     * @param interaction the interaction to perform
     */
    public record InteractionInformation(HasFluidInteraction predicate, FluidInteraction interaction) {
        /**
         * Constructor which checks the surroundings fluids for a specific type
         * and then transforms the source state into a block.
         *
         * @param type  the type of the fluid that must be surrounding the source
         * @param state the state of the block replacing the source
         */
        public InteractionInformation(class_3611 type, class_2680 state) {
            this(type, fluidState -> state);
        }

        /**
         * Constructor which transforms the source state into a block.
         *
         * @param predicate a test to see whether an interaction can occur
         * @param state     the state of the block replacing the source
         */
        public InteractionInformation(HasFluidInteraction predicate, class_2680 state) {
            this(predicate, fluidState -> state);
        }

        /**
         * Constructor which checks the surroundings fluids for a specific type
         * and then transforms the source state into a block.
         *
         * @param type     the type of the fluid that must be surrounding the source
         * @param getState a function to transform the source fluid into a block state
         */
        public InteractionInformation(class_3611 type, Function<class_3610, class_2680> getState) {
            this(
                (level, currentPos, relativePos, currentState) -> {
                    class_3610 state = level.method_8316(relativePos);
                    class_3611 fluid = state.method_15772();
                    if (!fluid.method_15793(state) && fluid instanceof class_3609 flowableFluid) {
                        fluid = flowableFluid.method_15751();
                    }
                    return fluid == type;
                }, getState
            );
        }

        /**
         * Constructor which transforms the source state into a block.
         *
         * @param predicate a test to see whether an interaction can occur
         * @param getState  a function to transform the source fluid into a block state
         */
        public InteractionInformation(HasFluidInteraction predicate, Function<class_3610, class_2680> getState) {
            this(
                predicate, (level, currentPos, relativePos, currentState) -> {
                    level.method_8501(currentPos, getState.apply(currentState));
                    level.method_20290(1501, currentPos, 0);
                }
            );
        }
    }

    /**
     * An interface which tests whether a source fluid can interact with its
     * surroundings.
     */
    @FunctionalInterface
    public interface HasFluidInteraction {
        /**
         * Returns whether the interaction can occur.
         *
         * @param level        the level the interaction takes place in
         * @param currentPos   the position of the source
         * @param relativePos  a position surrounding the source
         * @param currentState the state of the fluid surrounding the source
         * @return {@code true} if an interaction can occur, {@code false} otherwise
         */
        boolean test(class_1937 level, class_2338 currentPos, class_2338 relativePos, class_3610 currentState);
    }

    /**
     * An interface which performs an interaction for a source.
     */
    @FunctionalInterface
    public interface FluidInteraction {
        /**
         * Performs the interaction between the source and the surrounding data.
         *
         * @param level        the level the interaction takes place in
         * @param currentPos   the position of the source
         * @param relativePos  a position surrounding the source
         * @param currentState the state of the fluid surrounding the source
         */
        void interact(class_1937 level, class_2338 currentPos, class_2338 relativePos, class_3610 currentState);
    }
}