package com.zurrtum.create.content.processing.basin;

import com.google.common.collect.ImmutableList;
import com.zurrtum.create.AllBlockEntityTypes;
import com.zurrtum.create.AllBlockTags;
import com.zurrtum.create.AllClientHandle;
import com.zurrtum.create.catnip.animation.LerpedFloat;
import com.zurrtum.create.catnip.animation.LerpedFloat.Chaser;
import com.zurrtum.create.catnip.data.Couple;
import com.zurrtum.create.catnip.data.IntAttached;
import com.zurrtum.create.catnip.data.Iterate;
import com.zurrtum.create.content.kinetics.belt.behaviour.DirectBeltInputBehaviour;
import com.zurrtum.create.content.kinetics.mixer.MechanicalMixerBlockEntity;
import com.zurrtum.create.content.processing.burner.BlazeBurnerBlock;
import com.zurrtum.create.content.processing.burner.BlazeBurnerBlock.HeatLevel;
import com.zurrtum.create.foundation.blockEntity.SmartBlockEntity;
import com.zurrtum.create.foundation.blockEntity.behaviour.BlockEntityBehaviour;
import com.zurrtum.create.foundation.blockEntity.behaviour.filtering.ServerFilteringBehaviour;
import com.zurrtum.create.foundation.blockEntity.behaviour.fluid.SmartFluidTankBehaviour;
import com.zurrtum.create.foundation.blockEntity.behaviour.fluid.SmartFluidTankBehaviour.TankSegment;
import com.zurrtum.create.foundation.blockEntity.behaviour.inventory.InvManipulationBehaviour;
import com.zurrtum.create.foundation.codec.CreateCodecs;
import com.zurrtum.create.foundation.fluid.FluidHelper;
import com.zurrtum.create.foundation.item.ItemHelper;
import com.zurrtum.create.foundation.utility.BlockHelper;
import com.zurrtum.create.infrastructure.fluids.BucketFluidInventory;
import com.zurrtum.create.infrastructure.fluids.FluidInventory;
import com.zurrtum.create.infrastructure.fluids.FluidStack;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1263;
import net.minecraft.class_1264;
import net.minecraft.class_1799;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2586;
import net.minecraft.class_2680;

public class BasinBlockEntity extends SmartBlockEntity {
    public boolean areFluidsMoving;
    LerpedFloat ingredientRotationSpeed;
    public LerpedFloat ingredientRotation;

    public SmartFluidTankBehaviour inputTank;
    protected SmartFluidTankBehaviour outputTank;
    private ServerFilteringBehaviour filtering;
    private boolean contentsChanged;

    private Couple<SmartFluidTankBehaviour> tanks;

    public BasinInventory itemCapability;
    public BasinFluidHandler fluidCapability;

    List<class_2350> disabledSpoutputs;
    class_2350 preferredSpoutput;
    protected List<class_1799> spoutputBuffer;
    protected List<FluidStack> spoutputFluidBuffer;
    int recipeBackupCheck;

    public static final int OUTPUT_ANIMATION_TIME = 10;
    public List<IntAttached<class_1799>> visualizedOutputItems;
    public List<IntAttached<FluidStack>> visualizedOutputFluids;

    private @Nullable HeatLevel cachedHeatLevel;

    public BasinBlockEntity(class_2338 pos, class_2680 state) {
        super(AllBlockEntityTypes.BASIN, pos, state);
        areFluidsMoving = false;
        itemCapability = new BasinInventory(this);
        contentsChanged = true;
        ingredientRotation = LerpedFloat.angular().startWithValue(0);
        ingredientRotationSpeed = LerpedFloat.linear().startWithValue(0);

        tanks = Couple.create(inputTank, outputTank);
        visualizedOutputItems = Collections.synchronizedList(new ArrayList<>());
        visualizedOutputFluids = Collections.synchronizedList(new ArrayList<>());
        disabledSpoutputs = new ArrayList<>();
        preferredSpoutput = null;
        spoutputBuffer = new ArrayList<>();
        spoutputFluidBuffer = new ArrayList<>();
        recipeBackupCheck = 20;
    }

    @Override
    public void addBehaviours(List<BlockEntityBehaviour<?>> behaviours) {
        behaviours.add(new DirectBeltInputBehaviour(this));
        filtering = new ServerFilteringBehaviour(this).withCallback(newFilter -> contentsChanged = true).forRecipes();
        behaviours.add(filtering);

        inputTank = new SmartFluidTankBehaviour(
            SmartFluidTankBehaviour.INPUT,
            this,
            2,
            BucketFluidInventory.CAPACITY,
            true
        ).whenFluidUpdates(() -> contentsChanged = true);
        outputTank = new SmartFluidTankBehaviour(
            SmartFluidTankBehaviour.OUTPUT,
            this,
            2,
            BucketFluidInventory.CAPACITY,
            true
        ).whenFluidUpdates(() -> contentsChanged = true).forbidInsertion();
        behaviours.add(inputTank);
        behaviours.add(outputTank);

        fluidCapability = new BasinFluidHandler(outputTank.getTanks(), inputTank.getTanks());
    }

    @Override
    protected void read(class_11368 view, boolean clientPacket) {
        super.read(view, clientPacket);
        itemCapability.read(view);

        preferredSpoutput = view.method_71426("PreferredSpoutput", class_2350.field_29502).orElse(null);
        disabledSpoutputs.clear();
        spoutputBuffer.clear();
        spoutputFluidBuffer.clear();
        view.method_71426("DisabledSpoutput", CreateCodecs.DIRECTION_LIST_CODEC).ifPresent(disabledSpoutputs::addAll);
        view.method_71426("Overflow", CreateCodecs.ITEM_LIST_CODEC).ifPresent(spoutputBuffer::addAll);
        view.method_71426("FluidOverflow", CreateCodecs.FLUID_LIST_CODEC).ifPresent(spoutputFluidBuffer::addAll);

        if (!clientPacket)
            return;

        view.method_71437("VisualizedItems", class_1799.field_49266).method_71456()
            .forEach(stack -> visualizedOutputItems.add(IntAttached.with(OUTPUT_ANIMATION_TIME, stack)));
        view.method_71437("VisualizedFluids", FluidStack.OPTIONAL_CODEC).method_71456()
            .forEach(stack -> visualizedOutputFluids.add(IntAttached.with(OUTPUT_ANIMATION_TIME, stack)));
    }

    @Override
    public void write(class_11372 view, boolean clientPacket) {
        super.write(view, clientPacket);
        itemCapability.write(view);

        if (preferredSpoutput != null)
            view.method_71468("PreferredSpoutput", class_2350.field_29502, preferredSpoutput);
        view.method_71468("DisabledSpoutput", CreateCodecs.DIRECTION_LIST_CODEC, disabledSpoutputs);
        view.method_71468("Overflow", CreateCodecs.ITEM_LIST_CODEC, spoutputBuffer);
        view.method_71468("FluidOverflow", CreateCodecs.FLUID_LIST_CODEC, spoutputFluidBuffer);

        if (!clientPacket)
            return;

        class_11372.class_11373<class_1799> items = view.method_71467("VisualizedItems", class_1799.field_49266);
        visualizedOutputItems.stream().map(IntAttached::getValue).forEach(items::method_71484);
        class_11372.class_11373<FluidStack> fluids = view.method_71467("VisualizedFluids", FluidStack.OPTIONAL_CODEC);
        visualizedOutputFluids.stream().map(IntAttached::getValue).forEach(fluids::method_71484);
        visualizedOutputItems.clear();
        visualizedOutputFluids.clear();
    }

    @Override
    public void destroy() {
        super.destroy();
        class_1264.method_5451(field_11863, field_11867, itemCapability);
        spoutputBuffer.forEach(is -> class_2248.method_9577(field_11863, field_11867, is));
    }

    @Override
    public void remove() {
        super.remove();
        onEmptied();
    }

    public void onEmptied() {
        getOperator().ifPresent(be -> be.basinRemoved = true);
    }

    @Override
    public void lazyTick() {
        super.lazyTick();

        if (!field_11863.field_9236) {
            updateSpoutput();
            if (recipeBackupCheck-- > 0)
                return;
            recipeBackupCheck = 20;
            if (isEmpty())
                return;
            notifyChangeOfContents();
            return;
        }

        class_2586 blockEntity = field_11863.method_8321(field_11867.method_10086(2));
        if (!(blockEntity instanceof MechanicalMixerBlockEntity mixer)) {
            setAreFluidsMoving(false);
            return;
        }

        //        setAreFluidsMoving(mixer.running && mixer.runningTicks <= 20);
    }

    public boolean isEmpty() {
        return itemCapability.method_5442() && inputTank.isEmpty() && outputTank.isEmpty();
    }

    public void onWrenched(class_2350 face) {
        class_2680 blockState = method_11010();
        class_2350 currentFacing = blockState.method_11654(BasinBlock.FACING);

        disabledSpoutputs.remove(face);
        if (currentFacing == face) {
            if (preferredSpoutput == face)
                preferredSpoutput = null;
            disabledSpoutputs.add(face);
        } else
            preferredSpoutput = face;

        updateSpoutput();
    }

    private void updateSpoutput() {
        class_2680 blockState = method_11010();
        class_2350 currentFacing = blockState.method_11654(BasinBlock.FACING);
        class_2350 newFacing = class_2350.field_11033;
        for (class_2350 test : Iterate.horizontalDirections) {
            boolean canOutputTo = BasinBlock.canOutputTo(field_11863, field_11867, test);
            if (canOutputTo && !disabledSpoutputs.contains(test))
                newFacing = test;
        }

        if (preferredSpoutput != null && BasinBlock.canOutputTo(field_11863, field_11867, preferredSpoutput) && preferredSpoutput != class_2350.field_11036)
            newFacing = preferredSpoutput;

        if (newFacing == currentFacing)
            return;

        field_11863.method_8501(field_11867, blockState.method_11657(BasinBlock.FACING, newFacing));

        if (newFacing.method_10166().method_10178())
            return;

        for (int slot = 9; slot < 18; slot++) {
            class_1799 stack = itemCapability.method_5438(slot);
            if (stack.method_7960())
                continue;
            if (acceptOutputs(ImmutableList.of(stack), Collections.emptyList(), true)) {
                acceptOutputs(ImmutableList.of(stack), Collections.emptyList(), false);
                itemCapability.method_5447(slot, class_1799.field_8037);
            }
        }

        FluidInventory handler = outputTank.getCapability();
        for (int slot = 0; slot < 2; slot++) {
            FluidStack fs = handler.getStack(slot);
            if (fs.isEmpty())
                continue;
            fs = fs.copy();
            if (acceptOutputs(Collections.emptyList(), ImmutableList.of(fs), true)) {
                handler.setStack(slot, FluidStack.EMPTY);
                acceptOutputs(Collections.emptyList(), ImmutableList.of(fs), false);
            }
        }

        notifyChangeOfContents();
        notifyUpdate();
    }

    @Override
    public void tick() {
        cachedHeatLevel = null;

        super.tick();
        if (field_11863.field_9236) {
            AllClientHandle.INSTANCE.createBasinFluidParticles(field_11863, this);
            tickVisualizedOutputs();
            ingredientRotationSpeed.tickChaser();
            ingredientRotation.setValue(ingredientRotation.getValue() + ingredientRotationSpeed.getValue());
        }

        if ((!spoutputBuffer.isEmpty() || !spoutputFluidBuffer.isEmpty()) && !field_11863.field_9236)
            tryClearingSpoutputOverflow();
        if (!contentsChanged)
            return;

        contentsChanged = false;
        getOperator().ifPresent(be -> be.basinChecker.scheduleUpdate());

        for (class_2350 offset : Iterate.horizontalDirections) {
            class_2338 toUpdate = field_11867.method_10084().method_10093(offset);
            class_2680 stateToUpdate = field_11863.method_8320(toUpdate);
            if (stateToUpdate.method_26204() instanceof BasinBlock && stateToUpdate.method_11654(BasinBlock.FACING) == offset.method_10153()) {
                class_2586 be = field_11863.method_8321(toUpdate);
                if (be instanceof BasinBlockEntity)
                    ((BasinBlockEntity) be).contentsChanged = true;
            }
        }
    }

    private void tryClearingSpoutputOverflow() {
        class_2680 blockState = method_11010();
        if (!(blockState.method_26204() instanceof BasinBlock))
            return;
        class_2350 direction = blockState.method_11654(BasinBlock.FACING);
        class_2586 be = field_11863.method_8321(field_11867.method_10074().method_10093(direction));

        ServerFilteringBehaviour filter = null;
        InvManipulationBehaviour inserter = null;
        if (be != null) {
            filter = BlockEntityBehaviour.get(field_11863, be.method_11016(), ServerFilteringBehaviour.TYPE);
            inserter = BlockEntityBehaviour.get(field_11863, be.method_11016(), InvManipulationBehaviour.TYPE);
        }

        if (filter != null && filter.isRecipeFilter())
            filter = null; // Do not test spout outputs against the recipe filter

        class_2350 opposite = direction.method_10153();
        class_1263 targetInv = be == null ? null : ItemHelper.getInventory(field_11863, be.method_11016(), null, be, opposite);
        if (targetInv == null && inserter != null) {
            targetInv = inserter.getInventory();
        }

        FluidInventory targetTank = be == null ? null : FluidHelper.getFluidInventory(field_11863, be.method_11016(), null, be, opposite);

        boolean update = false;

        for (Iterator<class_1799> iterator = spoutputBuffer.iterator(); iterator.hasNext(); ) {
            class_1799 itemStack = iterator.next();

            if (direction == class_2350.field_11033) {
                class_2248.method_9577(field_11863, field_11867, itemStack);
                iterator.remove();
                update = true;
                continue;
            }

            if (targetInv == null)
                break;

            if (targetInv.countSpace(itemStack, 1) == 0)
                continue;
            if (filter != null && !filter.test(itemStack))
                continue;

            if (visualizedOutputItems.size() < 3)
                visualizedOutputItems.add(IntAttached.withZero(itemStack));
            update = true;

            int count = itemStack.method_7947();
            int insert = targetInv.insertExist(itemStack, opposite);
            if (insert == count)
                iterator.remove();
            else
                itemStack.method_7934(insert);
        }

        for (Iterator<FluidStack> iterator = spoutputFluidBuffer.iterator(); iterator.hasNext(); ) {
            FluidStack fluidStack = iterator.next();

            if (direction == class_2350.field_11033) {
                iterator.remove();
                update = true;
                continue;
            }

            if (targetTank == null)
                break;

            if (targetTank instanceof SmartFluidTankBehaviour.InternalFluidHandler) {
                if (!targetTank.forcePreciseInsert(fluidStack)) {
                    continue;
                }
            } else if (!targetTank.preciseInsert(fluidStack, opposite)) {
                continue;
            }
            update = true;
            iterator.remove();
            if (visualizedOutputFluids.size() < 3)
                visualizedOutputFluids.add(IntAttached.withZero(fluidStack));
        }

        if (update) {
            notifyChangeOfContents();
            sendData();
        }
    }

    public float getTotalFluidUnits(float partialTicks) {
        int renderedFluids = 0;
        float totalUnits = 0;

        for (SmartFluidTankBehaviour behaviour : getTanks()) {
            if (behaviour == null)
                continue;
            for (TankSegment tankSegment : behaviour.getTanks()) {
                if (tankSegment.getRenderedFluid().isEmpty())
                    continue;
                float units = tankSegment.getTotalUnits(partialTicks);
                if (units < 1)
                    continue;
                totalUnits += units;
                renderedFluids++;
            }
        }

        if (renderedFluids == 0)
            return 0;
        if (totalUnits < 1)
            return 0;
        return totalUnits;
    }

    private Optional<BasinOperatingBlockEntity> getOperator() {
        if (field_11863 == null)
            return Optional.empty();
        class_2586 be = field_11863.method_8321(field_11867.method_10086(2));
        if (be instanceof BasinOperatingBlockEntity)
            return Optional.of((BasinOperatingBlockEntity) be);
        return Optional.empty();
    }

    public ServerFilteringBehaviour getFilter() {
        return filtering;
    }

    public void notifyChangeOfContents() {
        contentsChanged = true;
    }

    public boolean canContinueProcessing() {
        return spoutputBuffer.isEmpty() && spoutputFluidBuffer.isEmpty();
    }

    public boolean acceptOutputs(List<class_1799> outputItems, List<FluidStack> outputFluids, boolean simulate) {
        itemCapability.disableCheck();
        outputTank.allowInsertion();
        boolean acceptOutputsInner = acceptOutputsInner(outputItems, outputFluids, simulate);
        itemCapability.enableCheck();
        outputTank.forbidInsertion();
        return acceptOutputsInner;
    }

    private boolean acceptOutputsInner(List<class_1799> outputItems, List<FluidStack> outputFluids, boolean simulate) {
        class_2680 blockState = method_11010();
        if (!(blockState.method_26204() instanceof BasinBlock))
            return false;

        class_2350 direction = blockState.method_11654(BasinBlock.FACING);
        if (direction != class_2350.field_11033) {

            class_2586 be = field_11863.method_8321(field_11867.method_10074().method_10093(direction));

            InvManipulationBehaviour inserter = be == null ? null : BlockEntityBehaviour.get(field_11863, be.method_11016(), InvManipulationBehaviour.TYPE);
            class_2350 opposite = direction.method_10153();
            class_1263 targetInv = be == null ? null : ItemHelper.getInventory(field_11863, be.method_11016(), null, be, opposite);
            if (targetInv == null && inserter != null) {
                targetInv = inserter.getInventory();
            }
            FluidInventory targetTank = be == null ? null : FluidHelper.getFluidInventory(field_11863, be.method_11016(), null, be, opposite);
            boolean externalTankNotPresent = targetTank == null;

            if (!outputItems.isEmpty() && targetInv == null)
                return false;
            if (!outputFluids.isEmpty() && externalTankNotPresent) {
                // Special case - fluid outputs but output only accepts items
                targetTank = outputTank.getCapability();
                if (targetTank == null)
                    return false;
                if (!acceptFluidOutputsIntoBasin(outputFluids, simulate, targetTank))
                    return false;
            }

            if (simulate)
                return true;
            for (class_1799 itemStack : outputItems)
                if (!itemStack.method_7960())
                    spoutputBuffer.add(itemStack.method_7972());
            if (!externalTankNotPresent)
                for (FluidStack fluidStack : outputFluids)
                    spoutputFluidBuffer.add(fluidStack.copy());
            return true;
        }

        if (!acceptItemOutputsIntoBasin(outputItems, simulate, itemCapability))
            return false;
        if (outputFluids.isEmpty())
            return true;
        return acceptFluidOutputsIntoBasin(outputFluids, simulate, outputTank.getCapability());
    }

    private boolean acceptFluidOutputsIntoBasin(List<FluidStack> outputFluids, boolean simulate, FluidInventory targetTank) {
        if (simulate) {
            return targetTank.countSpace(outputFluids);
        } else {
            targetTank.insert(outputFluids);
            return true;
        }
    }

    private boolean acceptItemOutputsIntoBasin(List<class_1799> outputItems, boolean simulate, class_1263 targetInv) {
        if (simulate) {
            return targetInv.countSpace(outputItems, 9, 17);
        } else {
            targetInv.insert(outputItems, 9, 17);
            return true;
        }
    }

    public void readOnlyItems(class_11368 view) {
        itemCapability.read(view);
    }

    public static HeatLevel getHeatLevelOf(class_2680 state) {
        if (state.method_28498(BlazeBurnerBlock.HEAT_LEVEL))
            return state.method_11654(BlazeBurnerBlock.HEAT_LEVEL);
        return state.method_26164(AllBlockTags.PASSIVE_BOILER_HEATERS) && BlockHelper.isNotUnheated(state) ? HeatLevel.SMOULDERING : HeatLevel.NONE;
    }

    public Couple<SmartFluidTankBehaviour> getTanks() {
        return tanks;
    }

    // client things

    private void tickVisualizedOutputs() {
        visualizedOutputFluids.forEach(IntAttached::decrement);
        visualizedOutputItems.forEach(IntAttached::decrement);
        visualizedOutputFluids.removeIf(IntAttached::isOrBelowZero);
        visualizedOutputItems.removeIf(IntAttached::isOrBelowZero);
    }

    public boolean setAreFluidsMoving(boolean areFluidsMoving) {
        this.areFluidsMoving = areFluidsMoving;
        ingredientRotationSpeed.chase(areFluidsMoving ? 20 : 0, .1f, Chaser.EXP);
        return areFluidsMoving;
    }

    public static class BasinFluidHandler implements FluidInventory {
        @SuppressWarnings("OptionalUsedAsFieldOrParameterType")
        private static final Optional<Integer> LIMIT = Optional.of(BucketFluidInventory.CAPACITY);
        private final TankSegment[] output;
        private final TankSegment[] input;

        public BasinFluidHandler(TankSegment[] output, TankSegment[] input) {
            this.output = output;
            this.input = input;
        }

        @Override
        public int getMaxAmountPerStack() {
            return BucketFluidInventory.CAPACITY;
        }

        @Override
        public FluidStack onExtract(FluidStack stack) {
            return removeMaxSize(stack, LIMIT);
        }

        @Override
        public boolean isValid(int slot, FluidStack stack) {
            if (slot < 2) {
                return false;
            }
            for (int i = 0, size = input.length, current = slot - 2; i < size; i++) {
                FluidStack fluid = input[i].getFluid();
                if (fluid.isEmpty()) {
                    continue;
                }
                if (matches(fluid, stack) && i != current) {
                    return false;
                }
            }
            return true;
        }

        @Override
        public int size() {
            return 4;
        }

        @Override
        public FluidStack getStack(int slot) {
            if (slot >= 4) {
                return FluidStack.EMPTY;
            }
            return slot < 2 ? output[slot].getFluid() : input[slot - 2].getFluid();
        }

        @Override
        public void setStack(int slot, FluidStack stack) {
            if (slot >= 4) {
                return;
            }
            TankSegment tank;
            if (slot < 2) {
                tank = output[slot];
            } else {
                tank = input[slot - 2];
            }
            tank.setFluid(stack);
        }

        @Override
        public void markDirty() {
            for (TankSegment tank : input) {
                tank.markDirty();
            }
            for (TankSegment tank : output) {
                tank.markDirty();
            }
        }
    }

    @NotNull HeatLevel getHeatLevel() {
        if (cachedHeatLevel == null) {
            if (field_11863 == null)
                return HeatLevel.NONE;

            cachedHeatLevel = getHeatLevelOf(field_11863.method_8320(method_11016().method_10087(1)));
        }
        return cachedHeatLevel;
    }
}
