package com.zurrtum.create.foundation.blockEntity;

import com.zurrtum.create.AllClientHandle;
import com.zurrtum.create.AllTransfer;
import com.zurrtum.create.api.schematic.nbt.PartialSafeNBT;
import com.zurrtum.create.api.schematic.requirement.SpecialBlockEntityItemRequirement;
import com.zurrtum.create.content.schematics.requirement.ItemRequirement;
import com.zurrtum.create.foundation.advancement.AdvancementBehaviour;
import com.zurrtum.create.foundation.advancement.CreateTrigger;
import com.zurrtum.create.foundation.blockEntity.behaviour.BehaviourType;
import com.zurrtum.create.foundation.blockEntity.behaviour.BlockEntityBehaviour;
import com.zurrtum.create.foundation.blockEntity.behaviour.CachedInventoryBehaviour;
import com.zurrtum.create.foundation.utility.IInteractionChecker;
import com.zurrtum.create.ponder.api.VirtualBlockEntity;
import it.unimi.dsi.fastutil.objects.Reference2ObjectArrayMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1657;
import net.minecraft.class_2338;
import net.minecraft.class_2591;
import net.minecraft.class_2680;
import net.minecraft.class_3222;
import net.minecraft.class_9129;

public abstract class SmartBlockEntity extends CachedRenderBBBlockEntity implements PartialSafeNBT, IInteractionChecker, SpecialBlockEntityItemRequirement, VirtualBlockEntity {

    private final Map<BehaviourType<?>, BlockEntityBehaviour<?>> behaviours = new Reference2ObjectArrayMap<>();
    protected int lazyTickRate;
    protected int lazyTickCounter;
    private boolean initialized = false;
    private boolean firstNbtRead = true;
    private boolean chunkUnloaded;

    // Used for simulating this BE in a client-only setting
    private boolean virtualMode;

    public SmartBlockEntity(class_2591<?> type, class_2338 pos, class_2680 state) {
        super(type, pos, state);

        setLazyTickRate(10);

        ArrayList<BlockEntityBehaviour<?>> list = new ArrayList<>();
        addBehaviours(list);
        AllTransfer.addBehaviours(this, list);
        list.forEach(b -> behaviours.put(b.getType(), b));
        list.clear();
        AllClientHandle.INSTANCE.addBehaviours(this, list);
        list.forEach(b -> behaviours.put(b.getType(), b));
    }

    public abstract void addBehaviours(List<BlockEntityBehaviour<?>> behaviours);

    /**
     * Gets called just before reading block entity data for behaviours. Register
     * anything here that depends on your custom BE data.
     */
    public void addBehavioursDeferred(List<BlockEntityBehaviour<?>> behaviours) {
    }

    public void initialize() {
        if (firstNbtRead) {
            firstNbtRead = false;
            //TODO
            //            NeoForge.EVENT_BUS.post(new BlockEntityBehaviourEvent(this, behaviours));
        }

        forEachBehaviour(BlockEntityBehaviour::initialize);
        lazyTick();
    }

    public void tick() {
        if (!initialized && method_11002()) {
            initialize();
            initialized = true;
        }

        if (lazyTickCounter-- <= 0) {
            lazyTickCounter = lazyTickRate;
            lazyTick();
        }

        forEachBehaviour(BlockEntityBehaviour::tick);
    }

    public void lazyTick() {
    }

    /**
     * Hook only these in future subclasses of STE
     */
    protected void write(class_11372 view, boolean clientPacket) {
        super.method_11007(view);
        forEachBehaviour(tb -> tb.write(view, clientPacket));
    }

    @Override
    public void writeSafe(class_11372 view) {
        super.method_11007(view);
        forEachBehaviour(tb -> {
            if (tb.isSafeNBT())
                tb.writeSafe(view);
        });
    }

    /**
     * Hook only these in future subclasses of STE
     */
    protected void read(class_11368 view, boolean clientPacket) {
        if (firstNbtRead) {
            firstNbtRead = false;
            ArrayList<BlockEntityBehaviour<?>> list = new ArrayList<>();
            addBehavioursDeferred(list);
            list.forEach(b -> behaviours.put(b.getType(), b));
        }
        super.method_11014(view);
        forEachBehaviour(tb -> tb.read(view, clientPacket));
    }

    @Override
    protected void method_11014(class_11368 view) {
        read(view, false);
    }

    public void onChunkUnloaded() {
        chunkUnloaded = true;
    }

    @Override
    public void method_11012() {
        super.method_11012();
        if (!chunkUnloaded)
            remove();
        invalidate();
    }

    /**
     * Block destroyed or Chunk unloaded. Usually invalidates capabilities
     */
    public void invalidate() {
        forEachBehaviour(BlockEntityBehaviour::unload);
    }

    /**
     * Block destroyed or picked up by a contraption. Usually detaches kinetics
     */
    public void remove() {
    }

    /**
     * Block destroyed or replaced. Requires Block to call IBE::onRemove
     */
    public void destroy() {
        forEachBehaviour(BlockEntityBehaviour::destroy);
    }

    @Override
    public void method_66473(class_2338 pos, class_2680 oldState) {
        super.method_66473(pos, oldState);
        destroy();
    }

    @Override
    protected void method_11007(class_11372 view) {
        write(view, false);
    }

    @Override
    public final void readClient(class_11368 view) {
        read(view, true);
    }

    @Override
    public final void writeClient(class_11372 view) {
        write(view, true);
    }

    @SuppressWarnings("unchecked")
    public <T extends BlockEntityBehaviour<?>> T getBehaviour(BehaviourType<T> type) {
        return (T) behaviours.get(type);
    }

    public void forEachBehaviour(Consumer<BlockEntityBehaviour<?>> action) {
        getAllBehaviours().forEach(action);
    }

    public Collection<BlockEntityBehaviour<?>> getAllBehaviours() {
        return behaviours.values();
    }

    @SuppressWarnings("unchecked")
    public <T extends SmartBlockEntity> void attachBehaviourLate(BlockEntityBehaviour<T> behaviour) {
        BehaviourType<?> type = behaviour.getType();
        behaviours.values().forEach(b -> b.onBehaviourAdded(type, behaviour));
        behaviours.put(type, behaviour);
        behaviour.blockEntity = (T) this;
        behaviour.initialize();
    }

    public ItemRequirement getRequiredItems(class_2680 state) {
        return getAllBehaviours().stream().reduce(ItemRequirement.NONE, (r, b) -> r.union(b.getRequiredItems()), ItemRequirement::union);
    }

    public void removeBehaviour(BehaviourType<?> type) {
        BlockEntityBehaviour<?> remove = behaviours.remove(type);
        if (remove != null) {
            remove.unload();
        }
    }

    public void setLazyTickRate(int slowTickRate) {
        this.lazyTickRate = slowTickRate;
        this.lazyTickCounter = slowTickRate;
    }

    public void markVirtual() {
        virtualMode = true;
    }

    public boolean isVirtual() {
        return virtualMode;
    }

    public boolean isChunkUnloaded() {
        return chunkUnloaded;
    }

    @Override
    public boolean canPlayerUse(class_1657 player) {
        if (field_11863 == null || field_11863.method_8321(field_11867) != this)
            return false;
        return player.method_5649(field_11867.method_10263() + 0.5D, field_11867.method_10264() + 0.5D, field_11867.method_10260() + 0.5D) <= 64.0D;
    }

    public void sendToMenu(class_9129 buffer) {
        buffer.method_10807(method_11016());
        buffer.method_10794(method_16887(buffer.method_56349()));
    }

    @SuppressWarnings("deprecation")
    public void refreshBlockState() {
        method_31664(method_10997().method_8320(method_11016()));
    }

    public void addAdvancementBehaviour(class_3222 player) {
        List<CreateTrigger> awardables = getAwardables();
        if (awardables != null) {
            behaviours.put(AdvancementBehaviour.TYPE, new AdvancementBehaviour(this, player, awardables.toArray(CreateTrigger[]::new)));
        }
    }

    public List<CreateTrigger> getAwardables() {
        return null;
    }

    public void award(CreateTrigger advancement) {
        AdvancementBehaviour behaviour = getBehaviour(AdvancementBehaviour.TYPE);
        if (behaviour != null)
            behaviour.awardPlayer(advancement);
    }

    public void awardIfNear(CreateTrigger advancement, int range) {
        AdvancementBehaviour behaviour = getBehaviour(AdvancementBehaviour.TYPE);
        if (behaviour != null)
            behaviour.awardPlayerIfNear(advancement, range);
    }

    public void resetTransferCache() {
        CachedInventoryBehaviour<?> behaviour = getBehaviour(CachedInventoryBehaviour.TYPE);
        if (behaviour != null) {
            behaviour.reset();
        }
    }
}