package com.zurrtum.create.client.content.contraptions.render;

import com.zurrtum.create.api.behaviour.movement.MovementBehaviour;
import com.zurrtum.create.client.catnip.render.SuperByteBufferCache;
import com.zurrtum.create.client.flywheel.api.visualization.VisualizationManager;
import com.zurrtum.create.client.foundation.virtualWorld.VirtualRenderWorld;
import com.zurrtum.create.content.contraptions.Contraption;
import com.zurrtum.create.content.kinetics.base.KineticBlockEntity;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import net.minecraft.class_11352;
import net.minecraft.class_11515;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_2343;
import net.minecraft.class_2487;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_3499.class_3501;
import net.minecraft.class_3532;
import net.minecraft.class_8942;

import static com.zurrtum.create.Create.LOGGER;

public class ClientContraption {

    private final VirtualRenderWorld renderLevel;
    /**
     * The block entities that should be rendered.
     * This will exclude e.g. drills and deployers which are rendered in contraptions as actors.
     * All block entities are created with {@link #renderLevel} as their level.
     */
    private final List<class_2586> renderedBlockEntities = new ArrayList<>();
    public final List<class_2586> renderedBlockEntityView = Collections.unmodifiableList(renderedBlockEntities);

    // Parallel array to renderedBlockEntities, true if the block entity should be rendered.
    public final BitSet shouldRenderBlockEntities = new BitSet();
    // Parallel array to renderedBlockEntities. Scratch space for marking block entities that errored during rendering.
    public final BitSet scratchErroredBlockEntities = new BitSet();

    private final ContraptionMatrices matrices = new ContraptionMatrices();
    protected final Contraption contraption;
    private int structureVersion = 0;
    private int childrenVersion = 0;

    public ClientContraption(Contraption contraption) {
        var level = contraption.entity.method_37908();
        this.contraption = contraption;

        class_2338 origin = contraption.anchor;
        int minY = VirtualRenderWorld.nextMultipleOf16(class_3532.method_15357(contraption.bounds.field_1322 - 1));
        int height = VirtualRenderWorld.nextMultipleOf16(class_3532.method_15384(contraption.bounds.field_1325 + 1)) - minY;
        renderLevel = new VirtualRenderWorld(level, minY, height, origin, this::invalidateStructure) {
            @Override
            public boolean supportsVisualization() {
                return VisualizationManager.supportsVisualization(level);
            }
        };

        setupRenderLevelAndRenderedBlockEntities();
    }

    public Contraption getContraption() {
        return contraption;
    }

    /**
     * A version integer incremented each time the render level changes.
     */
    public int structureVersion() {
        return structureVersion;
    }

    public int childrenVersion() {
        return childrenVersion;
    }

    public void resetRenderLevel() {
        renderedBlockEntities.clear();
        renderLevel.clear();
        shouldRenderBlockEntities.clear();

        setupRenderLevelAndRenderedBlockEntities();

        invalidateStructure();
        invalidateChildren();
    }

    public void invalidateChildren() {
        childrenVersion++;
    }

    public void invalidateStructure() {
        for (class_11515 renderType : class_11515.values()) {
            SuperByteBufferCache.getInstance().invalidate(ContraptionEntityRenderer.CONTRAPTION, Pair.of(contraption, renderType));
        }

        structureVersion++;
    }

    private void setupRenderLevelAndRenderedBlockEntities() {
        for (class_3501 info : contraption.getBlocks().values()) {
            renderLevel.method_8652(info.comp_1341(), info.comp_1342(), 0);

            class_2586 blockEntity = readBlockEntity(renderLevel, info, contraption.getIsLegacy().getBoolean(info.comp_1341()));

            if (blockEntity != null) {
                renderLevel.method_8438(blockEntity);

                // Don't render block entities that have an actor renderer registered in the MovementBehaviour.
                MovementBehaviour movementBehaviour = MovementBehaviour.REGISTRY.get(info.comp_1342());
                if (movementBehaviour == null || !movementBehaviour.disableBlockEntityRendering()) {
                    renderedBlockEntities.add(blockEntity);
                }
            }
        }

        shouldRenderBlockEntities.set(0, renderedBlockEntities.size());

        renderLevel.runLightEngine();
    }

    @Nullable
    public class_2586 readBlockEntity(class_1937 level, class_3501 info, boolean legacy) {
        class_2680 state = info.comp_1342();
        class_2338 pos = info.comp_1341();
        class_2487 nbt = info.comp_1343();

        if (legacy) {
            // for contraptions that were assembled pre-updateTags, we need to use the old strategy.
            if (nbt == null)
                return null;

            nbt.method_10569("x", pos.method_10263());
            nbt.method_10569("y", pos.method_10264());
            nbt.method_10569("z", pos.method_10260());

            class_2586 be = class_2586.method_11005(pos, state, nbt, level.method_30349());
            postprocessReadBlockEntity(level, be, state);
            return be;
        }

        if (!state.method_31709() || !(state.method_26204() instanceof class_2343 entityBlock))
            return null;

        class_2586 be = entityBlock.method_10123(pos, state);
        postprocessReadBlockEntity(level, be, state);
        if (be != null && nbt != null) {
            try (class_8942.class_11340 logging = new class_8942.class_11340(be.method_71402(), LOGGER)) {
                be.method_58690(class_11352.method_71417(logging, level.method_30349(), nbt));
            }
        }

        return be;
    }

    @SuppressWarnings("deprecation")
    protected static void postprocessReadBlockEntity(class_1937 level, @Nullable class_2586 be, class_2680 blockState) {
        if (be != null) {
            be.method_31662(level);
            be.method_31664(blockState);
            if (be instanceof KineticBlockEntity kbe) {
                kbe.setSpeed(0);
            }
        }
    }

    public VirtualRenderWorld getRenderLevel() {
        return renderLevel;
    }

    public ContraptionMatrices getMatrices() {
        return matrices;
    }

    public RenderedBlocks getRenderedBlocks() {
        return new RenderedBlocks(
            pos -> {
                class_3501 info = contraption.getBlocks().get(pos);
                if (info == null) {
                    return class_2246.field_10124.method_9564();
                }
                return info.comp_1342();
            }, contraption.getBlocks().keySet()
        );
    }

    @Nullable
    public class_2586 getBlockEntity(class_2338 localPos) {
        return renderLevel.method_8321(localPos);
    }

    /**
     * Get the BitSet marking which block entities should be rendered, potentially with additional filtering.
     *
     * <p>Implementors: DO NOT modify {@link #shouldRenderBlockEntities} directly.
     */
    public BitSet getAndAdjustShouldRenderBlockEntities() {
        return shouldRenderBlockEntities;
    }

    public record RenderedBlocks(Function<class_2338, class_2680> lookup, Iterable<class_2338> positions) {
    }

    /**
     * Entirely reset the client contraption, rebuilding the client level and re-running light updates.
     */
    @SuppressWarnings("unchecked")
    public static void resetClientContraption(Contraption contraption) {
        AtomicReference<ClientContraption> clientContraption = (AtomicReference<ClientContraption>) contraption.clientContraption;
        var maybeNullClientContraption = clientContraption.getAcquire();

        // Nothing to invalidate if it hasn't been created yet.
        if (maybeNullClientContraption != null) {
            maybeNullClientContraption.resetRenderLevel();
        }
    }

    /**
     * Invalidate the structure of the client contraption, triggering a rebuild of the main mesh.
     */
    @SuppressWarnings("unchecked")
    public static void invalidateClientContraptionStructure(Contraption contraption) {
        AtomicReference<ClientContraption> clientContraption = (AtomicReference<ClientContraption>) contraption.clientContraption;
        var maybeNullClientContraption = clientContraption.getAcquire();

        // Nothing to invalidate if it hasn't been created yet.
        if (maybeNullClientContraption != null) {
            maybeNullClientContraption.invalidateStructure();
        }
    }

    /**
     * Invalidate the children of the client contraption, triggering a rebuild of all child visuals.
     */
    @SuppressWarnings("unchecked")
    public static void invalidateClientContraptionChildren(Contraption contraption) {
        AtomicReference<ClientContraption> clientContraption = (AtomicReference<ClientContraption>) contraption.clientContraption;
        var maybeNullClientContraption = clientContraption.getAcquire();

        // Nothing to invalidate if it hasn't been created yet.
        if (maybeNullClientContraption != null) {
            maybeNullClientContraption.invalidateChildren();
        }
    }

    @SuppressWarnings("unchecked")
    @Nullable
    public static class_2586 getBlockEntityClientSide(Contraption contraption, class_2338 localPos) {
        AtomicReference<ClientContraption> clientContraption = (AtomicReference<ClientContraption>) contraption.clientContraption;
        var maybeNullClientContraption = clientContraption.getAcquire();

        if (maybeNullClientContraption == null) {
            return null;
        }

        return maybeNullClientContraption.getBlockEntity(localPos);
    }
}