/*
 * Decompiled with CFR 0.152.
 */
package net.oxcodsnet.roadarchitect.storage;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Objects;
import java.util.concurrent.ConcurrentMap;
import java.util.function.DoubleSupplier;
import java.util.function.IntSupplier;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;
import net.minecraft.core.Holder;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.world.level.biome.Biome;
import net.oxcodsnet.roadarchitect.config.RAConfigHolder;
import net.oxcodsnet.roadarchitect.config.records.CacheSettings;
import net.oxcodsnet.roadarchitect.storage.ColumnRecord;
import net.oxcodsnet.roadarchitect.storage.DimensionPaths;
import net.oxcodsnet.roadarchitect.storage.RegionColumnStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class CacheStorage {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)"RoadArchitect/CacheStorage");
    private static final String LEGACY_FILE_NAME = "road_cache.dat";
    private final CacheSettings settings;
    private final Cache<Long, ColumnRecord> runtime;
    private final RegionColumnStore regionStore;
    private final boolean persistHeights;
    private final boolean persistStabilities;
    private final boolean persistBiomes;

    private CacheStorage(ServerLevel world, CacheSettings settings) {
        this.settings = settings;
        this.persistHeights = settings.persistHeights();
        this.persistStabilities = settings.persistStabilities();
        this.persistBiomes = settings.persistBiomes();
        this.regionStore = new RegionColumnStore(world, this.persistHeights, this.persistStabilities, this.persistBiomes, settings.clampedRegionSize(), settings.persistedBudgetBytes());
        long maxWeight = Math.max(settings.runtimeBudgetBytes(), 0x1000000L);
        this.runtime = Caffeine.newBuilder().maximumWeight(maxWeight).weigher((key, value) -> value == null ? 0 : value.weightBytes()).recordStats().build();
        this.quarantineLegacyStore(world);
    }

    public static CacheStorage open(ServerLevel world) {
        CacheSettings settings = RAConfigHolder.get().cache().clampToRuntime();
        return new CacheStorage(world, settings);
    }

    public CacheSettings settings() {
        return this.settings;
    }

    public Integer getHeight(long key) {
        ColumnRecord record = this.getOrLoad(key);
        return record.hasHeight() ? record.height() : null;
    }

    public void putHeight(long key, int value) {
        this.mutate(key, existing -> existing.withHeight(value));
    }

    public void putHeightIfAbsent(long key, int value) {
        this.mutate(key, existing -> existing.hasHeight() ? existing : existing.withHeight(value));
    }

    public int computeHeightIfAbsent(long key, IntSupplier loader) {
        Integer cached = this.getHeight(key);
        if (cached != null) {
            return cached;
        }
        int value = loader.getAsInt();
        this.putHeightIfAbsent(key, value);
        return value;
    }

    public Double getStability(long key) {
        ColumnRecord record = this.getOrLoad(key);
        return record.hasStability() ? record.stability() : null;
    }

    public double computeStabilityIfAbsent(long key, DoubleSupplier loader) {
        Double cached = this.getStability(key);
        if (cached != null) {
            return cached;
        }
        double value = loader.getAsDouble();
        this.mutate(key, existing -> existing.hasStability() ? existing : existing.withStability(value));
        return value;
    }

    public Holder<Biome> getBiome(long key) {
        ColumnRecord record = this.getOrLoad(key);
        return record.hasBiome() ? record.biome() : null;
    }

    public Holder<Biome> computeBiomeIfAbsent(long key, Supplier<Holder<Biome>> loader) {
        Holder<Biome> cached = this.getBiome(key);
        if (cached != null) {
            return cached;
        }
        Holder<Biome> value = loader.get();
        if (value != null) {
            this.mutate(key, existing -> existing.hasBiome() ? existing : existing.withBiome(value));
        }
        return value;
    }

    public void putBiome(long key, Holder<Biome> biome) {
        if (biome == null) {
            return;
        }
        this.mutate(key, existing -> existing.withBiome(biome));
    }

    public void flush() {
        this.regionStore.flush();
    }

    public long runtimeWeightBytes() {
        return this.runtime.policy().eviction().map(eviction -> eviction.weightedSize().orElse(0L)).orElseGet(() -> this.runtime.estimatedSize() * 32L);
    }

    public long regionWeightBytes() {
        return this.regionStore.weightBytes();
    }

    private ColumnRecord getOrLoad(long key) {
        ColumnRecord record = (ColumnRecord)this.runtime.getIfPresent((Object)key);
        if (record != null) {
            return record;
        }
        ColumnRecord persisted = this.regionStore.read(key);
        if (persisted != null && !persisted.isEmpty()) {
            this.runtime.put((Object)key, (Object)this.merge(ColumnRecord.EMPTY, persisted));
            return persisted;
        }
        return ColumnRecord.EMPTY;
    }

    private ColumnRecord merge(ColumnRecord base, ColumnRecord persisted) {
        ColumnRecord merged = base;
        if (persisted.hasHeight() && !base.hasHeight()) {
            merged = merged.withHeight(persisted.height());
        }
        if (persisted.hasStability() && !base.hasStability()) {
            merged = merged.withStability(persisted.stability());
        }
        if (persisted.hasBiome() && !base.hasBiome()) {
            merged = merged.withBiome(persisted.biome());
        }
        return merged;
    }

    private void mutate(long key, UnaryOperator<ColumnRecord> operator) {
        ConcurrentMap map = this.runtime.asMap();
        map.compute(key, (k, existing) -> {
            ColumnRecord base = existing == null ? ColumnRecord.EMPTY : existing;
            ColumnRecord updated = (ColumnRecord)operator.apply(base);
            if (updated == null || updated.isEmpty()) {
                if (!base.isEmpty()) {
                    this.regionStore.write((long)k, ColumnRecord.EMPTY);
                }
                return null;
            }
            if (!(Objects.equals(base.height(), updated.height()) && Objects.equals(base.stability(), updated.stability()) && Objects.equals(base.biome(), updated.biome()))) {
                boolean shouldPersist;
                boolean bl = shouldPersist = this.persistHeights && !Objects.equals(base.height(), updated.height()) || this.persistStabilities && !Objects.equals(base.stability(), updated.stability()) || this.persistBiomes && !Objects.equals(base.biome(), updated.biome());
                if (shouldPersist) {
                    this.regionStore.write((long)k, updated);
                }
            }
            return updated;
        });
    }

    private void quarantineLegacyStore(ServerLevel world) {
        Path dataDir = DimensionPaths.resolveDimensionRoot(world).resolve("data");
        Path legacy = dataDir.resolve(LEGACY_FILE_NAME);
        if (!Files.exists(legacy, new LinkOption[0])) {
            return;
        }
        Path backup = legacy.resolveSibling("road_cache.dat.legacy");
        try {
            Files.createDirectories(backup.getParent(), new FileAttribute[0]);
            Files.move(legacy, backup, StandardCopyOption.REPLACE_EXISTING);
            LOGGER.warn("Legacy cache file '{}' was found and moved to '{}'. The new cache backend will rebuild data on demand.", (Object)legacy, (Object)backup);
        }
        catch (IOException e) {
            LOGGER.error("Failed to move legacy cache file {}", (Object)legacy, (Object)e);
        }
    }
}

