/*
 * Decompiled with CFR 0.152.
 */
package net.oxcodsnet.roadarchitect.storage;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.RemovalCause;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.Objects;
import net.minecraft.core.Holder;
import net.minecraft.core.HolderLookup;
import net.minecraft.core.registries.Registries;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.ListTag;
import net.minecraft.nbt.NbtAccounter;
import net.minecraft.nbt.NbtIo;
import net.minecraft.nbt.Tag;
import net.minecraft.resources.ResourceKey;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.world.level.biome.Biome;
import net.oxcodsnet.roadarchitect.storage.ColumnRecord;
import net.oxcodsnet.roadarchitect.storage.DimensionPaths;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class RegionColumnStore {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)"RoadArchitect/RegionColumnStore");
    private static final String LIST_KEY = "columns";
    private static final String VERSION_KEY = "version";
    private static final int FORMAT_VERSION = 1;
    private final Path regionDir;
    private final HolderLookup.RegistryLookup<Biome> biomeRegistry;
    private final boolean persistHeights;
    private final boolean persistStabilities;
    private final boolean persistBiomes;
    private final int regionSizeChunks;
    private final Cache<Long, RegionData> regions;
    private final long budgetBytes;

    RegionColumnStore(ServerLevel world, boolean persistHeights, boolean persistStabilities, boolean persistBiomes, int regionSizeChunks, long persistedBudgetBytes) {
        this.regionDir = DimensionPaths.resolveCacheDirectory(world);
        this.biomeRegistry = world.registryAccess().lookupOrThrow(Registries.BIOME);
        this.persistHeights = persistHeights;
        this.persistStabilities = persistStabilities;
        this.persistBiomes = persistBiomes;
        this.regionSizeChunks = Math.max(4, regionSizeChunks);
        long maxWeight = this.budgetBytes = Math.max(persistedBudgetBytes, 0x2000000L);
        this.regions = Caffeine.newBuilder().maximumWeight(maxWeight).weigher((key, value) -> value.weightBytes()).removalListener(this::onRegionRemoval).build();
    }

    ColumnRecord read(long columnKey) {
        RegionData region = (RegionData)this.regions.get((Object)this.regionKey(columnKey), this::loadRegion);
        return region.get(columnKey);
    }

    void write(long columnKey, ColumnRecord record) {
        long regionKey = this.regionKey(columnKey);
        RegionData region = (RegionData)this.regions.get((Object)regionKey, this::loadRegion);
        ColumnRecord persisted = this.filterPersisted(record);
        region.put(columnKey, persisted);
    }

    void flush() {
        this.regions.asMap().forEach(this::flushRegion);
    }

    long weightBytes() {
        return this.regions.policy().eviction().map(eviction -> eviction.weightedSize().orElse(0L)).orElseGet(() -> Math.max(1L, this.regions.estimatedSize()) * 512L);
    }

    private void onRegionRemoval(Long key, RegionData value, RemovalCause cause) {
        if (key == null || value == null) {
            return;
        }
        this.flushRegion(key, value);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private RegionData loadRegion(long regionKey) {
        Path path = this.regionPath(regionKey);
        RegionData data = new RegionData();
        if (!Files.exists(path, new LinkOption[0])) {
            return data;
        }
        try (InputStream in = Files.newInputStream(path, new OpenOption[0]);){
            CompoundTag tag = NbtIo.readCompressed((InputStream)in, (NbtAccounter)NbtAccounter.unlimitedHeap());
            if (tag == null) {
                RegionData regionData = data;
                return regionData;
            }
            tag.getList(LIST_KEY).ifPresent(list -> {
                for (int i = 0; i < list.size(); ++i) {
                    list.getCompound(i).ifPresent(entry -> {
                        ColumnRecord record;
                        ResourceLocation id;
                        String biomeId;
                        Long columnKeyBoxed = entry.getLong("k").orElse(null);
                        if (columnKeyBoxed == null) {
                            return;
                        }
                        long columnKey = columnKeyBoxed;
                        Integer height = this.persistHeights ? (Integer)entry.getInt("h").orElse(null) : null;
                        Double stability = this.persistStabilities ? (Double)entry.getDouble("s").orElse(null) : null;
                        Holder biome = null;
                        if (this.persistBiomes && (biomeId = (String)entry.getString("b").orElse(null)) != null && (id = ResourceLocation.tryParse((String)biomeId)) != null) {
                            ResourceKey key = ResourceKey.create((ResourceKey)Registries.BIOME, (ResourceLocation)id);
                            biome = this.biomeRegistry.get(key).orElse(null);
                        }
                        if (!(record = new ColumnRecord(height, stability, biome)).isEmpty()) {
                            data.putLoaded(columnKey, record);
                        }
                    });
                }
            });
        }
        catch (IOException e) {
            LOGGER.error("Failed to load cached region {}", (Object)path, (Object)e);
        }
        data.markClean();
        return data;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void flushRegion(long regionKey, RegionData region) {
        Long2ObjectMap<ColumnRecord> snapshot = region.snapshotIfDirty();
        if (snapshot == null) {
            return;
        }
        Path path = this.regionPath(regionKey);
        if (snapshot.isEmpty()) {
            try {
                Files.deleteIfExists(path);
            }
            catch (IOException e) {
                LOGGER.warn("Failed to delete empty cache region {}", (Object)path, (Object)e);
            }
            finally {
                region.markClean();
            }
            return;
        }
        CompoundTag root = new CompoundTag();
        root.putInt(VERSION_KEY, 1);
        ListTag list = new ListTag();
        for (Long2ObjectMap.Entry entry : snapshot.long2ObjectEntrySet()) {
            ColumnRecord record = (ColumnRecord)entry.getValue();
            if (record == null || record.isEmpty()) continue;
            CompoundTag columnTag = new CompoundTag();
            columnTag.putLong("k", entry.getLongKey());
            if (this.persistHeights && record.hasHeight()) {
                columnTag.putInt("h", record.height().intValue());
            }
            if (this.persistStabilities && record.hasStability()) {
                columnTag.putDouble("s", record.stability().doubleValue());
            }
            if (this.persistBiomes && record.hasBiome()) {
                record.biome().unwrapKey().map(ResourceKey::location).ifPresent(id -> columnTag.putString("b", id.toString()));
            }
            list.add((Object)columnTag);
        }
        root.put(LIST_KEY, (Tag)list);
        Path tmp = path.resolveSibling(path.getFileName().toString() + ".tmp");
        try (OutputStream out = Files.newOutputStream(tmp, new OpenOption[0]);){
            NbtIo.writeCompressed((CompoundTag)root, (OutputStream)out);
            Files.move(tmp, path, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
        }
        catch (IOException e) {
            LOGGER.error("Failed to write cache region {}", (Object)path, (Object)e);
            try {
                Files.deleteIfExists(tmp);
            }
            catch (IOException iOException) {
                // empty catch block
            }
            return;
        }
        region.markClean();
    }

    private ColumnRecord filterPersisted(ColumnRecord record) {
        Holder<Biome> biome;
        if (record == null || record.isEmpty()) {
            return ColumnRecord.EMPTY;
        }
        Integer height = this.persistHeights && record.hasHeight() ? record.height() : null;
        Double stability = this.persistStabilities && record.hasStability() ? record.stability() : null;
        Holder<Biome> holder = biome = this.persistBiomes && record.hasBiome() ? record.biome() : null;
        if (Objects.equals(height, record.height()) && Objects.equals(stability, record.stability()) && Objects.equals(biome, record.biome())) {
            return record;
        }
        return new ColumnRecord(height, stability, biome);
    }

    private long regionKey(long columnKey) {
        int blockX = (int)(columnKey >> 32);
        int blockZ = (int)columnKey;
        int chunkX = Math.floorDiv(blockX, 16);
        int chunkZ = Math.floorDiv(blockZ, 16);
        int regionX = Math.floorDiv(chunkX, this.regionSizeChunks);
        int regionZ = Math.floorDiv(chunkZ, this.regionSizeChunks);
        return (long)regionX << 32 | (long)regionZ & 0xFFFFFFFFL;
    }

    private Path regionPath(long regionKey) {
        int regionX = (int)(regionKey >> 32);
        int regionZ = (int)regionKey;
        String fileName = "region_" + regionX + "_" + regionZ + ".nbt";
        return this.regionDir.resolve(fileName);
    }

    private static final class RegionData {
        private final Long2ObjectMap<ColumnRecord> columns = new Long2ObjectOpenHashMap();
        private boolean dirty;

        private RegionData() {
        }

        synchronized ColumnRecord get(long key) {
            return (ColumnRecord)this.columns.get(key);
        }

        synchronized void put(long key, ColumnRecord record) {
            if (record == null || record.isEmpty()) {
                if (this.columns.remove(key) != null) {
                    this.dirty = true;
                }
                return;
            }
            ColumnRecord previous = (ColumnRecord)this.columns.put(key, (Object)record);
            if (!record.equals(previous)) {
                this.dirty = true;
            }
        }

        synchronized void putLoaded(long key, ColumnRecord record) {
            if (record == null || record.isEmpty()) {
                return;
            }
            this.columns.put(key, (Object)record);
        }

        synchronized void markClean() {
            this.dirty = false;
        }

        synchronized Long2ObjectMap<ColumnRecord> snapshotIfDirty() {
            if (!this.dirty) {
                return null;
            }
            return new Long2ObjectOpenHashMap(this.columns);
        }

        synchronized int weightBytes() {
            return Math.max(1, this.columns.size()) * 64;
        }
    }
}

