/*
 * Decompiled with CFR 0.152.
 */
package com.foundryx.managers;

import com.foundryx.Constants;
import com.foundryx.data.BanData;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Reader;
import java.lang.reflect.Type;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.time.format.DateTimeParseException;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

public final class BanDataManager {
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private static final Type LIST_TYPE = new TypeToken<List<PersistedBanData>>(){}.getType();
    private final Path directory;
    private final Path file;
    private static final String PLAYER_PREFIX = "player:";
    private static final String IP_PREFIX = "ip:";
    private final Map<String, BanData> cache = new ConcurrentHashMap<String, BanData>();

    public BanDataManager(Path directory) {
        this.directory = Objects.requireNonNull(directory, "directory");
        this.file = directory.resolve("banned_players.json");
        this.load();
    }

    public void addOrUpdate(BanData banData) {
        Objects.requireNonNull(banData, "banData");
        this.cache.put(BanDataManager.keyOf(banData), banData);
        this.save();
    }

    public Optional<BanData> get(UUID playerUuid) {
        Objects.requireNonNull(playerUuid, "playerUuid");
        return Optional.ofNullable(this.cache.get(BanDataManager.playerKey(playerUuid)));
    }

    public Optional<BanData> getByIp(String ipAddress) {
        Objects.requireNonNull(ipAddress, "ipAddress");
        return Optional.ofNullable(this.cache.get(BanDataManager.ipKey(ipAddress)));
    }

    public boolean remove(UUID playerUuid) {
        Objects.requireNonNull(playerUuid, "playerUuid");
        if (this.cache.remove(BanDataManager.playerKey(playerUuid)) != null) {
            this.save();
            return true;
        }
        return false;
    }

    public boolean removeByIp(String ipAddress) {
        Objects.requireNonNull(ipAddress, "ipAddress");
        if (this.cache.remove(BanDataManager.ipKey(ipAddress)) != null) {
            this.save();
            return true;
        }
        return false;
    }

    public Collection<BanData> getAll() {
        return List.copyOf(this.cache.values());
    }

    private void load() {
        this.cache.clear();
        if (!Files.exists(this.file, new LinkOption[0])) {
            return;
        }
        try (BufferedReader reader = Files.newBufferedReader(this.file);){
            List persisted = (List)GSON.fromJson((Reader)reader, LIST_TYPE);
            if (persisted == null) {
                return;
            }
            for (PersistedBanData entry : persisted) {
                boolean missingIp;
                if (entry == null || entry.banDate == null) continue;
                boolean missingUuid = entry.uuid == null || entry.uuid.isBlank();
                boolean bl = missingIp = entry.ipAddress == null || entry.ipAddress.isBlank();
                if (missingUuid && missingIp) continue;
                try {
                    UUID uuid = missingUuid ? null : UUID.fromString(entry.uuid);
                    Instant bannedAt = Instant.parse(entry.banDate);
                    Instant unbanAt = entry.unbanDate == null || entry.unbanDate.isBlank() ? null : Instant.parse(entry.unbanDate);
                    BanData banData = new BanData(uuid, entry.ipAddress, entry.nickname, entry.bannedBy, entry.reason, bannedAt, unbanAt);
                    this.cache.put(BanDataManager.keyOf(banData), banData);
                }
                catch (IllegalArgumentException | DateTimeParseException exception) {
                    Constants.LOG.warn("Skipping invalid ban data entry: {}", (Object)entry, (Object)exception);
                }
            }
        }
        catch (IOException exception) {
            Constants.LOG.error("Failed to read ban data", (Throwable)exception);
        }
    }

    private void save() {
        List<PersistedBanData> entries = this.cache.values().stream().map(PersistedBanData::from).toList();
        try {
            Files.createDirectories(this.directory, new FileAttribute[0]);
            try (BufferedWriter writer = Files.newBufferedWriter(this.file, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);){
                GSON.toJson(entries, LIST_TYPE, (Appendable)writer);
            }
        }
        catch (IOException exception) {
            Constants.LOG.error("Failed to write ban data", (Throwable)exception);
        }
    }

    private static String keyOf(BanData banData) {
        return banData.getPlayerUuid().map(BanDataManager::playerKey).orElseGet(() -> BanDataManager.ipKey(banData.getIpAddress().orElseThrow(() -> new IllegalArgumentException("BanData missing both UUID and IP address"))));
    }

    private static String playerKey(UUID uuid) {
        return PLAYER_PREFIX + String.valueOf(uuid);
    }

    private static String ipKey(String ipAddress) {
        return IP_PREFIX + ipAddress;
    }

    private record PersistedBanData(String uuid, String ipAddress, String nickname, String bannedBy, String reason, String banDate, String unbanDate) {
        private static PersistedBanData from(BanData data) {
            return new PersistedBanData(data.getPlayerUuid().map(UUID::toString).orElse(null), data.getIpAddress().orElse(null), data.getNickname(), data.getBannedBy(), data.getReason(), data.getBanDate().toString(), data.getUnbanDate().map(Instant::toString).orElse(null));
        }

        @Override
        public String toString() {
            return String.format(Locale.ROOT, "PersistedBanData{uuid=%s, ip=%s}", this.uuid, this.ipAddress);
        }
    }
}

