/*
 * Decompiled with CFR 0.152.
 */
package com.foundryx.managers;

import com.foundryx.Constants;
import com.foundryx.storage.FoundryxDataStorage;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonParseException;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.ItemLike;

public final class KitManager {
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private static final Path CONFIG_PATH = Path.of("config", "foundryx", "kits.json");
    private static final Object LOCK = new Object();
    private static Map<String, KitDefinition> kits = new HashMap<String, KitDefinition>();

    private KitManager() {
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void load() {
        Object object = LOCK;
        synchronized (object) {
            kits = KitManager.readKits();
        }
    }

    public static String permissionNode(KitDefinition kit) {
        return KitManager.permissionNode(kit.name());
    }

    public static String permissionNode(String kitName) {
        return String.format(Locale.ROOT, "foundryx.kit.%s", KitManager.normalise(kitName));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static Collection<KitDefinition> getKits() {
        Object object = LOCK;
        synchronized (object) {
            return List.copyOf(kits.values());
        }
    }

    public static List<String> getKitNames() {
        return KitManager.getKits().stream().map(KitDefinition::name).sorted(String::compareToIgnoreCase).toList();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static KitDefinition getKit(String name) {
        if (name == null) {
            return null;
        }
        Object object = LOCK;
        synchronized (object) {
            return kits.get(KitManager.normalise(name));
        }
    }

    public static void giveKit(ServerPlayer player, KitDefinition kit) {
        for (ItemStack stack : kit.items()) {
            ItemStack copy = stack.copy();
            boolean fullyAdded = player.getInventory().add(copy);
            if (fullyAdded && copy.isEmpty() || copy.isEmpty()) continue;
            player.drop(copy, false);
        }
        player.inventoryMenu.broadcastChanges();
    }

    public static void recordUsage(ServerPlayer player, KitDefinition kit) {
        if (kit.cooldownSeconds() <= 0L) {
            return;
        }
        FoundryxDataStorage storage = FoundryxDataStorage.get(player.level().getServer());
        storage.recordKitUsage(player.getUUID(), kit.name(), Instant.now().toEpochMilli());
    }

    public static long getRemainingCooldown(ServerPlayer player, KitDefinition kit) {
        long cooldownSeconds = Math.max(0L, kit.cooldownSeconds());
        if (cooldownSeconds <= 0L) {
            return 0L;
        }
        FoundryxDataStorage storage = FoundryxDataStorage.get(player.level().getServer());
        return storage.getRemainingKitCooldown(player.getUUID(), kit.name(), cooldownSeconds);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean deleteKit(String name) {
        if (name == null) {
            return false;
        }
        Object object = LOCK;
        synchronized (object) {
            KitDefinition removed = kits.remove(KitManager.normalise(name));
            if (removed == null) {
                return false;
            }
            KitManager.writeKits(List.copyOf(kits.values()));
            FoundryxDataStorage.clearKitUsageForAll(removed.name());
            return true;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static Map<String, KitDefinition> readKits() {
        KitManager.ensureConfigurationExists();
        try (BufferedReader reader = Files.newBufferedReader(CONFIG_PATH, StandardCharsets.UTF_8);){
            KitsFile file = (KitsFile)GSON.fromJson((Reader)reader, KitsFile.class);
            if (file == null || file.kits == null) {
                Map<String, KitDefinition> map = Map.of();
                return map;
            }
            HashMap<String, KitDefinition> loaded = new HashMap<String, KitDefinition>();
            for (StoredKit stored : file.kits) {
                KitDefinition definition = KitManager.parseKit(stored);
                if (definition == null) continue;
                loaded.put(KitManager.normalise(definition.name()), definition);
            }
            HashMap<String, KitDefinition> hashMap = loaded;
            return hashMap;
        }
        catch (IOException exception) {
            Constants.LOG.error("Failed to read kit configuration from {}", (Object)CONFIG_PATH.toAbsolutePath(), (Object)exception);
            return new HashMap<String, KitDefinition>();
        }
        catch (JsonParseException exception) {
            Constants.LOG.error("Failed to parse kit configuration from {}", (Object)CONFIG_PATH.toAbsolutePath(), (Object)exception);
        }
        return new HashMap<String, KitDefinition>();
    }

    private static KitDefinition parseKit(StoredKit stored) {
        if (stored == null || stored.name == null || stored.name.isBlank()) {
            return null;
        }
        String name = stored.name.trim();
        long cooldownSeconds = Math.max(0L, Optional.ofNullable(stored.cooldownSeconds).orElse(0L));
        if (stored.items == null || stored.items.isEmpty()) {
            Constants.LOG.warn("Kit '{}' has no items defined and will be ignored", (Object)name);
            return null;
        }
        ArrayList<ItemStack> items = new ArrayList<ItemStack>();
        for (StoredKitItem item : stored.items) {
            ItemStack stack = KitManager.parseItem(item);
            if (stack == null || stack.isEmpty()) continue;
            items.add(stack);
        }
        if (items.isEmpty()) {
            Constants.LOG.warn("Kit '{}' does not contain any valid items and will be ignored", (Object)name);
            return null;
        }
        return new KitDefinition(name, cooldownSeconds, List.copyOf(items));
    }

    private static ItemStack parseItem(StoredKitItem stored) {
        if (stored == null || stored.id == null || stored.id.isBlank()) {
            return null;
        }
        ResourceLocation identifier = ResourceLocation.tryParse((String)stored.id);
        if (identifier == null) {
            Constants.LOG.warn("Skipping kit item with invalid identifier '{}'", (Object)stored.id);
            return null;
        }
        Item item = BuiltInRegistries.ITEM.getOptional(identifier).orElse(null);
        if (item == null) {
            Constants.LOG.warn("Skipping unknown kit item '{}'", (Object)identifier);
            return null;
        }
        int count = stored.count == null ? 1 : Math.max(1, stored.count);
        ItemStack stack = new ItemStack((ItemLike)item, count);
        return stack;
    }

    private static void ensureConfigurationExists() {
        if (Files.exists(CONFIG_PATH, new LinkOption[0])) {
            return;
        }
        try {
            Files.createDirectories(CONFIG_PATH.getParent(), new FileAttribute[0]);
            KitManager.writeKits(Collections.emptyList());
        }
        catch (IOException exception) {
            Constants.LOG.error("Failed to create kit configuration file {}", (Object)CONFIG_PATH.toAbsolutePath(), (Object)exception);
        }
    }

    private static void writeKits(Collection<KitDefinition> definitions) {
        try {
            Files.createDirectories(CONFIG_PATH.getParent(), new FileAttribute[0]);
        }
        catch (IOException exception) {
            Constants.LOG.error("Failed to create kit configuration directory {}", (Object)CONFIG_PATH.getParent(), (Object)exception);
            return;
        }
        List stored = definitions.stream().map(KitManager::toStoredKit).collect(Collectors.toCollection(ArrayList::new));
        KitsFile file = new KitsFile(stored);
        try (BufferedWriter writer = Files.newBufferedWriter(CONFIG_PATH, StandardCharsets.UTF_8, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);){
            GSON.toJson((Object)file, (Appendable)writer);
        }
        catch (IOException exception) {
            Constants.LOG.error("Failed to write kit configuration to {}", (Object)CONFIG_PATH.toAbsolutePath(), (Object)exception);
        }
    }

    private static StoredKit toStoredKit(KitDefinition definition) {
        ArrayList<StoredKitItem> items = new ArrayList<StoredKitItem>();
        for (ItemStack stack : definition.items()) {
            ResourceLocation id = BuiltInRegistries.ITEM.getKey((Object)stack.getItem());
            if (id == null) continue;
            items.add(new StoredKitItem(id.toString(), stack.getCount()));
        }
        return new StoredKit(definition.name(), definition.cooldownSeconds(), items);
    }

    private static String normalise(String name) {
        return name.toLowerCase(Locale.ROOT);
    }

    public record KitDefinition(String name, long cooldownSeconds, List<ItemStack> items) {
        public KitDefinition {
            Objects.requireNonNull(name, "name");
            Objects.requireNonNull(items, "items");
        }
    }

    private record KitsFile(List<StoredKit> kits) {
    }

    private record StoredKit(String name, Long cooldownSeconds, List<StoredKitItem> items) {
    }

    private record StoredKitItem(String id, Integer count) {
    }
}

