/*
 * Decompiled with CFR 0.152.
 */
package com.foundryx.managers;

import com.foundryx.Constants;
import com.foundryx.data.WarpData;
import com.foundryx.storage.FoundryxDataStorage;
import com.foundryx.storage.database.DatabaseContext;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Reader;
import java.lang.reflect.Type;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Stream;
import net.minecraft.server.level.ServerPlayer;

public final class WarpDataManager {
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private static final Type LOCATION_MAP_TYPE = new TypeToken<Map<String, FoundryxDataStorage.LocationSnapshot>>(){}.getType();
    private final Path directory;
    private final Map<UUID, WarpData> cache = new HashMap<UUID, WarpData>();
    private final Map<String, WarpEntry> warpsByName = new HashMap<String, WarpEntry>();
    private final boolean useDatabase;
    private final DatabaseContext database;

    public WarpDataManager(Path directory) {
        this(directory, null);
    }

    public WarpDataManager(Path directory, DatabaseContext database) {
        this.directory = directory;
        this.database = database;
        boolean bl = this.useDatabase = database != null && database.isAvailable();
        if (this.useDatabase) {
            this.loadAllFromDatabase();
        } else {
            this.loadAll();
        }
    }

    public void setWarp(String name, ServerPlayer creator) {
        WarpData otherWarps;
        UUID owner = creator.getUUID();
        String key = WarpDataManager.normalise(name);
        WarpEntry existing = this.warpsByName.get(key);
        if (existing != null && !existing.owner().equals(owner) && (otherWarps = this.load(existing.owner())).remove(key)) {
            this.save(existing.owner(), otherWarps);
        }
        FoundryxDataStorage.LocationSnapshot snapshot = FoundryxDataStorage.LocationSnapshot.from(creator);
        if (this.useDatabase) {
            this.upsertWarp(owner, key, snapshot);
            this.warpsByName.put(key, new WarpEntry(owner, snapshot));
            return;
        }
        WarpData warps = this.load(owner);
        warps.set(name, snapshot);
        this.save(owner, warps);
        this.warpsByName.put(key, new WarpEntry(owner, warps.get(name).orElseThrow()));
    }

    public Optional<FoundryxDataStorage.TeleportTarget> getWarp(String name) {
        WarpEntry entry = this.warpsByName.get(WarpDataManager.normalise(name));
        if (entry == null) {
            return Optional.empty();
        }
        return entry.location().toTeleportTarget();
    }

    public boolean deleteWarp(String name) {
        String key = WarpDataManager.normalise(name);
        WarpEntry entry = this.warpsByName.get(key);
        if (entry == null) {
            return false;
        }
        if (this.useDatabase) {
            if (this.deleteWarpDatabase(entry.owner(), key)) {
                this.warpsByName.remove(key);
                return true;
            }
            return false;
        }
        this.warpsByName.remove(key);
        WarpData warps = this.load(entry.owner());
        if (!warps.remove(key)) {
            return false;
        }
        this.save(entry.owner(), warps);
        return true;
    }

    public Collection<String> getWarpNames() {
        if (this.warpsByName.isEmpty()) {
            return List.of();
        }
        ArrayList<String> names = new ArrayList<String>(this.warpsByName.keySet());
        names.sort(String::compareToIgnoreCase);
        return names;
    }

    private WarpData load(UUID owner) {
        if (this.useDatabase) {
            return this.readWarpsFromDatabase(owner);
        }
        return this.cache.computeIfAbsent(owner, this::readWarps);
    }

    private WarpData readWarps(UUID owner) {
        return this.readWarps(owner, this.filePath(owner));
    }

    private WarpData readWarps(UUID owner, Path file) {
        WarpData warpData;
        block9: {
            if (!Files.exists(file, new LinkOption[0])) {
                return new WarpData();
            }
            BufferedReader reader = Files.newBufferedReader(file);
            try {
                Map loaded = (Map)GSON.fromJson((Reader)reader, LOCATION_MAP_TYPE);
                warpData = new WarpData(loaded);
                if (reader == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (reader != null) {
                        try {
                            ((Reader)reader).close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException exception) {
                    Constants.LOG.error("Failed to read warp data for {}", (Object)owner, (Object)exception);
                    return new WarpData();
                }
            }
            ((Reader)reader).close();
        }
        return warpData;
    }

    private void save(UUID owner, WarpData warps) {
        if (this.useDatabase) {
            return;
        }
        Path file = this.filePath(owner);
        try {
            if (warps.isEmpty()) {
                this.cache.remove(owner);
                Files.deleteIfExists(file);
                return;
            }
            Files.createDirectories(this.directory, new FileAttribute[0]);
            try (BufferedWriter writer = Files.newBufferedWriter(file, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);){
                GSON.toJson(warps.asMap(), LOCATION_MAP_TYPE, (Appendable)writer);
            }
        }
        catch (IOException exception) {
            Constants.LOG.error("Failed to write warp data for {}", (Object)owner, (Object)exception);
        }
    }

    private void loadAll() {
        if (this.useDatabase) {
            return;
        }
        if (!Files.exists(this.directory, new LinkOption[0])) {
            return;
        }
        try (Stream<Path> paths = Files.list(this.directory);){
            paths.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).forEach(this::loadFile);
        }
        catch (IOException exception) {
            Constants.LOG.error("Failed to list warp data", (Throwable)exception);
        }
    }

    private void loadFile(Path file) {
        UUID owner = this.parseOwner(file);
        if (owner == null) {
            return;
        }
        WarpData warps = this.readWarps(owner, file);
        if (warps.isEmpty()) {
            return;
        }
        this.cache.put(owner, warps);
        warps.asMap().forEach((name, snapshot) -> this.warpsByName.put((String)name, new WarpEntry(owner, (FoundryxDataStorage.LocationSnapshot)snapshot)));
    }

    private UUID parseOwner(Path file) {
        String name = file.getFileName().toString();
        int dot = name.indexOf(46);
        if (dot < 0) {
            return null;
        }
        String uuid = name.substring(0, dot);
        try {
            return UUID.fromString(uuid);
        }
        catch (IllegalArgumentException exception) {
            Constants.LOG.warn("Skipping warp file with invalid UUID: {}", (Object)file);
            return null;
        }
    }

    private Path filePath(UUID owner) {
        return this.directory.resolve(owner.toString() + ".json");
    }

    private static String normalise(String name) {
        return name.toLowerCase(Locale.ROOT);
    }

    public void flush() {
    }

    private void loadAllFromDatabase() {
        this.warpsByName.clear();
        this.cache.clear();
        try (Connection connection = this.database.getConnection();
             PreparedStatement statement = connection.prepareStatement("SELECT owner_uuid, warp_name, dimension, x, y, z, yaw, pitch FROM foundryx_warps");
             ResultSet resultSet = statement.executeQuery();){
            while (resultSet.next()) {
                String ownerId = resultSet.getString("owner_uuid");
                String name = resultSet.getString("warp_name");
                if (ownerId == null || name == null) continue;
                UUID owner = this.parseUuid(ownerId);
                FoundryxDataStorage.LocationSnapshot snapshot = this.mapSnapshot(resultSet);
                if (owner == null || snapshot == null) continue;
                this.warpsByName.put(WarpDataManager.normalise(name), new WarpEntry(owner, snapshot));
            }
        }
        catch (SQLException exception) {
            Constants.LOG.error("Failed to load warp data from database", (Throwable)exception);
        }
    }

    private void upsertWarp(UUID owner, String key, FoundryxDataStorage.LocationSnapshot snapshot) {
        try (Connection connection = this.database.getConnection();){
            connection.setAutoCommit(false);
            try (PreparedStatement update = connection.prepareStatement("UPDATE foundryx_warps SET owner_uuid=?, dimension=?, x=?, y=?, z=?, yaw=?, pitch=? WHERE warp_name=?");){
                update.setString(1, owner.toString());
                this.bindLocation(update, 2, snapshot);
                update.setString(8, key);
                int updated = update.executeUpdate();
                if (updated == 0) {
                    try (PreparedStatement insert = connection.prepareStatement("INSERT INTO foundryx_warps (warp_name, owner_uuid, dimension, x, y, z, yaw, pitch) VALUES (?,?,?,?,?,?,?,?)");){
                        insert.setString(1, key);
                        insert.setString(2, owner.toString());
                        this.bindLocation(insert, 3, snapshot);
                        insert.executeUpdate();
                    }
                }
                connection.commit();
            }
            catch (SQLException exception) {
                connection.rollback();
                throw exception;
            }
            finally {
                connection.setAutoCommit(true);
            }
        }
        catch (SQLException exception) {
            Constants.LOG.error("Failed to persist warp '{}' for {}", new Object[]{key, owner, exception});
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private boolean deleteWarpDatabase(UUID owner, String key) {
        try (Connection connection = this.database.getConnection();){
            boolean bl;
            block14: {
                PreparedStatement statement = connection.prepareStatement("DELETE FROM foundryx_warps WHERE warp_name=?");
                try {
                    statement.setString(1, key);
                    boolean bl2 = bl = statement.executeUpdate() > 0;
                    if (statement == null) break block14;
                }
                catch (Throwable throwable) {
                    if (statement != null) {
                        try {
                            statement.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                statement.close();
            }
            return bl;
        }
        catch (SQLException exception) {
            Constants.LOG.error("Failed to delete warp '{}' for {}", new Object[]{key, owner, exception});
            return false;
        }
    }

    private WarpData readWarpsFromDatabase(UUID owner) {
        WarpData warps = new WarpData();
        if (owner == null) {
            return warps;
        }
        try (Connection connection = this.database.getConnection();
             PreparedStatement statement = connection.prepareStatement("SELECT warp_name, dimension, x, y, z, yaw, pitch FROM foundryx_warps WHERE owner_uuid=?");){
            statement.setString(1, owner.toString());
            try (ResultSet resultSet = statement.executeQuery();){
                while (resultSet.next()) {
                    String name = resultSet.getString("warp_name");
                    FoundryxDataStorage.LocationSnapshot snapshot = this.mapSnapshot(resultSet);
                    if (name == null || snapshot == null) continue;
                    warps.set(name, snapshot);
                }
            }
        }
        catch (SQLException exception) {
            Constants.LOG.error("Failed to load warps for {}", (Object)owner, (Object)exception);
        }
        return warps;
    }

    private void bindLocation(PreparedStatement statement, int offset, FoundryxDataStorage.LocationSnapshot snapshot) throws SQLException {
        statement.setString(offset, snapshot.dimension());
        statement.setDouble(offset + 1, snapshot.x());
        statement.setDouble(offset + 2, snapshot.y());
        statement.setDouble(offset + 3, snapshot.z());
        statement.setFloat(offset + 4, snapshot.yaw());
        statement.setFloat(offset + 5, snapshot.pitch());
    }

    private FoundryxDataStorage.LocationSnapshot mapSnapshot(ResultSet resultSet) throws SQLException {
        String dimension = resultSet.getString("dimension");
        if (dimension == null) {
            return null;
        }
        double x = resultSet.getDouble("x");
        double y = resultSet.getDouble("y");
        double z = resultSet.getDouble("z");
        float yaw = resultSet.getFloat("yaw");
        float pitch = resultSet.getFloat("pitch");
        return new FoundryxDataStorage.LocationSnapshot(dimension, x, y, z, yaw, pitch);
    }

    private UUID parseUuid(String value) {
        try {
            return UUID.fromString(value);
        }
        catch (IllegalArgumentException exception) {
            return null;
        }
    }

    private record WarpEntry(UUID owner, FoundryxDataStorage.LocationSnapshot location) {
    }
}

