/*
 * Decompiled with CFR 0.152.
 */
package com.foundryx.storage;

import com.foundryx.Constants;
import com.foundryx.config.ModConfig;
import com.foundryx.data.DeathData;
import com.foundryx.data.PlayerWhoisData;
import com.foundryx.managers.BackDataManager;
import com.foundryx.managers.BanDataManager;
import com.foundryx.managers.DeathDataManager;
import com.foundryx.managers.HomeDataManager;
import com.foundryx.managers.JailDataManager;
import com.foundryx.managers.SpawnManager;
import com.foundryx.managers.WarpDataManager;
import com.foundryx.storage.MailDataManager;
import com.foundryx.storage.UserDataManager;
import com.foundryx.storage.database.DatabaseContext;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import java.nio.file.Path;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.WeakHashMap;
import net.minecraft.core.BlockPos;
import net.minecraft.core.registries.Registries;
import net.minecraft.resources.ResourceKey;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.storage.LevelResource;

public final class FoundryxDataStorage {
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private static final Map<MinecraftServer, FoundryxDataStorage> INSTANCES = new WeakHashMap<MinecraftServer, FoundryxDataStorage>();
    private static final long AUTO_SAVE_INTERVAL_TICKS = 6000L;
    private final WarpDataManager warpManager;
    private final HomeDataManager homeManager;
    private final SpawnManager spawnManager;
    private final JailDataManager jailManager;
    private final UserDataManager userDataManager;
    private final MailDataManager mailDataManager;
    private final BanDataManager banDataManager;
    private final BackDataManager backDataManager;
    private final DeathDataManager deathDataManager;
    private final DatabaseContext databaseContext;
    private final boolean useDatabase;
    private long ticksUntilAutoSave;

    private FoundryxDataStorage(Path rootDirectory) {
        DatabaseContext databaseContext = null;
        if (ModConfig.get().storage().useDatabase() && (databaseContext = DatabaseContext.create(ModConfig.get().storage().database(), rootDirectory)) == null) {
            Constants.LOG.warn("Falling back to file-based storage because database configuration failed to initialise");
        }
        this.databaseContext = databaseContext;
        this.useDatabase = databaseContext != null && databaseContext.isAvailable();
        this.warpManager = new WarpDataManager(rootDirectory.resolve("warps"), this.databaseContext);
        this.homeManager = new HomeDataManager(rootDirectory.resolve("homes"), this.databaseContext);
        this.spawnManager = new SpawnManager(rootDirectory.resolve("spawn"), GSON, this.databaseContext);
        this.jailManager = new JailDataManager(rootDirectory.resolve("jail"), this.databaseContext);
        Path userDataDirectory = rootDirectory.resolve("userdata");
        MailDataManager mailDataManager = new MailDataManager(rootDirectory.resolve("mail"), userDataDirectory, GSON, this.databaseContext);
        this.userDataManager = new UserDataManager(userDataDirectory, mailDataManager, GSON, this.databaseContext);
        this.mailDataManager = mailDataManager;
        this.banDataManager = new BanDataManager(rootDirectory.resolve("bans"));
        this.backDataManager = new BackDataManager(rootDirectory.resolve("back"), this.databaseContext);
        this.deathDataManager = new DeathDataManager(rootDirectory.resolve("deaths"), this.databaseContext);
        this.ticksUntilAutoSave = 6000L;
    }

    public static FoundryxDataStorage get(MinecraftServer server) {
        return INSTANCES.computeIfAbsent(server, FoundryxDataStorage::loadFromServer);
    }

    private static FoundryxDataStorage loadFromServer(MinecraftServer server) {
        Path root = server.getWorldPath(LevelResource.ROOT);
        Path dataRoot = root.resolve("foundryx");
        return new FoundryxDataStorage(dataRoot);
    }

    public DatabaseContext getDatabaseContext() {
        return this.databaseContext;
    }

    public void tick() {
        if (!this.useDatabase) {
            return;
        }
        --this.ticksUntilAutoSave;
        if (this.ticksUntilAutoSave <= 0L) {
            this.autoSave();
            this.ticksUntilAutoSave = 6000L;
        }
    }

    private void autoSave() {
        this.homeManager.flush();
        this.warpManager.flush();
        this.spawnManager.flush();
        this.jailManager.flush();
        this.userDataManager.flush();
        this.mailDataManager.flush();
        this.backDataManager.flush();
        this.deathDataManager.flush();
    }

    public Optional<TeleportTarget> getSpawn() {
        return this.spawnManager.getSpawn();
    }

    public void setSpawn(ServerPlayer player) {
        this.spawnManager.setSpawn(LocationSnapshot.from(player));
    }

    public void setHome(UUID player, String name, ServerPlayer owner) {
        this.homeManager.setHome(player, name, owner);
    }

    public Optional<TeleportTarget> getHome(UUID player, String name) {
        return this.homeManager.getHome(player, name);
    }

    public boolean deleteHome(UUID player, String name) {
        return this.homeManager.deleteHome(player, name);
    }

    public Collection<String> getHomeNames(UUID player) {
        return this.homeManager.getHomeNames(player);
    }

    public void setWarp(String name, ServerPlayer creator) {
        this.warpManager.setWarp(name, creator);
    }

    public Optional<TeleportTarget> getWarp(String name) {
        return this.warpManager.getWarp(name);
    }

    public boolean deleteWarp(String name) {
        return this.warpManager.deleteWarp(name);
    }

    public Collection<String> getWarpNames() {
        return this.warpManager.getWarpNames();
    }

    public void rememberBack(ServerPlayer player) {
        this.backDataManager.rememberBack(player.getUUID(), LocationSnapshot.from(player));
    }

    public void rememberBack(UUID player, TeleportTarget target) {
        if (player == null || target == null) {
            return;
        }
        String dimension = target.dimension().location().toString();
        LocationSnapshot snapshot = new LocationSnapshot(dimension, target.x(), target.y(), target.z(), target.yaw(), target.pitch());
        this.backDataManager.rememberBack(player, snapshot);
    }

    public Optional<String> getNickname(UUID player) {
        return this.userDataManager.getNickname(player);
    }

    public void setNickname(UUID player, String nickname) {
        this.userDataManager.setNickname(player, nickname);
    }

    public BanDataManager getBanDataManager() {
        return this.banDataManager;
    }

    public Optional<TeleportTarget> consumeBack(UUID player) {
        return this.backDataManager.consumeBack(player);
    }

    public void recordDeath(ServerPlayer player, String cause) {
        if (player == null) {
            return;
        }
        LocationSnapshot snapshot = LocationSnapshot.from(player);
        this.deathDataManager.recordDeath(player.getUUID(), snapshot, cause, System.currentTimeMillis());
    }

    public Optional<DeathData> getDeathData(UUID player) {
        return this.deathDataManager.get(player);
    }

    public Optional<TeleportTarget> getLastDeathLocation(UUID player) {
        return this.deathDataManager.getTeleportTarget(player);
    }

    public boolean isJailed(UUID player) {
        return this.userDataManager.isJailed(player);
    }

    public void jailPlayer(UUID player, LocationSnapshot originalLocation, LocationSnapshot jailLocation, Long releaseAtMillis) {
        this.userDataManager.jailPlayer(player, originalLocation, jailLocation, releaseAtMillis);
    }

    public Optional<JailRecord> unjailPlayer(UUID player) {
        return this.userDataManager.unjailPlayer(player);
    }

    public Optional<JailRecord> getJailRecord(UUID player) {
        return this.userDataManager.getJailRecord(player);
    }

    public Optional<LocationSnapshot> getJailLocation(UUID player) {
        return this.userDataManager.getJailLocation(player);
    }

    public void setJailRegion(ResourceKey<Level> dimension, BlockPos first, BlockPos second, float yaw, float pitch) {
        this.jailManager.setRegion(dimension, first, second, yaw, pitch);
    }

    public Optional<JailRegion> getJailRegion() {
        return this.jailManager.getRegion();
    }

    public boolean deleteJailRegion() {
        return this.jailManager.deleteRegion();
    }

    public void setFrozen(UUID player, boolean frozen) {
        this.userDataManager.setFrozen(player, frozen);
    }

    public boolean isFrozen(UUID player) {
        return this.userDataManager.isFrozen(player);
    }

    public void setGodMode(UUID player, boolean enabled) {
        this.userDataManager.setGodMode(player, enabled);
    }

    public boolean hasGodMode(UUID player) {
        return this.userDataManager.hasGodMode(player);
    }

    public void mutePlayer(UUID player, long expiresAtMillis) {
        this.mutePlayer(player, expiresAtMillis, null);
    }

    public void mutePlayer(UUID player, long expiresAtMillis, String reason) {
        this.userDataManager.mutePlayer(player, expiresAtMillis, reason);
    }

    public void unmutePlayer(UUID player) {
        this.userDataManager.unmutePlayer(player);
    }

    public Optional<Long> getMuteExpiry(UUID player) {
        return this.userDataManager.getMuteExpiry(player);
    }

    public Optional<String> getMuteReason(UUID player) {
        return this.userDataManager.getMuteReason(player);
    }

    public boolean isMuted(UUID player) {
        return this.userDataManager.isMuted(player);
    }

    public void updateWhoisSnapshot(ServerPlayer player) {
        this.userDataManager.updateWhoisSnapshot(player);
    }

    public void recordLogin(ServerPlayer player) {
        this.userDataManager.recordLogin(player);
    }

    public void recordLogout(ServerPlayer player) {
        this.userDataManager.recordLogout(player);
    }

    public Optional<PlayerWhoisData> getWhoisData(UUID player) {
        return this.userDataManager.getWhoisData(player);
    }

    public Optional<UUID> findPlayerUuid(String name) {
        return this.userDataManager.findUuidByName(name);
    }

    public void addMail(UUID target, MailMessage message) {
        this.mailDataManager.addMail(target, message);
    }

    public List<MailMessage> getMail(UUID player) {
        return this.mailDataManager.getMail(player);
    }

    public List<MailMessage> clearMail(UUID player) {
        return this.mailDataManager.clearMail(player);
    }

    public long getRemainingKitCooldown(UUID player, String kitName, long cooldownSeconds) {
        return this.userDataManager.getRemainingKitCooldown(player, kitName, cooldownSeconds);
    }

    public void recordKitUsage(UUID player, String kitName, long usedAtMillis) {
        this.userDataManager.recordKitUsage(player, kitName, usedAtMillis);
    }

    public static void clearKitUsageForAll(String kitName) {
        INSTANCES.values().forEach(storage -> storage.userDataManager.clearKitUsage(kitName));
    }

    public double getBalance(UUID player) {
        return this.userDataManager.getBalance(player);
    }

    public void setBalance(UUID player, double amount) {
        this.userDataManager.setBalance(player, amount);
    }

    public void addBalance(UUID player, double amount) {
        this.userDataManager.addBalance(player, amount);
    }

    public boolean removeBalance(UUID player, double amount) {
        return this.userDataManager.removeBalance(player, amount);
    }

    public record LocationSnapshot(String dimension, double x, double y, double z, float yaw, float pitch) {
        public LocationSnapshot {
            Objects.requireNonNull(dimension, "dimension");
        }

        public static LocationSnapshot from(ServerPlayer player) {
            ResourceLocation location = player.level().dimension().location();
            return new LocationSnapshot(location.toString(), player.getX(), player.getY(), player.getZ(), player.getYRot(), player.getXRot());
        }

        public Optional<TeleportTarget> toTeleportTarget() {
            ResourceLocation id = ResourceLocation.tryParse((String)this.dimension);
            if (id == null) {
                return Optional.empty();
            }
            ResourceKey key = ResourceKey.create((ResourceKey)Registries.DIMENSION, (ResourceLocation)id);
            return Optional.of(new TeleportTarget((ResourceKey<Level>)key, this.x, this.y, this.z, this.yaw, this.pitch));
        }
    }

    public record TeleportTarget(ResourceKey<Level> dimension, double x, double y, double z, float yaw, float pitch) {
        public BlockPos toBlockPos() {
            return BlockPos.containing((double)this.x, (double)this.y, (double)this.z);
        }
    }

    public record MailMessage(UUID sender, long timestamp, String message) {
        public MailMessage {
            Objects.requireNonNull(message, "message");
        }
    }

    public record BlockPosition(int x, int y, int z) {
    }

    public record JailRegion(String dimension, BlockPosition minimum, BlockPosition maximum, float yaw, float pitch) {
        public JailRegion {
            Objects.requireNonNull(dimension, "dimension");
            Objects.requireNonNull(minimum, "minimum");
            Objects.requireNonNull(maximum, "maximum");
        }

        public LocationSnapshot toSnapshot() {
            double minX = Math.min(this.minimum.x(), this.maximum.x());
            double minY = Math.min(this.minimum.y(), this.maximum.y());
            double minZ = Math.min(this.minimum.z(), this.maximum.z());
            double maxX = Math.max(this.minimum.x(), this.maximum.x());
            double maxY = Math.max(this.minimum.y(), this.maximum.y());
            double maxZ = Math.max(this.minimum.z(), this.maximum.z());
            double x = (minX + maxX) / 2.0 + 0.5;
            double y = (minY + maxY) / 2.0 + 0.5;
            double z = (minZ + maxZ) / 2.0 + 0.5;
            return new LocationSnapshot(this.dimension, x, y, z, this.yaw, this.pitch);
        }

        public boolean contains(ServerPlayer player) {
            ResourceLocation location = ResourceLocation.tryParse((String)this.dimension);
            if (location == null) {
                return false;
            }
            if (!player.level().dimension().location().equals((Object)location)) {
                return false;
            }
            BlockPos position = player.blockPosition();
            int minX = Math.min(this.minimum.x(), this.maximum.x());
            int minY = Math.min(this.minimum.y(), this.maximum.y());
            int minZ = Math.min(this.minimum.z(), this.maximum.z());
            int maxX = Math.max(this.minimum.x(), this.maximum.x());
            int maxY = Math.max(this.minimum.y(), this.maximum.y());
            int maxZ = Math.max(this.minimum.z(), this.maximum.z());
            return position.getX() >= minX && position.getX() <= maxX && position.getY() >= minY && position.getY() <= maxY && position.getZ() >= minZ && position.getZ() <= maxZ;
        }
    }

    public record JailRecord(LocationSnapshot originalLocation, LocationSnapshot jailLocation, Long releaseAtMillis) {
    }
}

