/*
 * Decompiled with CFR 0.152.
 */
package com.foundryx.storage;

import com.foundryx.Constants;
import com.foundryx.data.KitData;
import com.foundryx.storage.FoundryxDataStorage;
import com.foundryx.storage.database.DatabaseContext;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.WeakHashMap;
import java.util.stream.Stream;
import net.minecraft.core.HolderLookup;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.NbtAccounter;
import net.minecraft.nbt.NbtIo;
import net.minecraft.server.MinecraftServer;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.storage.LevelResource;

public final class KitStorage {
    private static final Map<MinecraftServer, KitStorage> INSTANCES = new WeakHashMap<MinecraftServer, KitStorage>();
    private final MinecraftServer server;
    private final Path directory;
    private final Map<String, KitData> kits = new HashMap<String, KitData>();
    private final boolean useDatabase;
    private final DatabaseContext database;

    private KitStorage(MinecraftServer server) {
        this.server = server;
        Path root = server.getWorldPath(LevelResource.ROOT).resolve("foundryx");
        this.directory = root.resolve("kits");
        FoundryxDataStorage storage = FoundryxDataStorage.get(server);
        this.database = storage.getDatabaseContext();
        boolean bl = this.useDatabase = this.database != null && this.database.isAvailable();
        if (this.useDatabase) {
            this.loadAllFromDatabase();
        } else {
            this.loadAllFromFiles();
        }
    }

    public static KitStorage get(MinecraftServer server) {
        Objects.requireNonNull(server, "server");
        return INSTANCES.computeIfAbsent(server, KitStorage::new);
    }

    public KitData saveKit(String name, List<ItemStack> items, long cooldownSeconds) {
        Objects.requireNonNull(name, "name");
        KitData data = new KitData(name.trim(), items, cooldownSeconds);
        String key = KitStorage.normalise(data.name());
        this.kits.put(key, data);
        if (this.useDatabase) {
            this.saveToDatabase(key, data);
        } else {
            this.writeToFile(key, data);
        }
        return data;
    }

    public Optional<KitData> getKit(String name) {
        if (name == null) {
            return Optional.empty();
        }
        return Optional.ofNullable(this.kits.get(KitStorage.normalise(name)));
    }

    public List<KitData> getAllKits() {
        if (this.kits.isEmpty()) {
            return List.of();
        }
        return List.copyOf(this.kits.values());
    }

    private void loadAllFromFiles() {
        if (!Files.exists(this.directory, new LinkOption[0])) {
            return;
        }
        try (Stream<Path> files = Files.list(this.directory);){
            files.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).forEach(this::loadFromFile);
        }
        catch (IOException exception) {
            Constants.LOG.error("Failed to list kit data", (Throwable)exception);
        }
    }

    private void loadAllFromDatabase() {
        this.kits.clear();
        try (Connection connection = this.database.getConnection();
             PreparedStatement statement = connection.prepareStatement("SELECT kit_name, cooldown_seconds, nbt_payload FROM foundryx_kits");
             ResultSet resultSet = statement.executeQuery();){
            while (resultSet.next()) {
                KitData data;
                String name = resultSet.getString("kit_name");
                if (name == null || (data = this.readKitFromDatabase(resultSet)) == null) continue;
                this.kits.put(KitStorage.normalise(name), data);
            }
        }
        catch (SQLException exception) {
            Constants.LOG.error("Failed to load kits from database", (Throwable)exception);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private KitData readKitFromDatabase(ResultSet resultSet) throws SQLException {
        String payload;
        String name = resultSet.getString("kit_name");
        long cooldown = resultSet.getLong("cooldown_seconds");
        if (resultSet.wasNull()) {
            cooldown = 0L;
        }
        if ((payload = resultSet.getString("nbt_payload")) == null) return new KitData(name, List.of(), cooldown);
        if (payload.isBlank()) {
            return new KitData(name, List.of(), cooldown);
        }
        byte[] bytes = Base64.getDecoder().decode(payload);
        try (ByteArrayInputStream input = new ByteArrayInputStream(bytes);){
            CompoundTag tag = NbtIo.readCompressed((InputStream)input, (NbtAccounter)NbtAccounter.unlimitedHeap());
            KitData loaded = KitData.load(tag, (HolderLookup.Provider)this.server.registryAccess());
            if (loaded == null) {
                KitData kitData = new KitData(name, List.of(), cooldown);
                return kitData;
            }
            long effectiveCooldown = cooldown > 0L ? cooldown : loaded.cooldownSeconds();
            KitData kitData = new KitData(loaded.name(), loaded.items(), effectiveCooldown);
            return kitData;
        }
        catch (IOException exception) {
            Constants.LOG.error("Failed to decode kit '{}' from database", (Object)name, (Object)exception);
            return null;
        }
    }

    private void loadFromFile(Path path) {
        try (InputStream input = Files.newInputStream(path, StandardOpenOption.READ);){
            CompoundTag tag = NbtIo.readCompressed((InputStream)input, (NbtAccounter)NbtAccounter.unlimitedHeap());
            KitData data = KitData.load(tag, (HolderLookup.Provider)this.server.registryAccess());
            if (data != null) {
                this.kits.put(KitStorage.normalise(data.name()), data);
            }
        }
        catch (IOException exception) {
            Constants.LOG.error("Failed to read kit data from {}", (Object)path, (Object)exception);
        }
    }

    private void writeToFile(String key, KitData data) {
        Path file = this.directory.resolve(key + ".nbt");
        try {
            Files.createDirectories(this.directory, new FileAttribute[0]);
            CompoundTag tag = data.save((HolderLookup.Provider)this.server.registryAccess());
            try (OutputStream output = Files.newOutputStream(file, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);){
                NbtIo.writeCompressed((CompoundTag)tag, (OutputStream)output);
            }
        }
        catch (IOException exception) {
            Constants.LOG.error("Failed to write kit '{}'", (Object)data.name(), (Object)exception);
        }
    }

    private void saveToDatabase(String key, KitData data) {
        String payload = this.encodeKit(data);
        try (Connection connection = this.database.getConnection();){
            connection.setAutoCommit(false);
            try (PreparedStatement update = connection.prepareStatement("UPDATE foundryx_kits SET cooldown_seconds=?, nbt_payload=? WHERE kit_name=?");){
                this.setNullableLong(update, 1, data.cooldownSeconds());
                this.setNullableString(update, 2, payload);
                update.setString(3, key);
                int updated = update.executeUpdate();
                if (updated == 0) {
                    try (PreparedStatement insert = connection.prepareStatement("INSERT INTO foundryx_kits (kit_name, cooldown_seconds, nbt_payload) VALUES (?,?,?)");){
                        insert.setString(1, key);
                        this.setNullableLong(insert, 2, data.cooldownSeconds());
                        this.setNullableString(insert, 3, payload);
                        insert.executeUpdate();
                    }
                }
                connection.commit();
            }
            catch (SQLException exception) {
                connection.rollback();
                throw exception;
            }
            finally {
                connection.setAutoCommit(true);
            }
        }
        catch (SQLException exception) {
            Constants.LOG.error("Failed to persist kit '{}'", (Object)data.name(), (Object)exception);
        }
    }

    private String encodeKit(KitData data) {
        String string;
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        try {
            CompoundTag tag = data.save((HolderLookup.Provider)this.server.registryAccess());
            NbtIo.writeCompressed((CompoundTag)tag, (OutputStream)output);
            byte[] bytes = output.toByteArray();
            string = bytes.length == 0 ? null : Base64.getEncoder().encodeToString(bytes);
        }
        catch (Throwable throwable) {
            try {
                try {
                    output.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException exception) {
                Constants.LOG.error("Failed to encode kit '{}'", (Object)data.name(), (Object)exception);
                return null;
            }
        }
        output.close();
        return string;
    }

    private void setNullableString(PreparedStatement statement, int index, String value) throws SQLException {
        if (value == null || value.isBlank()) {
            statement.setNull(index, -1);
        } else {
            statement.setString(index, value);
        }
    }

    private void setNullableLong(PreparedStatement statement, int index, long value) throws SQLException {
        if (value <= 0L) {
            statement.setNull(index, -5);
        } else {
            statement.setLong(index, value);
        }
    }

    private static String normalise(String name) {
        return name.toLowerCase(Locale.ROOT);
    }
}

