/*
 * Decompiled with CFR 0.152.
 */
package com.foundryx.storage;

import com.foundryx.Constants;
import com.foundryx.storage.FoundryxDataStorage;
import com.foundryx.storage.MailData;
import com.foundryx.storage.database.DatabaseContext;
import com.google.gson.Gson;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Reader;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

final class MailDataManager {
    private final Path directory;
    private final Gson gson;
    private final Map<UUID, MailData> cache = new HashMap<UUID, MailData>();
    private final Path legacyUserDataDirectory;
    private final boolean useDatabase;
    private final DatabaseContext database;

    MailDataManager(Path directory, Path legacyUserDataDirectory, Gson gson) {
        this(directory, legacyUserDataDirectory, gson, null);
    }

    MailDataManager(Path directory, Path legacyUserDataDirectory, Gson gson, DatabaseContext database) {
        this.directory = directory;
        this.legacyUserDataDirectory = legacyUserDataDirectory;
        this.gson = gson;
        this.database = database;
        this.useDatabase = database != null && database.isAvailable();
    }

    void addMail(UUID player, FoundryxDataStorage.MailMessage message) {
        if (message == null || message.message() == null) {
            return;
        }
        if (this.useDatabase) {
            this.insertMailRecord(player, message);
            return;
        }
        MailData data = this.load(player);
        data.messages.add(message);
        this.save(player, data);
    }

    List<FoundryxDataStorage.MailMessage> getMail(UUID player) {
        if (this.useDatabase) {
            return this.getMailFromDatabase(player);
        }
        return List.copyOf(this.load((UUID)player).messages);
    }

    List<FoundryxDataStorage.MailMessage> clearMail(UUID player) {
        if (this.useDatabase) {
            return this.clearMailFromDatabase(player);
        }
        MailData data = this.load(player);
        if (data.messages.isEmpty()) {
            return List.of();
        }
        ArrayList<FoundryxDataStorage.MailMessage> removed = new ArrayList<FoundryxDataStorage.MailMessage>(data.messages);
        data.messages.clear();
        this.save(player, data);
        return removed;
    }

    void ensureMailMigrated(UUID player) {
        if (this.useDatabase) {
            this.migrateFileMailToDatabase(player);
            return;
        }
        this.load(player);
    }

    private MailData load(UUID player) {
        if (this.useDatabase) {
            return this.readMailFromDatabase(player);
        }
        return this.cache.computeIfAbsent(player, this::readMailData);
    }

    private MailData readMailData(UUID player) {
        MailData data;
        block13: {
            data = new MailData();
            Path file = this.filePath(player);
            if (!Files.exists(file, new LinkOption[0])) {
                this.migrateLegacyMail(player, data, null);
                return data;
            }
            try (BufferedReader reader = Files.newBufferedReader(file);){
                StoredMailData persisted = (StoredMailData)this.gson.fromJson((Reader)reader, StoredMailData.class);
                if (persisted == null || persisted.mail() == null) break block13;
                for (StoredMail entry : persisted.mail()) {
                    if (entry == null || entry.message() == null) continue;
                    UUID sender = null;
                    if (entry.sender() != null) {
                        try {
                            sender = UUID.fromString(entry.sender());
                        }
                        catch (IllegalArgumentException exception) {
                            Constants.LOG.warn("Skipping invalid mail sender '{}' for {}", new Object[]{entry.sender(), player, exception});
                        }
                    }
                    data.messages.add(new FoundryxDataStorage.MailMessage(sender, entry.timestamp(), entry.message()));
                }
            }
            catch (IOException exception) {
                Constants.LOG.error("Failed to read mail data for {}", (Object)player, (Object)exception);
            }
        }
        return data;
    }

    private boolean migrateLegacyMail(UUID player, MailData data, Set<MailMessageKey> knownMessages) {
        block18: {
            if (this.legacyUserDataDirectory == null) {
                return false;
            }
            Path legacyFile = this.legacyUserDataDirectory.resolve(player.toString() + ".json");
            if (!Files.exists(legacyFile, new LinkOption[0])) {
                return false;
            }
            try (BufferedReader reader = Files.newBufferedReader(legacyFile);){
                LegacyUserData legacy = (LegacyUserData)this.gson.fromJson((Reader)reader, LegacyUserData.class);
                if (legacy == null || legacy.mail() == null) break block18;
                for (StoredMail entry : legacy.mail()) {
                    if (entry == null || entry.message() == null) continue;
                    UUID sender = null;
                    if (entry.sender() != null) {
                        try {
                            sender = UUID.fromString(entry.sender());
                        }
                        catch (IllegalArgumentException exception) {
                            Constants.LOG.warn("Skipping invalid mail sender '{}' for {}", new Object[]{entry.sender(), player, exception});
                        }
                    }
                    data.messages.add(new FoundryxDataStorage.MailMessage(sender, entry.timestamp(), entry.message()));
                }
            }
            catch (IOException exception) {
                Constants.LOG.error("Failed to migrate mail data for {}", (Object)player, (Object)exception);
            }
        }
        if (data.messages.isEmpty()) {
            this.deleteLegacyMailFile(player);
            return false;
        }
        boolean migrated = false;
        if (this.useDatabase) {
            for (FoundryxDataStorage.MailMessage message : data.messages) {
                if (message == null || message.message() == null || knownMessages != null && !knownMessages.add(MailMessageKey.of(message))) continue;
                this.insertMailRecord(player, message);
                migrated = true;
            }
        } else {
            this.save(player, data);
            migrated = true;
        }
        this.deleteLegacyMailFile(player);
        return migrated;
    }

    private void save(UUID player, MailData data) {
        if (this.useDatabase) {
            return;
        }
        Path file = this.filePath(player);
        try {
            if (data.isEmpty()) {
                this.cache.remove(player);
                Files.deleteIfExists(file);
                return;
            }
            Files.createDirectories(this.directory, new FileAttribute[0]);
            StoredMailData persisted = new StoredMailData(this.toStoredMail(data.messages));
            try (BufferedWriter writer = Files.newBufferedWriter(file, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);){
                this.gson.toJson((Object)persisted, (Appendable)writer);
            }
        }
        catch (IOException exception) {
            Constants.LOG.error("Failed to write mail data for {}", (Object)player, (Object)exception);
        }
    }

    private Path filePath(UUID player) {
        return this.directory.resolve(player.toString() + ".json");
    }

    void flush() {
    }

    private MailData readMailFromDatabase(UUID player) {
        MailData data = new MailData();
        if (player == null) {
            return data;
        }
        try (Connection connection = this.database.getConnection();
             PreparedStatement statement = connection.prepareStatement("SELECT mail_id, sender_uuid, sent_at, message FROM foundryx_mail WHERE player_uuid=? ORDER BY sent_at ASC");){
            statement.setString(1, player.toString());
            try (ResultSet resultSet = statement.executeQuery();){
                while (resultSet.next()) {
                    data.messages.add(this.mapMessage(resultSet));
                }
            }
        }
        catch (SQLException exception) {
            Constants.LOG.error("Failed to load mail for {}", (Object)player, (Object)exception);
        }
        return data;
    }

    private void insertMailRecord(UUID player, FoundryxDataStorage.MailMessage message) {
        try (Connection connection = this.database.getConnection();
             PreparedStatement statement = connection.prepareStatement("INSERT INTO foundryx_mail (mail_id, player_uuid, sender_uuid, sent_at, message) VALUES (?,?,?,?,?)");){
            statement.setString(1, UUID.randomUUID().toString());
            statement.setString(2, player.toString());
            if (message.sender() == null) {
                statement.setNull(3, 12);
            } else {
                statement.setString(3, message.sender().toString());
            }
            statement.setLong(4, message.timestamp());
            statement.setString(5, message.message());
            statement.executeUpdate();
        }
        catch (SQLException exception) {
            Constants.LOG.error("Failed to store mail for {}", (Object)player, (Object)exception);
        }
    }

    private List<FoundryxDataStorage.MailMessage> getMailFromDatabase(UUID player) {
        return List.copyOf(this.readMailFromDatabase((UUID)player).messages);
    }

    private List<FoundryxDataStorage.MailMessage> clearMailFromDatabase(UUID player) {
        MailData data = this.readMailFromDatabase(player);
        List<FoundryxDataStorage.MailMessage> existing = List.copyOf(data.messages);
        if (existing.isEmpty()) {
            return List.of();
        }
        try (Connection connection = this.database.getConnection();
             PreparedStatement statement = connection.prepareStatement("DELETE FROM foundryx_mail WHERE player_uuid=?");){
            statement.setString(1, player.toString());
            statement.executeUpdate();
        }
        catch (SQLException exception) {
            Constants.LOG.error("Failed to clear mail for {}", (Object)player, (Object)exception);
        }
        return existing;
    }

    private void migrateFileMailToDatabase(UUID player) {
        if (player == null) {
            return;
        }
        Path file = this.filePath(player);
        MailData existing = this.readMailFromDatabase(player);
        HashSet<MailMessageKey> knownMessages = new HashSet<MailMessageKey>();
        for (FoundryxDataStorage.MailMessage message : existing.messages) {
            if (message == null || message.message() == null) continue;
            knownMessages.add(MailMessageKey.of(message));
        }
        if (Files.exists(file, new LinkOption[0])) {
            MailData data = this.readMailData(player);
            if (!data.messages.isEmpty()) {
                for (FoundryxDataStorage.MailMessage message : data.messages) {
                    MailMessageKey key;
                    if (message == null || message.message() == null || !knownMessages.add(key = MailMessageKey.of(message))) continue;
                    this.insertMailRecord(player, message);
                }
            }
        } else {
            this.migrateLegacyMail(player, new MailData(), knownMessages);
        }
        try {
            Files.deleteIfExists(file);
        }
        catch (IOException exception) {
            Constants.LOG.warn("Failed to delete migrated mail file for {}", (Object)player, (Object)exception);
        }
        this.deleteLegacyMailFile(player);
    }

    private void deleteLegacyMailFile(UUID player) {
        if (this.legacyUserDataDirectory == null) {
            return;
        }
        Path legacyFile = this.legacyUserDataDirectory.resolve(player.toString() + ".json");
        try {
            Files.deleteIfExists(legacyFile);
        }
        catch (IOException exception) {
            Constants.LOG.warn("Failed to delete legacy mail file for {}", (Object)player, (Object)exception);
        }
    }

    private FoundryxDataStorage.MailMessage mapMessage(ResultSet resultSet) throws SQLException {
        String senderId = resultSet.getString("sender_uuid");
        UUID sender = null;
        if (senderId != null && !senderId.isBlank()) {
            try {
                sender = UUID.fromString(senderId);
            }
            catch (IllegalArgumentException exception) {
                sender = null;
            }
        }
        long sentAt = resultSet.getLong("sent_at");
        String message = resultSet.getString("message");
        return new FoundryxDataStorage.MailMessage(sender, sentAt, message);
    }

    private List<StoredMail> toStoredMail(List<FoundryxDataStorage.MailMessage> messages) {
        if (messages.isEmpty()) {
            return List.of();
        }
        ArrayList<StoredMail> stored = new ArrayList<StoredMail>();
        for (FoundryxDataStorage.MailMessage message : messages) {
            stored.add(new StoredMail(message.sender() == null ? null : message.sender().toString(), message.timestamp(), message.message()));
        }
        return stored;
    }

    private record StoredMailData(List<StoredMail> mail) {
    }

    private record StoredMail(String sender, long timestamp, String message) {
    }

    private record LegacyUserData(List<StoredMail> mail) {
    }

    private record MailMessageKey(UUID sender, long timestamp, String message) {
        static MailMessageKey of(FoundryxDataStorage.MailMessage message) {
            return new MailMessageKey(message.sender(), message.timestamp(), message.message());
        }
    }
}

