/*
 * Decompiled with CFR 0.152.
 */
package com.foundryx.storage;

import com.foundryx.Constants;
import com.foundryx.data.HomeData;
import com.foundryx.data.WarpData;
import com.foundryx.storage.FoundryxDataStorage;
import com.foundryx.storage.UserData;
import com.foundryx.storage.database.DatabaseContext;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonParseException;
import com.google.gson.reflect.TypeToken;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.lang.reflect.Type;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.NbtAccounter;
import net.minecraft.nbt.NbtIo;
import net.minecraft.server.MinecraftServer;
import net.minecraft.world.level.storage.LevelResource;

public final class StorageMigration {
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private static final Type LOCATION_MAP_TYPE = new TypeToken<Map<String, FoundryxDataStorage.LocationSnapshot>>(){}.getType();

    private StorageMigration() {
    }

    public static MigrationReport exportToJson(MinecraftServer server, FoundryxDataStorage storage) throws StorageMigrationException {
        Objects.requireNonNull(server, "server");
        Objects.requireNonNull(storage, "storage");
        DatabaseContext database = storage.getDatabaseContext();
        if (database == null || !database.isAvailable()) {
            throw new StorageMigrationException("database_unavailable");
        }
        Path root = server.getWorldPath(LevelResource.ROOT).resolve("foundryx");
        try {
            int users = StorageMigration.exportUsers(database, root.resolve("userdata"));
            int homes = StorageMigration.exportHomes(database, root.resolve("homes"));
            int warps = StorageMigration.exportWarps(database, root.resolve("warps"));
            int mail = StorageMigration.exportMail(database, root.resolve("mail"));
            int backs = StorageMigration.exportBackData(database, root.resolve("back"));
            int deaths = StorageMigration.exportDeathData(database, root.resolve("deaths"));
            StorageMigration.exportSpawn(database, root.resolve("spawn"));
            StorageMigration.exportJail(database, root.resolve("jail"));
            int enderChests = StorageMigration.exportEnderChests(database, root.resolve("enderchests"));
            int kits = StorageMigration.exportKits(database, root.resolve("kits"));
            return new MigrationReport(users, homes, warps, mail, backs, deaths, kits, enderChests);
        }
        catch (IOException exception) {
            throw new StorageMigrationException("io_error", exception);
        }
        catch (SQLException exception) {
            throw new StorageMigrationException("sql_error", exception);
        }
    }

    public static MigrationReport importFromJson(MinecraftServer server, FoundryxDataStorage storage) throws StorageMigrationException {
        Objects.requireNonNull(server, "server");
        Objects.requireNonNull(storage, "storage");
        DatabaseContext database = storage.getDatabaseContext();
        if (database == null || !database.isAvailable()) {
            throw new StorageMigrationException("database_unavailable");
        }
        Path root = server.getWorldPath(LevelResource.ROOT).resolve("foundryx");
        try {
            int users = StorageMigration.importUsers(database, root.resolve("userdata"));
            int homes = StorageMigration.importHomes(database, root.resolve("homes"));
            int warps = StorageMigration.importWarps(database, root.resolve("warps"));
            int mail = StorageMigration.importMail(database, root.resolve("mail"));
            int backs = StorageMigration.importBackData(database, root.resolve("back"));
            int deaths = StorageMigration.importDeathData(database, root.resolve("deaths"));
            StorageMigration.importSpawn(database, root.resolve("spawn"));
            StorageMigration.importJail(database, root.resolve("jail"));
            int enderChests = StorageMigration.importEnderChests(database, root.resolve("enderchests"));
            int kits = StorageMigration.importKits(database, root.resolve("kits"));
            return new MigrationReport(users, homes, warps, mail, backs, deaths, kits, enderChests);
        }
        catch (IOException exception) {
            throw new StorageMigrationException("io_error", exception);
        }
        catch (JsonParseException | SQLException exception) {
            throw new StorageMigrationException("sql_error", exception);
        }
    }

    private static int exportUsers(DatabaseContext database, Path directory) throws SQLException, IOException {
        UserData data;
        UUID player;
        ResultSet resultSet;
        PreparedStatement statement;
        HashMap<UUID, UserData> users = new HashMap<UUID, UserData>();
        try (Connection connection = database.getConnection();){
            statement = connection.prepareStatement("SELECT player_uuid, nickname, last_known_name, last_login, last_seen, play_time, god_mode, frozen, mute_expiry, mute_reason, last_ip, balance, last_health, food_level, saturation, operator, can_fly, game_mode, jail_origin_dimension, jail_origin_x, jail_origin_y, jail_origin_z, jail_origin_yaw, jail_origin_pitch, jail_location_dimension, jail_location_x, jail_location_y, jail_location_z, jail_location_yaw, jail_location_pitch, jail_release FROM foundryx_user_data");
            try {
                resultSet = statement.executeQuery();
                try {
                    while (resultSet.next()) {
                        Boolean canFly;
                        player = StorageMigration.parseUuid(resultSet.getString("player_uuid"));
                        if (player == null) continue;
                        data = new UserData();
                        data.nickname = StorageMigration.normalizeNickname(resultSet.getString("nickname"));
                        data.lastKnownName = resultSet.getString("last_known_name");
                        data.lastLogin = StorageMigration.getNullableLong(resultSet, "last_login");
                        data.lastSeen = StorageMigration.getNullableLong(resultSet, "last_seen");
                        data.playTime = StorageMigration.getNullableLong(resultSet, "play_time");
                        data.godMode = resultSet.getBoolean("god_mode");
                        data.frozen = resultSet.getBoolean("frozen");
                        data.muteExpiry = StorageMigration.getNullableLong(resultSet, "mute_expiry");
                        data.muteReason = resultSet.getString("mute_reason");
                        data.lastIp = StorageMigration.sanitizeIp(resultSet.getString("last_ip"));
                        Double balance = StorageMigration.getNullableDouble(resultSet, "balance");
                        if (balance != null) {
                            data.balance = Math.max(0.0, balance);
                        }
                        data.lastHealth = StorageMigration.getNullableDouble(resultSet, "last_health");
                        data.foodLevel = StorageMigration.getNullableInteger(resultSet, "food_level");
                        data.saturation = StorageMigration.getNullableFloat(resultSet, "saturation");
                        Boolean operator = StorageMigration.getNullableBoolean(resultSet, "operator");
                        if (operator != null) {
                            data.operator = operator;
                        }
                        if ((canFly = StorageMigration.getNullableBoolean(resultSet, "can_fly")) != null) {
                            data.canFly = canFly;
                        }
                        data.gameMode = StorageMigration.getNullableInteger(resultSet, "game_mode");
                        FoundryxDataStorage.LocationSnapshot origin = StorageMigration.buildSnapshotFromRow(resultSet, "jail_origin");
                        FoundryxDataStorage.LocationSnapshot jail = StorageMigration.buildSnapshotFromRow(resultSet, "jail_location");
                        Long release = StorageMigration.getNullableLong(resultSet, "jail_release");
                        if (origin != null || jail != null || release != null) {
                            data.jailRecord = new FoundryxDataStorage.JailRecord(origin, jail, release);
                        }
                        users.put(player, data);
                    }
                }
                finally {
                    if (resultSet != null) {
                        resultSet.close();
                    }
                }
            }
            finally {
                if (statement != null) {
                    statement.close();
                }
            }
        }
        connection = database.getConnection();
        try {
            statement = connection.prepareStatement("SELECT player_uuid, kit_name, last_used_at FROM foundryx_kit_usage");
            try {
                resultSet = statement.executeQuery();
                try {
                    while (resultSet.next()) {
                        player = StorageMigration.parseUuid(resultSet.getString("player_uuid"));
                        if (player == null) continue;
                        data = users.computeIfAbsent(player, ignored -> new UserData());
                        String kit = StorageMigration.normalizeKey(resultSet.getString("kit_name"));
                        Long usedAt = StorageMigration.getNullableLong(resultSet, "last_used_at");
                        if (kit == null || usedAt == null) continue;
                        data.kitUsage.put(kit, usedAt);
                    }
                }
                finally {
                    if (resultSet != null) {
                        resultSet.close();
                    }
                }
            }
            finally {
                if (statement != null) {
                    statement.close();
                }
            }
        }
        finally {
            if (connection != null) {
                connection.close();
            }
        }
        StorageMigration.clearDirectory(directory, "*.json");
        int written = 0;
        for (Map.Entry entry : users.entrySet()) {
            player = (UUID)entry.getKey();
            data = (UserData)entry.getValue();
            Path file = directory.resolve(player.toString() + ".json");
            if (data == null || data.isEmpty()) {
                Files.deleteIfExists(file);
                continue;
            }
            Files.createDirectories(directory, new FileAttribute[0]);
            PersistedUserData persisted = new PersistedUserData(data.jailRecord, data.muteExpiry, data.kitUsage.isEmpty() ? Map.of() : Map.copyOf(data.kitUsage), data.godMode, data.frozen, data.nickname, data.lastKnownName, data.lastHealth, data.foodLevel, data.saturation, data.operator, data.canFly, data.gameMode, data.lastIp, data.lastLogin, data.lastSeen, data.playTime, data.muteReason, UserData.isZero(data.balance) ? null : Double.valueOf(data.balance));
            try (BufferedWriter writer = Files.newBufferedWriter(file, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);){
                GSON.toJson((Object)persisted, (Appendable)writer);
            }
            ++written;
        }
        return written;
    }

    private static int exportHomes(DatabaseContext database, Path directory) throws SQLException, IOException {
        HashMap<UUID, HomeData> homes = new HashMap<UUID, HomeData>();
        try (Connection connection = database.getConnection();
             PreparedStatement statement = connection.prepareStatement("SELECT player_uuid, home_name, dimension, x, y, z, yaw, pitch FROM foundryx_home_data");
             ResultSet resultSet = statement.executeQuery();){
            while (resultSet.next()) {
                UUID owner = StorageMigration.parseUuid(resultSet.getString("player_uuid"));
                if (owner == null) continue;
                String name = resultSet.getString("home_name");
                FoundryxDataStorage.LocationSnapshot snapshot = StorageMigration.mapSnapshot(resultSet);
                if (name == null || snapshot == null) continue;
                homes.computeIfAbsent(owner, ignored -> new HomeData()).set(name, snapshot);
            }
        }
        StorageMigration.clearDirectory(directory, "*.json");
        int written = 0;
        for (Map.Entry entry : homes.entrySet()) {
            HomeData data = (HomeData)entry.getValue();
            if (data == null || data.isEmpty()) continue;
            Path file = directory.resolve(((UUID)entry.getKey()).toString() + ".json");
            Files.createDirectories(directory, new FileAttribute[0]);
            try (BufferedWriter writer = Files.newBufferedWriter(file, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);){
                GSON.toJson(data.asMap(), LOCATION_MAP_TYPE, (Appendable)writer);
            }
            ++written;
        }
        return written;
    }

    private static int exportWarps(DatabaseContext database, Path directory) throws SQLException, IOException {
        HashMap<UUID, WarpData> warps = new HashMap<UUID, WarpData>();
        try (Connection connection = database.getConnection();
             PreparedStatement statement = connection.prepareStatement("SELECT owner_uuid, warp_name, dimension, x, y, z, yaw, pitch FROM foundryx_warps");
             ResultSet resultSet = statement.executeQuery();){
            while (resultSet.next()) {
                UUID owner = StorageMigration.parseUuid(resultSet.getString("owner_uuid"));
                if (owner == null) continue;
                String name = resultSet.getString("warp_name");
                FoundryxDataStorage.LocationSnapshot snapshot = StorageMigration.mapSnapshot(resultSet);
                if (name == null || snapshot == null) continue;
                warps.computeIfAbsent(owner, ignored -> new WarpData()).set(name, snapshot);
            }
        }
        StorageMigration.clearDirectory(directory, "*.json");
        int written = 0;
        for (Map.Entry entry : warps.entrySet()) {
            WarpData data = (WarpData)entry.getValue();
            if (data == null || data.isEmpty()) continue;
            Path file = directory.resolve(((UUID)entry.getKey()).toString() + ".json");
            Files.createDirectories(directory, new FileAttribute[0]);
            try (BufferedWriter writer = Files.newBufferedWriter(file, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);){
                GSON.toJson(data.asMap(), LOCATION_MAP_TYPE, (Appendable)writer);
            }
            ++written;
        }
        return written;
    }

    private static void exportSpawn(DatabaseContext database, Path directory) throws SQLException, IOException {
        FoundryxDataStorage.LocationSnapshot snapshot = null;
        try (Connection connection = database.getConnection();
             PreparedStatement statement = connection.prepareStatement("SELECT dimension, x, y, z, yaw, pitch FROM foundryx_spawn_points WHERE id=1");
             ResultSet resultSet = statement.executeQuery();){
            if (resultSet.next()) {
                snapshot = StorageMigration.mapSnapshot(resultSet);
            }
        }
        Path file = directory.resolve("spawn.json");
        if (snapshot == null) {
            Files.deleteIfExists(file);
            return;
        }
        Files.createDirectories(directory, new FileAttribute[0]);
        SpawnDataContainer data = new SpawnDataContainer(snapshot);
        try (BufferedWriter writer = Files.newBufferedWriter(file, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);){
            GSON.toJson((Object)data, (Appendable)writer);
        }
    }

    private static void exportJail(DatabaseContext database, Path directory) throws SQLException, IOException {
        FoundryxDataStorage.JailRegion region = null;
        try (Connection connection = database.getConnection();
             PreparedStatement statement = connection.prepareStatement("SELECT dimension, min_x, min_y, min_z, max_x, max_y, max_z, yaw, pitch FROM foundryx_jail_data WHERE id=1");
             ResultSet resultSet = statement.executeQuery();){
            if (resultSet.next()) {
                region = StorageMigration.mapRegion(resultSet);
            }
        }
        Path file = directory.resolve("region.json");
        if (region == null) {
            Files.deleteIfExists(file);
            return;
        }
        Files.createDirectories(directory, new FileAttribute[0]);
        try (BufferedWriter writer = Files.newBufferedWriter(file, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);){
            GSON.toJson((Object)region, (Appendable)writer);
        }
    }

    private static int exportBackData(DatabaseContext database, Path directory) throws SQLException, IOException {
        HashMap<UUID, PersistedBackData> backs = new HashMap<UUID, PersistedBackData>();
        try (Connection connection = database.getConnection();
             PreparedStatement statement = connection.prepareStatement("SELECT player_uuid, dimension, x, y, z, yaw, pitch, recorded_at FROM foundryx_back_data");
             ResultSet resultSet = statement.executeQuery();){
            while (resultSet.next()) {
                UUID player = StorageMigration.parseUuid(resultSet.getString("player_uuid"));
                FoundryxDataStorage.LocationSnapshot snapshot = StorageMigration.mapSnapshot(resultSet);
                Long recordedAt = StorageMigration.getNullableLong(resultSet, "recorded_at");
                if (player == null || snapshot == null) continue;
                backs.put(player, new PersistedBackData(snapshot, recordedAt));
            }
        }
        StorageMigration.clearDirectory(directory, "*.json");
        int written = 0;
        for (Map.Entry entry : backs.entrySet()) {
            PersistedBackData data = (PersistedBackData)entry.getValue();
            if (data == null || data.location() == null) continue;
            Path file = directory.resolve(((UUID)entry.getKey()).toString() + ".json");
            Files.createDirectories(directory, new FileAttribute[0]);
            try (BufferedWriter writer = Files.newBufferedWriter(file, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);){
                GSON.toJson((Object)data, (Appendable)writer);
            }
            ++written;
        }
        return written;
    }

    private static int exportDeathData(DatabaseContext database, Path directory) throws SQLException, IOException {
        HashMap<UUID, PersistedDeathData> deaths = new HashMap<UUID, PersistedDeathData>();
        try (Connection connection = database.getConnection();
             PreparedStatement statement = connection.prepareStatement("SELECT player_uuid, dimension, x, y, z, yaw, pitch, cause, death_time FROM foundryx_death_data");
             ResultSet resultSet = statement.executeQuery();){
            while (resultSet.next()) {
                UUID player = StorageMigration.parseUuid(resultSet.getString("player_uuid"));
                FoundryxDataStorage.LocationSnapshot snapshot = StorageMigration.mapSnapshot(resultSet);
                Long timestamp = StorageMigration.getNullableLong(resultSet, "death_time");
                String cause = resultSet.getString("cause");
                if (player == null || snapshot == null) continue;
                deaths.put(player, new PersistedDeathData(snapshot, timestamp, cause));
            }
        }
        StorageMigration.clearDirectory(directory, "*.json");
        int written = 0;
        for (Map.Entry entry : deaths.entrySet()) {
            PersistedDeathData data = (PersistedDeathData)entry.getValue();
            if (data == null || data.location() == null) continue;
            Path file = directory.resolve(((UUID)entry.getKey()).toString() + ".json");
            Files.createDirectories(directory, new FileAttribute[0]);
            try (BufferedWriter writer = Files.newBufferedWriter(file, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);){
                GSON.toJson((Object)data, (Appendable)writer);
            }
            ++written;
        }
        return written;
    }

    private static int exportMail(DatabaseContext database, Path directory) throws SQLException, IOException {
        HashMap<UUID, List> grouped = new HashMap<UUID, List>();
        try (Connection connection = database.getConnection();
             PreparedStatement statement = connection.prepareStatement("SELECT player_uuid, sender_uuid, sent_at, message FROM foundryx_mail ORDER BY player_uuid, sent_at");
             ResultSet resultSet = statement.executeQuery();){
            while (resultSet.next()) {
                UUID player = StorageMigration.parseUuid(resultSet.getString("player_uuid"));
                if (player == null) continue;
                UUID sender = StorageMigration.parseUuid(resultSet.getString("sender_uuid"));
                long sentAt = resultSet.getLong("sent_at");
                String message = resultSet.getString("message");
                if (message == null) continue;
                grouped.computeIfAbsent(player, ignored -> new ArrayList()).add(new StoredMail(sender == null ? null : sender.toString(), sentAt, message));
            }
        }
        StorageMigration.clearDirectory(directory, "*.json");
        int written = 0;
        for (Map.Entry entry : grouped.entrySet()) {
            List mail = (List)entry.getValue();
            if (mail.isEmpty()) continue;
            Path file = directory.resolve(((UUID)entry.getKey()).toString() + ".json");
            Files.createDirectories(directory, new FileAttribute[0]);
            try (BufferedWriter writer = Files.newBufferedWriter(file, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);){
                GSON.toJson((Object)new StoredMailData(mail), (Appendable)writer);
            }
            ++written;
        }
        return written;
    }

    private static int exportEnderChests(DatabaseContext database, Path directory) throws SQLException, IOException {
        HashMap<UUID, EnderChestPayload> payloads = new HashMap<UUID, EnderChestPayload>();
        try (Connection connection = database.getConnection();
             PreparedStatement statement = connection.prepareStatement("SELECT player_uuid, last_known_name, nbt_payload FROM foundryx_ender_chests");
             ResultSet resultSet = statement.executeQuery();){
            while (resultSet.next()) {
                UUID player = StorageMigration.parseUuid(resultSet.getString("player_uuid"));
                if (player == null) continue;
                String encoded = resultSet.getString("nbt_payload");
                String lastKnown = StorageMigration.sanitize(resultSet.getString("last_known_name"));
                payloads.put(player, new EnderChestPayload(encoded, lastKnown));
            }
        }
        StorageMigration.clearDirectory(directory, "*.nbt");
        int written = 0;
        for (Map.Entry entry : payloads.entrySet()) {
            EnderChestPayload payload = (EnderChestPayload)entry.getValue();
            if (payload == null || payload.encoded() == null || payload.encoded().isBlank()) continue;
            byte[] decoded = Base64.getDecoder().decode(payload.encoded());
            CompoundTag tag = StorageMigration.readCompressedTag(decoded);
            if (tag == null) {
                tag = new CompoundTag();
            }
            if (payload.lastKnown() != null && !payload.lastKnown().isBlank()) {
                tag.putString("LastKnownName", payload.lastKnown());
            }
            byte[] bytes = StorageMigration.writeCompressedTag(tag);
            Path file = directory.resolve(((UUID)entry.getKey()).toString() + ".nbt");
            Files.createDirectories(directory, new FileAttribute[0]);
            Files.write(file, bytes, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);
            ++written;
        }
        return written;
    }

    private static int exportKits(DatabaseContext database, Path directory) throws SQLException, IOException {
        HashMap<String, KitPayload> kits = new HashMap<String, KitPayload>();
        try (Connection connection = database.getConnection();
             PreparedStatement statement = connection.prepareStatement("SELECT kit_name, cooldown_seconds, nbt_payload FROM foundryx_kits");
             ResultSet resultSet = statement.executeQuery();){
            while (resultSet.next()) {
                String name = StorageMigration.normalizeKey(resultSet.getString("kit_name"));
                if (name == null) continue;
                Long cooldown = StorageMigration.getNullableLong(resultSet, "cooldown_seconds");
                String encoded = resultSet.getString("nbt_payload");
                kits.put(name, new KitPayload(encoded, cooldown == null ? 0L : cooldown));
            }
        }
        StorageMigration.clearDirectory(directory, "*.nbt");
        int written = 0;
        for (Map.Entry entry : kits.entrySet()) {
            KitPayload payload = (KitPayload)entry.getValue();
            if (payload == null || payload.encoded() == null || payload.encoded().isBlank()) continue;
            byte[] decoded = Base64.getDecoder().decode(payload.encoded());
            CompoundTag tag = StorageMigration.readCompressedTag(decoded);
            if (tag == null) {
                tag = new CompoundTag();
            }
            tag.putString("Name", (String)entry.getKey());
            tag.putLong("Cooldown", payload.cooldown());
            byte[] bytes = StorageMigration.writeCompressedTag(tag);
            Path file = directory.resolve((String)entry.getKey() + ".nbt");
            Files.createDirectories(directory, new FileAttribute[0]);
            Files.write(file, bytes, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);
            ++written;
        }
        return written;
    }

    private static int importUsers(DatabaseContext database, Path directory) throws SQLException, IOException {
        StorageMigration.clearTable(database, "foundryx_user_data");
        StorageMigration.clearTable(database, "foundryx_kit_usage");
        if (!Files.isDirectory(directory, new LinkOption[0])) {
            return 0;
        }
        int imported = 0;
        try (Connection connection = database.getConnection();){
            connection.setAutoCommit(false);
            try (PreparedStatement insertUser = connection.prepareStatement("INSERT INTO foundryx_user_data (player_uuid, nickname, last_known_name, last_login, last_seen, play_time, god_mode, frozen, mute_expiry, mute_reason, last_ip, balance, last_health, food_level, saturation, operator, can_fly, game_mode, jail_origin_dimension, jail_origin_x, jail_origin_y, jail_origin_z, jail_origin_yaw, jail_origin_pitch, jail_location_dimension, jail_location_x, jail_location_y, jail_location_z, jail_location_yaw, jail_location_pitch, jail_release) VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)");
                 PreparedStatement insertKit = connection.prepareStatement("INSERT INTO foundryx_kit_usage (player_uuid, kit_name, last_used_at) VALUES (?,?,?)");){
                try (DirectoryStream<Path> files = Files.newDirectoryStream(directory, "*.json");){
                    for (Path file : files) {
                        PersistedUserData persisted;
                        UUID player = StorageMigration.parseUuid(StorageMigration.filenameWithoutExtension(file.getFileName().toString()));
                        if (player == null) continue;
                        try (BufferedReader reader = Files.newBufferedReader(file);){
                            persisted = (PersistedUserData)GSON.fromJson((Reader)reader, PersistedUserData.class);
                        }
                        if (persisted == null) continue;
                        StorageMigration.bindUser(insertUser, player, persisted);
                        insertUser.addBatch();
                        ++imported;
                        if (persisted.kitUsage() == null || persisted.kitUsage().isEmpty()) continue;
                        for (Map.Entry<String, Long> entry : persisted.kitUsage().entrySet()) {
                            if (entry.getKey() == null || entry.getValue() == null) continue;
                            insertKit.setString(1, player.toString());
                            insertKit.setString(2, StorageMigration.normalizeKey(entry.getKey()));
                            insertKit.setLong(3, entry.getValue());
                            insertKit.addBatch();
                        }
                    }
                }
                insertUser.executeBatch();
                insertKit.executeBatch();
                connection.commit();
            }
            catch (JsonParseException | IOException | SQLException exception) {
                connection.rollback();
                throw exception;
            }
            finally {
                connection.setAutoCommit(true);
            }
        }
        return imported;
    }

    private static int importHomes(DatabaseContext database, Path directory) throws SQLException, IOException {
        StorageMigration.clearTable(database, "foundryx_home_data");
        if (!Files.isDirectory(directory, new LinkOption[0])) {
            return 0;
        }
        int imported = 0;
        try (Connection connection = database.getConnection();
             PreparedStatement insert = connection.prepareStatement("INSERT INTO foundryx_home_data (player_uuid, home_name, dimension, x, y, z, yaw, pitch) VALUES (?,?,?,?,?,?,?,?)");){
            connection.setAutoCommit(false);
            try (DirectoryStream<Path> files = Files.newDirectoryStream(directory, "*.json");){
                for (Path file : files) {
                    Map map;
                    UUID owner = StorageMigration.parseUuid(StorageMigration.filenameWithoutExtension(file.getFileName().toString()));
                    if (owner == null) continue;
                    try (BufferedReader reader = Files.newBufferedReader(file);){
                        map = (Map)GSON.fromJson((Reader)reader, LOCATION_MAP_TYPE);
                    }
                    if (map == null || map.isEmpty()) continue;
                    for (Map.Entry entry : map.entrySet()) {
                        if (entry.getKey() == null || entry.getValue() == null) continue;
                        insert.setString(1, owner.toString());
                        insert.setString(2, StorageMigration.normalizeKey((String)entry.getKey()));
                        StorageMigration.bindSnapshot(insert, 3, (FoundryxDataStorage.LocationSnapshot)entry.getValue());
                        insert.addBatch();
                    }
                    ++imported;
                }
            }
            insert.executeBatch();
            connection.commit();
            connection.setAutoCommit(true);
        }
        return imported;
    }

    private static int importWarps(DatabaseContext database, Path directory) throws SQLException, IOException {
        StorageMigration.clearTable(database, "foundryx_warps");
        if (!Files.isDirectory(directory, new LinkOption[0])) {
            return 0;
        }
        int imported = 0;
        try (Connection connection = database.getConnection();
             PreparedStatement insert = connection.prepareStatement("INSERT INTO foundryx_warps (warp_name, owner_uuid, dimension, x, y, z, yaw, pitch) VALUES (?,?,?,?,?,?,?,?)");){
            connection.setAutoCommit(false);
            try (DirectoryStream<Path> files = Files.newDirectoryStream(directory, "*.json");){
                for (Path file : files) {
                    Map map;
                    UUID owner = StorageMigration.parseUuid(StorageMigration.filenameWithoutExtension(file.getFileName().toString()));
                    if (owner == null) continue;
                    try (BufferedReader reader = Files.newBufferedReader(file);){
                        map = (Map)GSON.fromJson((Reader)reader, LOCATION_MAP_TYPE);
                    }
                    if (map == null || map.isEmpty()) continue;
                    for (Map.Entry entry : map.entrySet()) {
                        if (entry.getKey() == null || entry.getValue() == null) continue;
                        insert.setString(1, StorageMigration.normalizeKey((String)entry.getKey()));
                        insert.setString(2, owner.toString());
                        StorageMigration.bindSnapshot(insert, 3, (FoundryxDataStorage.LocationSnapshot)entry.getValue());
                        insert.addBatch();
                    }
                    ++imported;
                }
            }
            insert.executeBatch();
            connection.commit();
            connection.setAutoCommit(true);
        }
        return imported;
    }

    private static void importSpawn(DatabaseContext database, Path directory) throws SQLException, IOException {
        SpawnDataContainer data;
        StorageMigration.clearTable(database, "foundryx_spawn_points");
        Path file = directory.resolve("spawn.json");
        if (!Files.exists(file, new LinkOption[0])) {
            return;
        }
        try (BufferedReader reader = Files.newBufferedReader(file);){
            data = (SpawnDataContainer)GSON.fromJson((Reader)reader, SpawnDataContainer.class);
        }
        if (data == null || data.location == null) {
            return;
        }
        try (Connection connection = database.getConnection();
             PreparedStatement insert = connection.prepareStatement("INSERT INTO foundryx_spawn_points (id, dimension, x, y, z, yaw, pitch) VALUES (?,?,?,?,?,?,?)");){
            insert.setInt(1, 1);
            StorageMigration.bindSnapshot(insert, 2, data.location);
            insert.executeUpdate();
        }
    }

    private static void importJail(DatabaseContext database, Path directory) throws SQLException, IOException {
        FoundryxDataStorage.JailRegion region;
        StorageMigration.clearTable(database, "foundryx_jail_data");
        Path file = directory.resolve("region.json");
        if (!Files.exists(file, new LinkOption[0])) {
            return;
        }
        try (BufferedReader reader = Files.newBufferedReader(file);){
            region = (FoundryxDataStorage.JailRegion)GSON.fromJson((Reader)reader, FoundryxDataStorage.JailRegion.class);
        }
        if (region == null) {
            return;
        }
        try (Connection connection = database.getConnection();
             PreparedStatement insert = connection.prepareStatement("INSERT INTO foundryx_jail_data (id, dimension, min_x, min_y, min_z, max_x, max_y, max_z, yaw, pitch) VALUES (?,?,?,?,?,?,?,?,?,?)");){
            insert.setInt(1, 1);
            insert.setString(2, region.dimension());
            insert.setInt(3, region.minimum().x());
            insert.setInt(4, region.minimum().y());
            insert.setInt(5, region.minimum().z());
            insert.setInt(6, region.maximum().x());
            insert.setInt(7, region.maximum().y());
            insert.setInt(8, region.maximum().z());
            insert.setFloat(9, region.yaw());
            insert.setFloat(10, region.pitch());
            insert.executeUpdate();
        }
    }

    private static int importBackData(DatabaseContext database, Path directory) throws SQLException, IOException {
        StorageMigration.clearTable(database, "foundryx_back_data");
        if (!Files.isDirectory(directory, new LinkOption[0])) {
            return 0;
        }
        int imported = 0;
        try (Connection connection = database.getConnection();
             PreparedStatement insert = connection.prepareStatement("INSERT INTO foundryx_back_data (player_uuid, dimension, x, y, z, yaw, pitch, recorded_at) VALUES (?,?,?,?,?,?,?,?)");){
            connection.setAutoCommit(false);
            try (DirectoryStream<Path> files = Files.newDirectoryStream(directory, "*.json");){
                for (Path file : files) {
                    PersistedBackData data;
                    UUID player = StorageMigration.parseUuid(StorageMigration.filenameWithoutExtension(file.getFileName().toString()));
                    if (player == null) continue;
                    try (BufferedReader reader = Files.newBufferedReader(file);){
                        data = (PersistedBackData)GSON.fromJson((Reader)reader, PersistedBackData.class);
                    }
                    if (data == null || data.location() == null) continue;
                    insert.setString(1, player.toString());
                    StorageMigration.bindSnapshot(insert, 2, data.location());
                    if (data.recordedAt() == null) {
                        insert.setNull(8, -5);
                    } else {
                        insert.setLong(8, data.recordedAt());
                    }
                    insert.addBatch();
                    ++imported;
                }
            }
            insert.executeBatch();
            connection.commit();
            connection.setAutoCommit(true);
        }
        return imported;
    }

    private static int importDeathData(DatabaseContext database, Path directory) throws SQLException, IOException {
        StorageMigration.clearTable(database, "foundryx_death_data");
        if (!Files.isDirectory(directory, new LinkOption[0])) {
            return 0;
        }
        int imported = 0;
        try (Connection connection = database.getConnection();
             PreparedStatement insert = connection.prepareStatement("INSERT INTO foundryx_death_data (player_uuid, dimension, x, y, z, yaw, pitch, cause, death_time) VALUES (?,?,?,?,?,?,?,?,?)");){
            connection.setAutoCommit(false);
            try (DirectoryStream<Path> files = Files.newDirectoryStream(directory, "*.json");){
                for (Path file : files) {
                    PersistedDeathData data;
                    UUID player = StorageMigration.parseUuid(StorageMigration.filenameWithoutExtension(file.getFileName().toString()));
                    if (player == null) continue;
                    try (BufferedReader reader = Files.newBufferedReader(file);){
                        data = (PersistedDeathData)GSON.fromJson((Reader)reader, PersistedDeathData.class);
                    }
                    if (data == null || data.location() == null) continue;
                    insert.setString(1, player.toString());
                    StorageMigration.bindSnapshot(insert, 2, data.location());
                    if (data.cause() == null || data.cause().isBlank()) {
                        insert.setNull(8, 12);
                    } else {
                        insert.setString(8, data.cause());
                    }
                    if (data.deathTime() == null) {
                        insert.setNull(9, -5);
                    } else {
                        insert.setLong(9, data.deathTime());
                    }
                    insert.addBatch();
                    ++imported;
                }
            }
            insert.executeBatch();
            connection.commit();
            connection.setAutoCommit(true);
        }
        return imported;
    }

    private static int importMail(DatabaseContext database, Path directory) throws SQLException, IOException {
        StorageMigration.clearTable(database, "foundryx_mail");
        if (!Files.isDirectory(directory, new LinkOption[0])) {
            return 0;
        }
        int imported = 0;
        try (Connection connection = database.getConnection();
             PreparedStatement insert = connection.prepareStatement("INSERT INTO foundryx_mail (mail_id, player_uuid, sender_uuid, sent_at, message) VALUES (?,?,?,?,?)");){
            connection.setAutoCommit(false);
            try (DirectoryStream<Path> files = Files.newDirectoryStream(directory, "*.json");){
                for (Path file : files) {
                    StoredMailData data;
                    UUID player = StorageMigration.parseUuid(StorageMigration.filenameWithoutExtension(file.getFileName().toString()));
                    if (player == null) continue;
                    try (BufferedReader reader = Files.newBufferedReader(file);){
                        data = (StoredMailData)GSON.fromJson((Reader)reader, StoredMailData.class);
                    }
                    if (data == null || data.mail() == null || data.mail().isEmpty()) continue;
                    for (StoredMail mail : data.mail()) {
                        if (mail == null || mail.message() == null) continue;
                        insert.setString(1, UUID.randomUUID().toString());
                        insert.setString(2, player.toString());
                        if (mail.sender() == null || mail.sender().isBlank()) {
                            insert.setNull(3, 12);
                        } else {
                            insert.setString(3, mail.sender());
                        }
                        insert.setLong(4, mail.timestamp());
                        insert.setString(5, mail.message());
                        insert.addBatch();
                    }
                    ++imported;
                }
            }
            insert.executeBatch();
            connection.commit();
            connection.setAutoCommit(true);
        }
        return imported;
    }

    private static int importEnderChests(DatabaseContext database, Path directory) throws SQLException, IOException {
        StorageMigration.clearTable(database, "foundryx_ender_chests");
        if (!Files.isDirectory(directory, new LinkOption[0])) {
            return 0;
        }
        int imported = 0;
        try (Connection connection = database.getConnection();
             PreparedStatement insert = connection.prepareStatement("INSERT INTO foundryx_ender_chests (player_uuid, last_known_name, nbt_payload) VALUES (?,?,?)");){
            connection.setAutoCommit(false);
            try (DirectoryStream<Path> files = Files.newDirectoryStream(directory, "*.nbt");){
                for (Path file : files) {
                    CompoundTag tag;
                    byte[] bytes;
                    UUID player = StorageMigration.parseUuid(StorageMigration.filenameWithoutExtension(file.getFileName().toString()));
                    if (player == null || (bytes = Files.readAllBytes(file)).length == 0 || (tag = StorageMigration.readCompressedTag(bytes)) == null) continue;
                    String lastKnown = tag.getString("LastKnownName").orElse(null);
                    byte[] payload = StorageMigration.writeCompressedTag(tag);
                    insert.setString(1, player.toString());
                    if (lastKnown == null || lastKnown.isBlank()) {
                        insert.setNull(2, 12);
                    } else {
                        insert.setString(2, lastKnown);
                    }
                    insert.setString(3, Base64.getEncoder().encodeToString(payload));
                    insert.addBatch();
                    ++imported;
                }
            }
            insert.executeBatch();
            connection.commit();
            connection.setAutoCommit(true);
        }
        return imported;
    }

    private static int importKits(DatabaseContext database, Path directory) throws SQLException, IOException {
        StorageMigration.clearTable(database, "foundryx_kits");
        if (!Files.isDirectory(directory, new LinkOption[0])) {
            return 0;
        }
        int imported = 0;
        try (Connection connection = database.getConnection();
             PreparedStatement insert = connection.prepareStatement("INSERT INTO foundryx_kits (kit_name, cooldown_seconds, nbt_payload) VALUES (?,?,?)");){
            connection.setAutoCommit(false);
            try (DirectoryStream<Path> files = Files.newDirectoryStream(directory, "*.nbt");){
                for (Path file : files) {
                    CompoundTag tag;
                    byte[] bytes;
                    String key = StorageMigration.normalizeKey(StorageMigration.filenameWithoutExtension(file.getFileName().toString()));
                    if (key == null || (bytes = Files.readAllBytes(file)).length == 0 || (tag = StorageMigration.readCompressedTag(bytes)) == null) continue;
                    long cooldown = tag.getLong("Cooldown").orElse(0L);
                    byte[] payload = StorageMigration.writeCompressedTag(tag);
                    insert.setString(1, key);
                    if (cooldown <= 0L) {
                        insert.setNull(2, -5);
                    } else {
                        insert.setLong(2, cooldown);
                    }
                    insert.setString(3, Base64.getEncoder().encodeToString(payload));
                    insert.addBatch();
                    ++imported;
                }
            }
            insert.executeBatch();
            connection.commit();
            connection.setAutoCommit(true);
        }
        return imported;
    }

    private static void clearTable(DatabaseContext database, String table) throws SQLException {
        try (Connection connection = database.getConnection();
             Statement statement = connection.createStatement();){
            statement.executeUpdate("DELETE FROM " + table);
        }
    }

    private static void bindUser(PreparedStatement statement, UUID player, PersistedUserData data) throws SQLException {
        statement.setString(1, player.toString());
        StorageMigration.setNullableString(statement, 2, StorageMigration.normalizeNickname(data.nickname()));
        StorageMigration.setNullableString(statement, 3, data.lastKnownName());
        StorageMigration.setNullableLong(statement, 4, data.lastLogin());
        StorageMigration.setNullableLong(statement, 5, data.lastSeen());
        StorageMigration.setNullableLong(statement, 6, data.playTime());
        statement.setBoolean(7, data.godMode() != null && data.godMode() != false);
        statement.setBoolean(8, data.frozen() != null && data.frozen() != false);
        StorageMigration.setNullableLong(statement, 9, data.muteExpiry());
        StorageMigration.setNullableString(statement, 10, data.muteReason());
        StorageMigration.setNullableString(statement, 11, StorageMigration.sanitizeIp(data.lastIp()));
        StorageMigration.setNullableDouble(statement, 12, data.balance());
        StorageMigration.setNullableDouble(statement, 13, data.lastHealth());
        StorageMigration.setNullableInteger(statement, 14, data.foodLevel());
        StorageMigration.setNullableFloat(statement, 15, data.saturation());
        StorageMigration.setNullableBoolean(statement, 16, data.operator());
        StorageMigration.setNullableBoolean(statement, 17, data.canFly());
        StorageMigration.setNullableInteger(statement, 18, data.gameMode());
        FoundryxDataStorage.LocationSnapshot origin = data.jail() == null ? null : data.jail().originalLocation();
        FoundryxDataStorage.LocationSnapshot jail = data.jail() == null ? null : data.jail().jailLocation();
        StorageMigration.bindSnapshot(statement, 19, origin);
        StorageMigration.bindSnapshot(statement, 25, jail);
        StorageMigration.setNullableLong(statement, 31, data.jail() == null ? null : data.jail().releaseAtMillis());
    }

    private static void bindSnapshot(PreparedStatement statement, int offset, FoundryxDataStorage.LocationSnapshot snapshot) throws SQLException {
        if (snapshot == null) {
            statement.setNull(offset, 12);
            statement.setNull(offset + 1, 8);
            statement.setNull(offset + 2, 8);
            statement.setNull(offset + 3, 8);
            statement.setNull(offset + 4, 7);
            statement.setNull(offset + 5, 7);
            return;
        }
        statement.setString(offset, snapshot.dimension());
        statement.setDouble(offset + 1, snapshot.x());
        statement.setDouble(offset + 2, snapshot.y());
        statement.setDouble(offset + 3, snapshot.z());
        statement.setFloat(offset + 4, snapshot.yaw());
        statement.setFloat(offset + 5, snapshot.pitch());
    }

    private static FoundryxDataStorage.LocationSnapshot mapSnapshot(ResultSet resultSet) throws SQLException {
        String dimension = resultSet.getString("dimension");
        if (dimension == null) {
            return null;
        }
        double x = resultSet.getDouble("x");
        double y = resultSet.getDouble("y");
        double z = resultSet.getDouble("z");
        float yaw = resultSet.getFloat("yaw");
        float pitch = resultSet.getFloat("pitch");
        return new FoundryxDataStorage.LocationSnapshot(dimension, x, y, z, yaw, pitch);
    }

    private static FoundryxDataStorage.JailRegion mapRegion(ResultSet resultSet) throws SQLException {
        String dimension = resultSet.getString("dimension");
        if (dimension == null) {
            return null;
        }
        FoundryxDataStorage.BlockPosition min = new FoundryxDataStorage.BlockPosition(resultSet.getInt("min_x"), resultSet.getInt("min_y"), resultSet.getInt("min_z"));
        FoundryxDataStorage.BlockPosition max = new FoundryxDataStorage.BlockPosition(resultSet.getInt("max_x"), resultSet.getInt("max_y"), resultSet.getInt("max_z"));
        float yaw = resultSet.getFloat("yaw");
        float pitch = resultSet.getFloat("pitch");
        return new FoundryxDataStorage.JailRegion(dimension, min, max, yaw, pitch);
    }

    private static FoundryxDataStorage.LocationSnapshot buildSnapshotFromRow(ResultSet resultSet, String prefix) throws SQLException {
        String dimension = resultSet.getString(prefix + "_dimension");
        if (dimension == null || dimension.isBlank()) {
            return null;
        }
        double x = resultSet.getDouble(prefix + "_x");
        double y = resultSet.getDouble(prefix + "_y");
        double z = resultSet.getDouble(prefix + "_z");
        float yaw = resultSet.getFloat(prefix + "_yaw");
        float pitch = resultSet.getFloat(prefix + "_pitch");
        return new FoundryxDataStorage.LocationSnapshot(dimension, x, y, z, yaw, pitch);
    }

    private static void clearDirectory(Path directory, String glob) throws IOException {
        if (!Files.exists(directory, new LinkOption[0])) {
            return;
        }
        try (DirectoryStream<Path> files = Files.newDirectoryStream(directory, glob);){
            for (Path file : files) {
                Files.deleteIfExists(file);
            }
        }
    }

    private static String filenameWithoutExtension(String name) {
        int dot = name.lastIndexOf(46);
        return dot >= 0 ? name.substring(0, dot) : name;
    }

    private static UUID parseUuid(String raw) {
        if (raw == null || raw.isBlank()) {
            return null;
        }
        try {
            return UUID.fromString(raw.trim());
        }
        catch (IllegalArgumentException exception) {
            return null;
        }
    }

    private static String normalizeKey(String key) {
        if (key == null) {
            return null;
        }
        String trimmed = key.trim();
        return trimmed.isEmpty() ? null : trimmed.toLowerCase(Locale.ROOT);
    }

    private static String normalizeNickname(String nickname) {
        if (nickname == null) {
            return null;
        }
        String trimmed = nickname.trim();
        return trimmed.isEmpty() ? null : trimmed;
    }

    private static String sanitize(String value) {
        if (value == null) {
            return null;
        }
        String trimmed = value.trim();
        return trimmed.isEmpty() ? null : trimmed;
    }

    private static String sanitizeIp(String ip) {
        String sanitized = StorageMigration.sanitize(ip);
        if (sanitized != null && sanitized.isBlank()) {
            return null;
        }
        return sanitized;
    }

    private static Long getNullableLong(ResultSet resultSet, String column) throws SQLException {
        long value = resultSet.getLong(column);
        return resultSet.wasNull() ? null : Long.valueOf(value);
    }

    private static Double getNullableDouble(ResultSet resultSet, String column) throws SQLException {
        double value = resultSet.getDouble(column);
        return resultSet.wasNull() ? null : Double.valueOf(value);
    }

    private static Integer getNullableInteger(ResultSet resultSet, String column) throws SQLException {
        int value = resultSet.getInt(column);
        return resultSet.wasNull() ? null : Integer.valueOf(value);
    }

    private static Float getNullableFloat(ResultSet resultSet, String column) throws SQLException {
        float value = resultSet.getFloat(column);
        return resultSet.wasNull() ? null : Float.valueOf(value);
    }

    private static Boolean getNullableBoolean(ResultSet resultSet, String column) throws SQLException {
        boolean value = resultSet.getBoolean(column);
        return resultSet.wasNull() ? null : Boolean.valueOf(value);
    }

    private static void setNullableString(PreparedStatement statement, int index, String value) throws SQLException {
        if (value == null || value.isBlank()) {
            statement.setNull(index, 12);
        } else {
            statement.setString(index, value);
        }
    }

    private static void setNullableLong(PreparedStatement statement, int index, Long value) throws SQLException {
        if (value == null) {
            statement.setNull(index, -5);
        } else {
            statement.setLong(index, value);
        }
    }

    private static void setNullableInteger(PreparedStatement statement, int index, Integer value) throws SQLException {
        if (value == null) {
            statement.setNull(index, 4);
        } else {
            statement.setInt(index, value);
        }
    }

    private static void setNullableDouble(PreparedStatement statement, int index, Double value) throws SQLException {
        if (value == null) {
            statement.setNull(index, 8);
        } else {
            statement.setDouble(index, value);
        }
    }

    private static void setNullableFloat(PreparedStatement statement, int index, Float value) throws SQLException {
        if (value == null) {
            statement.setNull(index, 7);
        } else {
            statement.setFloat(index, value.floatValue());
        }
    }

    private static void setNullableBoolean(PreparedStatement statement, int index, Boolean value) throws SQLException {
        if (value == null) {
            statement.setNull(index, 16);
        } else {
            statement.setBoolean(index, value);
        }
    }

    private static CompoundTag readCompressedTag(byte[] payload) {
        CompoundTag compoundTag;
        if (payload == null || payload.length == 0) {
            return null;
        }
        ByteArrayInputStream input = new ByteArrayInputStream(payload);
        try {
            compoundTag = NbtIo.readCompressed((InputStream)input, (NbtAccounter)NbtAccounter.unlimitedHeap());
        }
        catch (Throwable throwable) {
            try {
                try {
                    input.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException exception) {
                Constants.LOG.error("Failed to read compressed NBT payload", (Throwable)exception);
                return null;
            }
        }
        input.close();
        return compoundTag;
    }

    private static byte[] writeCompressedTag(CompoundTag tag) {
        byte[] byArray;
        if (tag == null) {
            return new byte[0];
        }
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        try {
            NbtIo.writeCompressed((CompoundTag)tag, (OutputStream)output);
            byArray = output.toByteArray();
        }
        catch (Throwable throwable) {
            try {
                try {
                    output.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException exception) {
                Constants.LOG.error("Failed to write compressed NBT payload", (Throwable)exception);
                return new byte[0];
            }
        }
        output.close();
        return byArray;
    }

    public static final class StorageMigrationException
    extends Exception {
        private final String reason;

        public StorageMigrationException(String reason) {
            super(reason);
            this.reason = reason;
        }

        public StorageMigrationException(String reason, Throwable cause) {
            super(reason, cause);
            this.reason = reason;
        }

        public String reason() {
            return this.reason;
        }
    }

    public record MigrationReport(int users, int homes, int warps, int mail, int backs, int deaths, int kits, int enderChests) {
    }

    private record PersistedUserData(FoundryxDataStorage.JailRecord jail, Long muteExpiry, Map<String, Long> kitUsage, Boolean godMode, Boolean frozen, String nickname, String lastKnownName, Double lastHealth, Integer foodLevel, Float saturation, Boolean operator, Boolean canFly, Integer gameMode, String lastIp, Long lastLogin, Long lastSeen, Long playTime, String muteReason, Double balance) {
    }

    private record SpawnDataContainer(FoundryxDataStorage.LocationSnapshot location) {
    }

    private record PersistedBackData(FoundryxDataStorage.LocationSnapshot location, Long recordedAt) {
    }

    private record PersistedDeathData(FoundryxDataStorage.LocationSnapshot location, Long deathTime, String cause) {
    }

    private record StoredMail(String sender, long timestamp, String message) {
    }

    private record StoredMailData(List<StoredMail> mail) {
    }

    private record EnderChestPayload(String encoded, String lastKnown) {
    }

    private record KitPayload(String encoded, long cooldown) {
    }
}

