/*
 * Decompiled with CFR 0.152.
 */
package com.foundryx.storage.database;

import com.foundryx.Constants;
import com.foundryx.config.ModConfig;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.Map;
import java.util.stream.Collectors;

public final class DatabaseContext {
    public static final String USER_TABLE_NAME = "foundryx_user_data";
    public static final String ENDERCHEST_TABLE_NAME = "foundryx_ender_chests";
    public static final String JAIL_TABLE_NAME = "foundryx_jail_data";
    public static final String HOME_TABLE_NAME = "foundryx_home_data";
    public static final String WARP_TABLE_NAME = "foundryx_warps";
    public static final String SPAWN_TABLE_NAME = "foundryx_spawn_points";
    public static final String KIT_TABLE_NAME = "foundryx_kits";
    public static final String KIT_USAGE_TABLE_NAME = "foundryx_kit_usage";
    public static final String MAIL_TABLE_NAME = "foundryx_mail";
    public static final String DEATH_TABLE_NAME = "foundryx_death_data";
    public static final String BACK_TABLE_NAME = "foundryx_back_data";
    private final DatabaseDialect dialect;
    private final String jdbcUrl;
    private final String username;
    private final String password;
    private volatile boolean available;

    private DatabaseContext(DatabaseDialect dialect, String jdbcUrl, String username, String password) {
        this.dialect = dialect;
        this.jdbcUrl = jdbcUrl;
        this.username = username == null ? "" : username;
        this.password = password == null ? "" : password;
        this.available = this.initialise();
    }

    public static DatabaseContext create(ModConfig.DatabaseSettings settings, Path storageRoot) {
        DatabaseContext context;
        boolean urlExplicitlyConfigured;
        if (settings == null) {
            return null;
        }
        String configuredUrl = settings.url();
        DatabaseDialect dialect = DatabaseDialect.from(settings.type());
        if (dialect == null) {
            dialect = DatabaseDialect.fromJdbcUrl(configuredUrl);
        }
        if (dialect == null) {
            Constants.LOG.error("Unknown database type '{}' and unable to infer from URL '{}' - falling back to file storage", (Object)settings.type(), (Object)configuredUrl);
            return null;
        }
        String url = configuredUrl;
        boolean bl = urlExplicitlyConfigured = url != null && !url.isBlank();
        if (!urlExplicitlyConfigured) {
            url = DatabaseContext.buildJdbcUrl(dialect, settings, storageRoot);
        }
        if (url == null || url.isBlank()) {
            Constants.LOG.error("Failed to resolve JDBC URL for database type '{}' (explicit URL: '{}')", (Object)settings.type(), (Object)configuredUrl);
            return null;
        }
        if (dialect == DatabaseDialect.H2 && urlExplicitlyConfigured) {
            String properties = settings.h2().hasExplicitProperties() ? settings.h2().properties() : null;
            url = DatabaseContext.appendH2Properties(url, properties);
        }
        try {
            Class.forName(dialect.driverClass());
        }
        catch (ClassNotFoundException exception) {
            Constants.LOG.error("Database driver '{}' is not available on the classpath", (Object)dialect.driverClass(), (Object)exception);
            return null;
        }
        String username = settings.username();
        String password = settings.password();
        if (dialect == DatabaseDialect.H2) {
            username = "";
            password = "";
        }
        if (!(context = new DatabaseContext(dialect, url, username, password)).isAvailable()) {
            return null;
        }
        return context;
    }

    public boolean isAvailable() {
        return this.available;
    }

    public Connection getConnection() throws SQLException {
        if (this.username.isBlank()) {
            return DriverManager.getConnection(this.jdbcUrl);
        }
        return DriverManager.getConnection(this.jdbcUrl, this.username, this.password);
    }

    private boolean initialise() {
        boolean bl;
        block8: {
            Connection connection = this.getConnection();
            try {
                connection.setAutoCommit(true);
                this.createTables(connection);
                Constants.LOG.info("Connected to Foundryx database using {}", (Object)this.dialect.readableName());
                bl = true;
                if (connection == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (connection != null) {
                        try {
                            connection.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (SQLException exception) {
                    Constants.LOG.error("Failed to initialise database connection", (Throwable)exception);
                    return false;
                }
            }
            connection.close();
        }
        return bl;
    }

    private void createTables(Connection connection) throws SQLException {
        try (Statement statement = connection.createStatement();){
            statement.executeUpdate("CREATE TABLE IF NOT EXISTS foundryx_user_data (player_uuid VARCHAR(40) PRIMARY KEY,nickname VARCHAR(191),last_known_name VARCHAR(191),last_login BIGINT,last_seen BIGINT,play_time BIGINT,god_mode BOOLEAN,frozen BOOLEAN,mute_expiry BIGINT,mute_reason TEXT,last_ip VARCHAR(191),balance DOUBLE PRECISION)");
        }
        this.ensureUserDataColumns(connection);
        statement = connection.createStatement();
        try {
            statement.executeUpdate("CREATE TABLE IF NOT EXISTS foundryx_ender_chests (player_uuid VARCHAR(40) PRIMARY KEY,last_known_name VARCHAR(191),nbt_payload TEXT)");
        }
        finally {
            if (statement != null) {
                statement.close();
            }
        }
        statement = connection.createStatement();
        try {
            statement.executeUpdate("CREATE TABLE IF NOT EXISTS foundryx_jail_data (id INTEGER PRIMARY KEY,dimension VARCHAR(191),min_x INT,min_y INT,min_z INT,max_x INT,max_y INT,max_z INT,yaw REAL,pitch REAL)");
        }
        finally {
            if (statement != null) {
                statement.close();
            }
        }
        statement = connection.createStatement();
        try {
            statement.executeUpdate("CREATE TABLE IF NOT EXISTS foundryx_home_data (player_uuid VARCHAR(40) NOT NULL,home_name VARCHAR(191) NOT NULL,dimension VARCHAR(191),x DOUBLE PRECISION,y DOUBLE PRECISION,z DOUBLE PRECISION,yaw REAL,pitch REAL,PRIMARY KEY (player_uuid, home_name))");
        }
        finally {
            if (statement != null) {
                statement.close();
            }
        }
        statement = connection.createStatement();
        try {
            statement.executeUpdate("CREATE TABLE IF NOT EXISTS foundryx_warps (warp_name VARCHAR(191) PRIMARY KEY,owner_uuid VARCHAR(40),dimension VARCHAR(191),x DOUBLE PRECISION,y DOUBLE PRECISION,z DOUBLE PRECISION,yaw REAL,pitch REAL)");
        }
        finally {
            if (statement != null) {
                statement.close();
            }
        }
        statement = connection.createStatement();
        try {
            statement.executeUpdate("CREATE TABLE IF NOT EXISTS foundryx_spawn_points (id INTEGER PRIMARY KEY,dimension VARCHAR(191),x DOUBLE PRECISION,y DOUBLE PRECISION,z DOUBLE PRECISION,yaw REAL,pitch REAL)");
        }
        finally {
            if (statement != null) {
                statement.close();
            }
        }
        statement = connection.createStatement();
        try {
            statement.executeUpdate("CREATE TABLE IF NOT EXISTS foundryx_kits (kit_name VARCHAR(191) PRIMARY KEY,cooldown_seconds BIGINT,nbt_payload TEXT)");
        }
        finally {
            if (statement != null) {
                statement.close();
            }
        }
        statement = connection.createStatement();
        try {
            statement.executeUpdate("CREATE TABLE IF NOT EXISTS foundryx_kit_usage (player_uuid VARCHAR(40) NOT NULL,kit_name VARCHAR(191) NOT NULL,last_used_at BIGINT,PRIMARY KEY (player_uuid, kit_name))");
        }
        finally {
            if (statement != null) {
                statement.close();
            }
        }
        statement = connection.createStatement();
        try {
            statement.executeUpdate("CREATE TABLE IF NOT EXISTS foundryx_mail (mail_id VARCHAR(40) PRIMARY KEY,player_uuid VARCHAR(40) NOT NULL,sender_uuid VARCHAR(40),sent_at BIGINT NOT NULL,message TEXT NOT NULL)");
        }
        finally {
            if (statement != null) {
                statement.close();
            }
        }
        statement = connection.createStatement();
        try {
            statement.executeUpdate("CREATE TABLE IF NOT EXISTS foundryx_death_data (player_uuid VARCHAR(40) PRIMARY KEY,dimension VARCHAR(191),x DOUBLE PRECISION,y DOUBLE PRECISION,z DOUBLE PRECISION,yaw REAL,pitch REAL,cause TEXT,death_time BIGINT)");
        }
        finally {
            if (statement != null) {
                statement.close();
            }
        }
        statement = connection.createStatement();
        try {
            statement.executeUpdate("CREATE TABLE IF NOT EXISTS foundryx_back_data (player_uuid VARCHAR(40) PRIMARY KEY,dimension VARCHAR(191),x DOUBLE PRECISION,y DOUBLE PRECISION,z DOUBLE PRECISION,yaw REAL,pitch REAL,recorded_at BIGINT)");
        }
        finally {
            if (statement != null) {
                statement.close();
            }
        }
    }

    private void ensureUserDataColumns(Connection connection) throws SQLException {
        this.addColumnIfMissing(connection, USER_TABLE_NAME, "last_health DOUBLE PRECISION");
        this.addColumnIfMissing(connection, USER_TABLE_NAME, "food_level INT");
        this.addColumnIfMissing(connection, USER_TABLE_NAME, "saturation REAL");
        this.addColumnIfMissing(connection, USER_TABLE_NAME, "operator BOOLEAN");
        this.addColumnIfMissing(connection, USER_TABLE_NAME, "can_fly BOOLEAN");
        this.addColumnIfMissing(connection, USER_TABLE_NAME, "game_mode INT");
        this.addColumnIfMissing(connection, USER_TABLE_NAME, "jail_origin_dimension VARCHAR(191)");
        this.addColumnIfMissing(connection, USER_TABLE_NAME, "jail_origin_x DOUBLE PRECISION");
        this.addColumnIfMissing(connection, USER_TABLE_NAME, "jail_origin_y DOUBLE PRECISION");
        this.addColumnIfMissing(connection, USER_TABLE_NAME, "jail_origin_z DOUBLE PRECISION");
        this.addColumnIfMissing(connection, USER_TABLE_NAME, "jail_origin_yaw REAL");
        this.addColumnIfMissing(connection, USER_TABLE_NAME, "jail_origin_pitch REAL");
        this.addColumnIfMissing(connection, USER_TABLE_NAME, "jail_location_dimension VARCHAR(191)");
        this.addColumnIfMissing(connection, USER_TABLE_NAME, "jail_location_x DOUBLE PRECISION");
        this.addColumnIfMissing(connection, USER_TABLE_NAME, "jail_location_y DOUBLE PRECISION");
        this.addColumnIfMissing(connection, USER_TABLE_NAME, "jail_location_z DOUBLE PRECISION");
        this.addColumnIfMissing(connection, USER_TABLE_NAME, "jail_location_yaw REAL");
        this.addColumnIfMissing(connection, USER_TABLE_NAME, "jail_location_pitch REAL");
        this.addColumnIfMissing(connection, USER_TABLE_NAME, "jail_release BIGINT");
    }

    private void addColumnIfMissing(Connection connection, String table, String definition) throws SQLException {
        String columnName = definition.split("\\s+")[0];
        if (this.columnExists(connection, table, columnName)) {
            return;
        }
        try (Statement statement = connection.createStatement();){
            statement.executeUpdate("ALTER TABLE " + table + " ADD COLUMN " + definition);
        }
    }

    private boolean columnExists(Connection connection, String table, String column) throws SQLException {
        DatabaseMetaData metaData = connection.getMetaData();
        String tableName = table;
        String columnName = column;
        try (ResultSet result = metaData.getColumns(connection.getCatalog(), null, tableName, columnName);){
            if (result.next()) {
                boolean bl = true;
                return bl;
            }
        }
        result = metaData.getColumns(connection.getCatalog(), null, tableName.toUpperCase(Locale.ROOT), columnName.toUpperCase(Locale.ROOT));
        try {
            if (result.next()) {
                boolean bl = true;
                return bl;
            }
        }
        finally {
            if (result != null) {
                result.close();
            }
        }
        result = metaData.getColumns(connection.getCatalog(), null, tableName.toLowerCase(Locale.ROOT), columnName.toLowerCase(Locale.ROOT));
        try {
            if (result.next()) {
                boolean bl = true;
                return bl;
            }
        }
        finally {
            if (result != null) {
                result.close();
            }
        }
        return false;
    }

    private static String buildJdbcUrl(DatabaseDialect dialect, ModConfig.DatabaseSettings settings, Path storageRoot) {
        String properties;
        ModConfig.DatabaseSettings.SqlSettings sqlSettings = settings.sql();
        String host = sqlSettings.host();
        int port = sqlSettings.port() > 0 ? sqlSettings.port() : dialect.defaultPort();
        String database = sqlSettings.database();
        String baseUrl = switch (dialect.ordinal()) {
            case 0 -> String.format(Locale.ROOT, "jdbc:postgresql://%s:%d/%s", host, port, database);
            case 1 -> String.format(Locale.ROOT, "jdbc:mysql://%s:%d/%s", host, port, database);
            case 2 -> String.format(Locale.ROOT, "jdbc:mariadb://%s:%d/%s", host, port, database);
            case 3 -> DatabaseContext.buildH2Url(settings, storageRoot);
            default -> null;
        };
        if (baseUrl == null) {
            return null;
        }
        String string = properties = dialect == DatabaseDialect.H2 ? settings.h2().properties() : sqlSettings.properties();
        if (dialect == DatabaseDialect.H2) {
            return DatabaseContext.appendH2Properties(baseUrl, properties);
        }
        if (properties == null || properties.isBlank()) {
            return baseUrl;
        }
        return baseUrl + (String)(properties.startsWith("?") ? properties : "?" + properties);
    }

    private static String buildH2Url(ModConfig.DatabaseSettings settings, Path storageRoot) {
        String trimmed;
        ModConfig.DatabaseSettings.H2Settings h2 = settings.h2();
        String configured = h2.file();
        String string = trimmed = configured == null ? "" : configured.trim();
        if (trimmed.isEmpty()) {
            trimmed = "database/foundryx";
        }
        if (trimmed.startsWith("jdbc:h2:")) {
            return trimmed;
        }
        if (trimmed.startsWith("mem:") || trimmed.startsWith("file:") || trimmed.startsWith("zip:")) {
            return "jdbc:h2:" + trimmed;
        }
        Path target = h2.resolveFilePath(storageRoot);
        Path parent = target.getParent();
        if (parent != null) {
            try {
                Files.createDirectories(parent, new FileAttribute[0]);
            }
            catch (IOException exception) {
                Constants.LOG.warn("Failed to create directories for H2 database path {}", (Object)parent, (Object)exception);
            }
        }
        String path = target.toAbsolutePath().toString().replace("\\", "/");
        return "jdbc:h2:file:" + path;
    }

    private static String appendH2Properties(String baseUrl, String configuredProperties) {
        if (baseUrl == null || baseUrl.isBlank()) {
            return baseUrl;
        }
        int separatorIndex = baseUrl.indexOf(59);
        String base = separatorIndex >= 0 ? baseUrl.substring(0, separatorIndex) : baseUrl;
        String existing = separatorIndex >= 0 ? baseUrl.substring(separatorIndex + 1) : "";
        LinkedHashMap<String, H2Property> properties = new LinkedHashMap<String, H2Property>();
        properties.putAll(DatabaseContext.parseH2Properties(existing));
        boolean fileDatabase = base.startsWith("jdbc:h2:file:");
        if (fileDatabase) {
            DatabaseContext.ensureDefaultH2Property(properties, "AUTO_SERVER", "TRUE");
            DatabaseContext.ensureDefaultH2Property(properties, "LOCK_TIMEOUT", "60000");
            DatabaseContext.ensureDefaultH2Property(properties, "DB_CLOSE_DELAY", "-1");
        }
        if (configuredProperties != null && !configuredProperties.isBlank()) {
            String processed = configuredProperties.startsWith(";") ? configuredProperties.substring(1) : configuredProperties;
            properties.putAll(DatabaseContext.parseH2Properties(processed));
        }
        if (properties.isEmpty()) {
            return base;
        }
        String joined = properties.values().stream().map(H2Property::toSegment).collect(Collectors.joining(";"));
        return base + ";" + joined;
    }

    private static void ensureDefaultH2Property(Map<String, H2Property> properties, String key, String value) {
        String normalised = key.toUpperCase(Locale.ROOT);
        properties.putIfAbsent(normalised, new H2Property(key, value));
    }

    private static Map<String, H2Property> parseH2Properties(String propertyString) {
        String[] segments;
        LinkedHashMap<String, H2Property> result = new LinkedHashMap<String, H2Property>();
        if (propertyString == null || propertyString.isBlank()) {
            return result;
        }
        for (String rawSegment : segments = propertyString.split(";")) {
            String value;
            String key;
            String segment = rawSegment.trim();
            if (segment.isEmpty()) continue;
            int equalsIndex = segment.indexOf(61);
            if (equalsIndex >= 0) {
                key = segment.substring(0, equalsIndex);
                value = segment.substring(equalsIndex + 1);
            } else {
                key = segment;
                value = "";
            }
            String normalised = key.toUpperCase(Locale.ROOT);
            result.put(normalised, new H2Property(key, value));
        }
        return result;
    }

    private static enum DatabaseDialect {
        POSTGRESQL("postgresql", "org.postgresql.Driver", 5432),
        MYSQL("mysql", "com.mysql.cj.jdbc.Driver", 3306),
        MARIADB("mariadb", "org.mariadb.jdbc.Driver", 3306),
        H2("h2", "org.h2.Driver", -1);

        private final String key;
        private final String driverClass;
        private final int defaultPort;

        private DatabaseDialect(String key, String driverClass, int defaultPort) {
            this.key = key;
            this.driverClass = driverClass;
            this.defaultPort = defaultPort;
        }

        public String driverClass() {
            return this.driverClass;
        }

        public int defaultPort() {
            return this.defaultPort;
        }

        public String readableName() {
            return this.key.toUpperCase(Locale.ROOT);
        }

        public static DatabaseDialect from(String type) {
            if (type == null) {
                return null;
            }
            String normalised = type.toLowerCase(Locale.ROOT);
            for (DatabaseDialect dialect : DatabaseDialect.values()) {
                if (!dialect.key.equals(normalised)) continue;
                return dialect;
            }
            return null;
        }

        public static DatabaseDialect fromJdbcUrl(String url) {
            if (url == null || url.isBlank()) {
                return null;
            }
            String lower = url.toLowerCase(Locale.ROOT);
            if (lower.startsWith("jdbc:")) {
                lower = lower.substring("jdbc:".length());
            }
            for (DatabaseDialect dialect : DatabaseDialect.values()) {
                if (!lower.startsWith(dialect.key + ":") && !lower.startsWith(dialect.key + "://")) continue;
                return dialect;
            }
            return null;
        }
    }

    private static final class H2Property {
        private final String key;
        private final String value;

        private H2Property(String key, String value) {
            this.key = key == null ? "" : key;
            this.value = value == null ? "" : value;
        }

        private String toSegment() {
            if (this.value.isEmpty()) {
                return this.key;
            }
            return this.key + "=" + this.value;
        }
    }
}

