const fs = require('fs');
const path = require('path');

class MinecraftRecipeConverter {
    constructor() {
        this.currentDir = process.cwd();
        this.foundFolder = null;
        this.folderType = null;
    }

    // Поиск папки recipes или recipe
    findRecipeFolder() {
        const recipesPath = path.join(this.currentDir, 'recipes');
        const recipePath = path.join(this.currentDir, 'recipe');

        if (fs.existsSync(recipesPath) && fs.statSync(recipesPath).isDirectory()) {
            this.foundFolder = recipesPath;
            this.folderType = 'recipes'; // Формат <1.21
            return true;
        }

        if (fs.existsSync(recipePath) && fs.statSync(recipePath).isDirectory()) {
            this.foundFolder = recipePath;
            this.folderType = 'recipe'; // Формат 1.21>
            return true;
        }

        return false;
    }

    // Рекурсивный поиск всех .json файлов в папке
    findJsonFiles(dir) {
        let jsonFiles = [];
        
        try {
            const files = fs.readdirSync(dir);
            
            for (const file of files) {
                const fullPath = path.join(dir, file);
                const stat = fs.statSync(fullPath);
                
                if (stat.isDirectory()) {
                    // Рекурсивно ищем в подпапках
                    jsonFiles = jsonFiles.concat(this.findJsonFiles(fullPath));
                } else if (path.extname(file).toLowerCase() === '.json') {
                    jsonFiles.push(fullPath);
                }
            }
        } catch (error) {
            console.error(`Ошибка при чтении директории ${dir}:`, error.message);
        }
        
        return jsonFiles;
    }

    // Конвертация содержимого JSON файла
    convertJsonContent(content, fromFormat) {
        try {
            const data = JSON.parse(content);
            
            // Проверяем, есть ли структура result
            if (data.result && typeof data.result === 'object') {
                if (fromFormat === 'old' && data.result.item !== undefined) {
                    // Конвертируем из старого формата (<1.21) в новый (1.21>)
                    // item -> id
                    data.result.id = data.result.item;
                    delete data.result.item;
                    return { converted: true, data: JSON.stringify(data, null, 2) };
                } else if (fromFormat === 'new' && data.result.id !== undefined) {
                    // Конвертируем из нового формата (1.21>) в старый (<1.21)
                    // id -> item
                    data.result.item = data.result.id;
                    delete data.result.id;
                    return { converted: true, data: JSON.stringify(data, null, 2) };
                }
            }
            
            return { converted: false, data: content };
        } catch (error) {
            console.error('Ошибка при парсинге JSON:', error.message);
            return { converted: false, data: content };
        }
    }

    // Обработка одного JSON файла
    processJsonFile(filePath) {
        try {
            const content = fs.readFileSync(filePath, 'utf8');
            
            // Определяем направление конвертации на основе типа найденной папки
            const fromFormat = this.folderType === 'recipes' ? 'old' : 'new';
            
            const result = this.convertJsonContent(content, fromFormat);
            
            if (result.converted) {
                // Создаем бэкап оригинального файла
                const backupPath = filePath + '.backup';
                if (!fs.existsSync(backupPath)) {
                    fs.writeFileSync(backupPath, content, 'utf8');
                }
                
                // Записываем конвертированное содержимое
                fs.writeFileSync(filePath, result.data, 'utf8');
                
                const direction = fromFormat === 'old' ? 'item -> id' : 'id -> item';
                console.log(`✅ Конвертирован: ${filePath} (${direction})`);
                return true;
            } else {
                console.log(`⏭️  Пропущен: ${filePath} (не требует конвертации)`);
                return false;
            }
        } catch (error) {
            console.error(`❌ Ошибка при обработке ${filePath}:`, error.message);
            return false;
        }
    }

    // Основная функция запуска
    run() {
        console.log('🔍 Поиск папки с рецептами...');
        
        if (!this.findRecipeFolder()) {
            console.log('❌ Папка "recipes" или "recipe" не найдена в текущей директории');
            return;
        }

        console.log(`✅ Найдена папка: ${this.foundFolder}`);
        
        const targetFormat = this.folderType === 'recipes' ? 'новый (1.21+)' : 'старый (<1.21)';
        console.log(`📋 Конвертация в формат: ${targetFormat}`);
        
        console.log('🔍 Поиск JSON файлов...');
        const jsonFiles = this.findJsonFiles(this.foundFolder);
        
        if (jsonFiles.length === 0) {
            console.log('❌ JSON файлы не найдены');
            return;
        }

        console.log(`📁 Найдено ${jsonFiles.length} JSON файл(ов)`);
        console.log('🔄 Начинаю конвертацию...\n');

        let convertedCount = 0;
        let totalCount = jsonFiles.length;

        for (const filePath of jsonFiles) {
            if (this.processJsonFile(filePath)) {
                convertedCount++;
            }
        }

        console.log(`\n🎉 Готово! Конвертировано ${convertedCount} из ${totalCount} файлов`);
        console.log('💾 Оригинальные файлы сохранены как .backup');
    }
}

// Запуск конвертера
const converter = new MinecraftRecipeConverter();
converter.run();