/*
 * Copyright (c) 2025. gitoido-mc
 * This Source Code Form is subject to the terms of the GNU General Public License v3.0.
 * If a copy of the GNU General Public License v3.0 was not distributed with this file,
 * you can obtain one at https://github.com/gitoido-mc/rad-gyms/blob/main/LICENSE.
 */

package lol.gito.radgyms.common.event

import com.cobblemon.mod.common.api.Priority
import com.cobblemon.mod.common.api.battles.model.actor.AIBattleActor
import com.cobblemon.mod.common.api.battles.model.actor.ActorType
import com.cobblemon.mod.common.api.events.CobblemonEvents
import com.cobblemon.mod.common.api.events.battles.BattleFaintedEvent
import com.cobblemon.mod.common.api.events.battles.BattleFledEvent
import com.cobblemon.mod.common.api.events.battles.BattleStartedEvent
import com.cobblemon.mod.common.api.events.battles.BattleVictoryEvent
import com.cobblemon.mod.common.api.pokemon.PokemonSpecies
import com.cobblemon.mod.common.api.types.ElementalTypes
import com.cobblemon.mod.common.platform.events.PlatformEvents
import com.cobblemon.mod.common.platform.events.ServerEvent
import com.cobblemon.mod.common.platform.events.ServerPlayerEvent
import lol.gito.radgyms.common.RadGyms.CONFIG
import lol.gito.radgyms.common.RadGyms.RCT
import lol.gito.radgyms.common.RadGyms.debug
import lol.gito.radgyms.common.api.enumeration.GymBattleEndReason
import lol.gito.radgyms.common.api.event.GymEvents
import lol.gito.radgyms.common.api.event.GymEvents.CACHE_ROLL_POKE
import lol.gito.radgyms.common.api.event.GymEvents.GENERATE_REWARD
import lol.gito.radgyms.common.api.event.GymEvents.GYM_ENTER
import lol.gito.radgyms.common.api.event.GymEvents.GYM_LEAVE
import lol.gito.radgyms.common.api.event.GymEvents.TRAINER_BATTLE_END
import lol.gito.radgyms.common.api.event.GymEvents.TRAINER_BATTLE_START
import lol.gito.radgyms.common.api.event.GymEvents.TRAINER_INTERACT
import lol.gito.radgyms.common.entity.Trainer
import lol.gito.radgyms.common.event.cache.CacheRollPokeHandler
import lol.gito.radgyms.common.event.cache.ShinyCharmCheckHandler
import lol.gito.radgyms.common.event.gyms.*
import lol.gito.radgyms.common.gym.GymTeardownService
import lol.gito.radgyms.common.gym.SpeciesManager.SPECIES_BY_TYPE
import lol.gito.radgyms.common.gym.SpeciesManager.SPECIES_TIMESTAMP
import lol.gito.radgyms.common.gym.SpeciesManager.speciesOfType
import lol.gito.radgyms.common.net.server.payload.ServerSettingsS2C
import lol.gito.radgyms.common.registry.RadGymsBlocks
import lol.gito.radgyms.common.registry.RadGymsDimensions
import lol.gito.radgyms.common.state.RadGymsState
import lol.gito.radgyms.common.util.hasRadGymsTrainers
import kotlin.time.TimeSource.Monotonic.markNow

object EventManager {
    fun register() {
        debug("Registering event handlers")
        // Minecraft events
        PlatformEvents.SERVER_STARTING.subscribe(Priority.NORMAL, ::onServerStarting)
        PlatformEvents.SERVER_STOPPING.subscribe(Priority.HIGHEST, ::onServerStopping)
        PlatformEvents.SERVER_PLAYER_LOGIN.subscribe(Priority.NORMAL, ::onPlayerJoin)
        PlatformEvents.SERVER_PLAYER_LOGOUT.subscribe(Priority.HIGHEST, ::onPlayerDisconnect)
        PlatformEvents.RIGHT_CLICK_BLOCK.subscribe(Priority.NORMAL, ::onBlockInteract)

        // Cobblemon events
        PokemonSpecies.observable.subscribe(Priority.LOWEST) { _ ->
            debug("Cobblemon species observable triggered, updating elemental gyms species map")
            onSpeciesUpdate()
        }

        // Cobblemon battle events
        CobblemonEvents.BATTLE_STARTED_PRE.subscribe(Priority.NORMAL, ::onBattleStart)
        CobblemonEvents.BATTLE_VICTORY.subscribe(Priority.NORMAL, ::onBattleWon)
        CobblemonEvents.BATTLE_FLED.subscribe(Priority.NORMAL, ::onBattleFled)
        CobblemonEvents.BATTLE_FAINTED.subscribe(Priority.NORMAL, ::onBattleFainted)

        // Mod events
        GYM_ENTER.subscribe(Priority.LOWEST, ::GymEnterHandler)
        GYM_LEAVE.subscribe(Priority.LOWEST, ::GymLeaveHandler)

        TRAINER_INTERACT.subscribe(Priority.LOWEST, ::TrainerInteractHandler)
        TRAINER_BATTLE_END.subscribe(Priority.LOWEST, ::TrainerBattleEndHandler)

        GENERATE_REWARD.subscribe(Priority.LOWEST, ::GenerateRewardHandler)

        CACHE_ROLL_POKE.subscribe(Priority.LOWEST, ::ShinyCharmCheckHandler)
        CACHE_ROLL_POKE.subscribe(Priority.LOWEST, ::CacheRollPokeHandler)
    }


    @Suppress("UNUSED_PARAMETER")
    private fun onBlockInteract(event: ServerPlayerEvent.RightClickBlock) {
        if (event.player.level().dimension() == RadGymsDimensions.RADGYMS_LEVEL_KEY) {
            if (CONFIG.debug == true) return
            if (event.player.level().getBlockState(event.pos).block == RadGymsBlocks.GYM_EXIT) return
            event.cancel()
        }
    }

    private fun onServerStarting(event: ServerEvent.Starting) {
        val trainerRegistry = RCT.trainerRegistry
        debug("initializing RCT trainer mod registry")
        trainerRegistry.init(event.server)
    }

    private fun onServerStopping(event: ServerEvent.Stopping) {
        debug("cleaning up all gyms")
        RadGymsState.getServerState(event.server).gymInstanceMap.let {
            it.forEach { (playerUuid, gym) ->
                GymTeardownService.spawnExitBlock(event.server, gym)
                GymTeardownService.destructOfflineGym(event.server, playerUuid, gym)
            }
            it.clear()
        }
    }

    private fun onPlayerJoin(event: ServerPlayerEvent) {
        debug("Sending server settings to player ${event.player.name}")
        ServerSettingsS2C(
            CONFIG.maxEntranceUses!!,
            CONFIG.shardRewards!!,
            CONFIG.lapisBoostAmount!!,
            CONFIG.ignoredSpecies!!,
            CONFIG.ignoredForms!!,
            CONFIG.minLevel!!,
            CONFIG.maxLevel!!
        ).sendToPlayer(event.player)

        try {
            debug("Adding player ${event.player.name} to gyms trainer registry")
            RCT.trainerRegistry.registerPlayer(event.player.uuid.toString(), event.player)
            val playerData = RadGymsState.getPlayerState(event.player)
            debug("player gym visits: ${playerData.visits}, has return coords? ${playerData.returnCoords != null}")
        } catch (_: IllegalArgumentException) {
            debug("Player ${event.player.name} is already present in gyms trainer registry, skipping")
        }
    }

    private fun onPlayerDisconnect(event: ServerPlayerEvent) {
        debug("Removing player ${event.player.name} from RCT trainer mod registry")
        RCT.trainerRegistry.unregisterById(event.player.uuid.toString())

        if (event.player.level().dimension() == RadGymsDimensions.RADGYMS_LEVEL_KEY) {
            GymTeardownService.spawnExitBlock(event.player.server, RadGymsState.getGymForPlayer(event.player)!!)
            GymTeardownService.destructGym(event.player, removeCoords = false)
        }
    }

    private fun onSpeciesUpdate() {
        val now = markNow()
        if (SPECIES_TIMESTAMP > now) return
        SPECIES_BY_TYPE.clear()

        ElementalTypes.all().forEach {
            SPECIES_BY_TYPE[it.showdownId] = speciesOfType(it)
            debug("Added ${SPECIES_BY_TYPE[it.showdownId]?.size} ${it.showdownId} entries to species map")
        }
        SPECIES_TIMESTAMP = markNow()
    }

    private fun onBattleStart(event: BattleStartedEvent.Pre) {
        // Early bail if not gym related
        if (!hasRadGymsTrainers(event)) return

        val players = event.battle.players
        val trainers = event.battle.actors
            .filter { it.type == ActorType.NPC && it is AIBattleActor }
            .map { it as AIBattleActor }
            .mapNotNull { RCT.trainerRegistry.getById(it.uuid.toString())?.entity }
            .filter { it is Trainer }

        TRAINER_BATTLE_START.postThen(
            GymEvents.TrainerBattleStartEvent(players, trainers.map { it as Trainer }, event.battle),
            { subEvent -> if (subEvent.isCanceled) event.cancel() },
            { _ -> debug("Gym trainer battle started for players: ${players.joinToString(" ") { it.name.string }}") },
        )
    }

    private fun onBattleWon(event: BattleVictoryEvent) {
        // Early bail if it was wild poke battle
        if (event.wasWildCapture) return
        // Early bail if not gym related
        if (!hasRadGymsTrainers(event)) return
        if (event.losers.none { it.type == ActorType.NPC }) return
        if (event.winners.none { it.type == ActorType.PLAYER }) return

        TRAINER_BATTLE_END.emit(
            GymEvents.TrainerBattleEndEvent(
                GymBattleEndReason.BATTLE_WON,
                event.winners,
                event.losers,
                event.battle
            )
        )
    }

    private fun onBattleFled(event: BattleFledEvent) {
        // Early bail if not gym related
        if (!hasRadGymsTrainers(event)) return

        TRAINER_BATTLE_END.emit(
            GymEvents.TrainerBattleEndEvent(
                GymBattleEndReason.BATTLE_FLED,
                event.battle.winners,
                event.battle.losers,
                event.battle
            )
        )
    }

    private fun onBattleFainted(event: BattleFaintedEvent) {
        // Early bail if not gym related
        if (!hasRadGymsTrainers(event)) return

        // Kudos Tim for whiteout mod
        val killed = event.killed
        val entity = killed.entity ?: return
        val owner = entity.owner ?: return
        if (killed.actor.type != ActorType.PLAYER) return
        if (!killed.actor.pokemonList.all { it.health == 0 }) return
        if (owner.isDeadOrDying) return

        TRAINER_BATTLE_END.emit(
            GymEvents.TrainerBattleEndEvent(
                GymBattleEndReason.BATTLE_LOST,
                event.battle.winners,
                event.battle.losers,
                event.battle
            )
        )
    }
}
