/*
 * Copyright 2023 The Quilt Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

@file:Suppress("MemberVisibilityCanBePrivate")

package settingdust.kinecraft.serialization.codec

import com.mojang.serialization.Codec
import kotlinx.serialization.KSerializer
import kotlinx.serialization.modules.SerializersModule
import kotlinx.serialization.serializer
import settingdust.kinecraft.serialization.codec.internal.SerializerCodec
import settingdust.kinecraft.serialization.codec.options.CodecOptions
import settingdust.kinecraft.serialization.codec.options.CodecOptionsBuilder
import kotlin.reflect.KType
import kotlin.reflect.typeOf

/**
 * The entry point to QKL serialization, CodecFactory allows [creating][create] [codecs][Codec] from
 * [serializers][KSerializer] generated by the kotlinx.serialization plugin (or written manually).
 *
 * @param options serialization options to be forwarded to created codecs.
 */
public sealed class CodecFactory(public val options: CodecOptions) {
    private val serializersModule: SerializersModule = SerializersModule {
        include(options.serializersModule)

        options.codecs.forEach { it.apply(this) }
    }

    /** Creates a new [Codec] from the provided [serializer]. */
    public fun <T> create(serializer: KSerializer<T>): Codec<T> {
        return SerializerCodec(serializer, options, serializersModule)
    }

    /** Creates a new [Codec] using the serializer for the given [type]. */
    public fun <T> create(type: KType): Codec<T> {
        @Suppress("UNCHECKED_CAST")
        return create(serializersModule.serializer(type) as KSerializer<T>)
    }

    /**
     * Creates a new [Codec] using the serializer for the given type [T].
     *
     * @see serializer
     */
    public inline fun <reified T> create(): Codec<T> {
        return create(typeOf<T>())
    }

    /** An instance of [CodecFactory] that uses [default options][CodecOptions.DEFAULT]. */
    public companion object Default : CodecFactory(CodecOptions.DEFAULT)

    // sealed is always abstract, so here's a concrete impl
    internal class Impl(options: CodecOptions) : CodecFactory(options)
}

/**
 * Creates a new [CodecFactory] instance, using [from]'s [CodecOptions] as a parent (or
 * [CodecOptions.DEFAULT] if not given), and applying [config] to further specify the options.
 */
public fun CodecFactory(
    from: CodecFactory = CodecFactory.Default,
    config: CodecOptionsBuilder.() -> Unit
): CodecFactory {
    return CodecFactory.Impl(CodecOptionsBuilder(from.options).apply(config).build())
}
